package com.biz.crm.util;

import com.biz.crm.common.param.RedisParam;
import com.biz.crm.mdm.feign.MdmDictFeign;
import com.biz.crm.nebular.mdm.dict.resp.*;
import com.biz.crm.nebular.mdm.dict.resp.DictGroupRedisVo;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据字典工具类
 **/
@Slf4j
public class DictUtil {

    private static RedisService redisService;

    private static MdmDictFeign mdmDictFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmDictFeign(MdmDictFeign mdmDictFeign) {
        DictUtil.mdmDictFeign = mdmDictFeign;
    }

    /**
     * 根据字典类型编码集合查询对应的value的map集合
     *
     * @param dictTypeCodes 分类编码集合
     * @return map
     */
    public static Map<String, Map<String, String>> getDictValueMapsByCodes(List<String> dictTypeCodes) {
        Map<String, Map<String, String>> maps = new HashMap<>(16);
        try {
            if (dictTypeCodes != null && dictTypeCodes.size() > 0) {
                Map<String, DictGroupRedisVo> groupMaps = getDictGroupMap(dictTypeCodes);
                for (String dictTypeCode :
                        dictTypeCodes) {
                    if (groupMaps.containsKey(dictTypeCode)) {
                        Map<String, String> map = new LinkedHashMap<>(16);
                        DictGroupRedisVo dictGroup = groupMaps.get(dictTypeCode);
                        if (dictGroup != null) {
                            for (DictDataRedisVo item :
                                    dictGroup.getDataList()) {
                                if (StringUtils.isEmpty(item.getParentDictCode())) {
                                    map.put(item.getDictCode(), item.getDictValue());
                                }
                            }
                        }
                        maps.put(dictTypeCode, map);
                    }
                }
            }
        } catch (Exception e) {
            log.error("获取字典数据异常!!!!!!!!");
            e.printStackTrace();
        }
        return maps;
    }

    /**
     * 根据字典类型编码查询对应的值的集合
     *
     * @param dictTypeCode 分类编码
     * @return map
     */
    public static Map<String, String> getDictValueMapsByCodes(String dictTypeCode) {
        return dictMap(dictTypeCode);
    }

    /**
     * 根据字典类型编码查询对应的值的集合(只返回第一层)
     *
     * @param dictTypeCode 字典类型编码
     * @return map
     */
    public static Map<String, String> dictMap(String dictTypeCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            DictGroupRedisVo dict = getDictGroup(dictTypeCode);
            if (dict != null) {
                for (DictDataRedisVo item :
                        dict.getDataList()) {
                    if (StringUtils.isEmpty(item.getParentDictCode())) {
                        map.put(item.getDictCode(), item.getDictValue());
                    }
                }
            }
        }
        return map;
    }

    /**
     * 根据字典类型编码和字典编码获取字典值(只从第一层里找)
     *
     * @param dictTypeCode 字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static String dictValue(String dictTypeCode, String dictCode) {
        if (StringUtils.isNotEmpty(dictTypeCode) && StringUtils.isNotEmpty(dictCode)) {
            Map<String, String> map = dictMap(dictTypeCode);
            if (map.containsKey(dictCode)) {
                return map.get(dictCode);
            }
        }
        return null;
    }

    /**
     * 根据编码获取数据字典值树形结构
     *
     * @param dictTypeCode
     * @return
     */
    public static List<DictDataVo> tree(String dictTypeCode) {
        List<DictDataVo> tree = new ArrayList<>();
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            DictGroupRedisVo dictGroup = getDictGroup(dictTypeCode);
            if (dictGroup != null) {
                List<DictDataRedisVo> dataList = dictGroup.getDataList();
                if (dataList != null) {
                    List<DictDataVo> list = new ArrayList<>();
                    dataList.forEach(item -> {
                        DictDataVo data = CrmBeanUtil.copy(item, DictDataVo.class);
                        if (CollectionUtil.listNotEmptyNotSizeZero(dictGroup.getConfList())) {
                            Map<String, String> extendMap = new HashMap<>(16);
                            for (DictAttrConfRedisVo conf :
                                    dictGroup.getConfList()) {
                                Object obj = ReflectUtil.getFiledValueByName(item, conf.getExtField());
                                extendMap.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
                            }
                            data.setExtendMap(extendMap);
                        }
                        list.add(data);
                    });
                    return generateTree(list);
                }
            }
        }
        return tree;
    }

    /**
     * 获取全部值的list
     *
     * @param dictTypeCode
     * @return
     */
    public static List<DictDataVo> list(String dictTypeCode) {
        List<DictDataVo> list = new ArrayList<>();
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            DictGroupRedisVo dictGroup = getDictGroup(dictTypeCode);
            if (dictGroup != null) {
                List<DictDataRedisVo> dataList = dictGroup.getDataList();
                dataList.forEach(item -> {
                    if (StringUtils.isEmpty(item.getParentDictCode())) {
                        DictDataVo data = CrmBeanUtil.copy(item, DictDataVo.class);
                        if (CollectionUtil.listNotEmptyNotSizeZero(dictGroup.getConfList())) {
                            Map<String, String> extendMap = new HashMap<>(16);
                            for (DictAttrConfRedisVo conf :
                                    dictGroup.getConfList()) {
                                Object obj = ReflectUtil.getFiledValueByName(item, conf.getExtField());
                                extendMap.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
                            }
                            data.setExtendMap(extendMap);
                        }
                        list.add(data);
                    }
                });
            }
        }
        return list;
    }

    /**
     * 获取字典的直接下级Map
     *
     * @param dictTypeCode 数据字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static Map<String, String> getChildrenMap(String dictTypeCode, String dictCode) {
        Map<String, String> map = new HashMap<>(8);
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            DictGroupRedisVo group = getDictGroup(dictTypeCode);
            if (group != null) {
                List<DictDataRedisVo> dataList = group.getDataList();
                if (dataList != null && dataList.size() > 0) {
                    map = dataList.stream().filter(item -> dictCode.equals(item.getParentDictCode())).collect(Collectors.toMap(DictDataRedisVo::getDictCode, DictDataRedisVo::getDictValue));
                }
            }
        }
        return map;
    }

    /**
     * 获取字典的直接下级List
     *
     * @param dictTypeCode 数据字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static List<DictDataVo> getChildrenList(String dictTypeCode, String dictCode) {
        List<DictDataVo> list = new ArrayList<>();
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            DictGroupRedisVo group = getDictGroup(dictTypeCode);
            if (group != null) {
                List<DictDataRedisVo> dataList = group.getDataList();
                if (dataList != null && dataList.size() > 0) {
                    List<DictDataRedisVo> children = dataList.stream().filter(item -> dictCode.equals(item.getParentDictCode())).collect(Collectors.toList());
                    children.forEach(item -> {
                        DictDataVo data = CrmBeanUtil.copy(item, DictDataVo.class);
                        if (CollectionUtil.listNotEmptyNotSizeZero(group.getConfList())) {
                            Map<String, String> extendMap = new HashMap<>(16);
                            for (DictAttrConfRedisVo conf :
                                    group.getConfList()) {
                                Object obj = ReflectUtil.getFiledValueByName(item, conf.getExtField());
                                extendMap.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
                            }
                            data.setExtendMap(extendMap);
                        }
                        list.add(data);
                    });
                }
            }
        }
        return list;
    }

    /**
     * 获取扩展字段Map
     *
     * @param dictTypeCode 数据字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static Map<String, String> getExtendMap(String dictTypeCode, String dictCode) {
        Map<String, String> map = new HashMap<>(8);
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            DictGroupRedisVo group = getDictGroup(dictTypeCode);
            if (group != null) {
                List<DictAttrConfRedisVo> confList = group.getConfList();
                if (confList != null && confList.size() > 0) {
                    for (DictDataRedisVo data :
                            group.getDataList()) {
                        if (dictCode.equals(data.getDictCode())) {
                            for (DictAttrConfRedisVo conf :
                                    confList) {
                                Object obj = ReflectUtil.getFiledValueByName(data, conf.getExtField());
                                map.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
                            }
                        }
                    }
                }
            }
        }
        return map;
    }

    private static DictGroupRedisVo getDictGroup(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            Map<String, DictGroupRedisVo> dictGroupMap = getDictGroupMap(Collections.singletonList(dictTypeCode));
            if (!dictGroupMap.isEmpty()) {
                return dictGroupMap.get(dictTypeCode);
            }
        }
        return null;
    }

    private static Map<String, DictGroupRedisVo> getDictGroupMap(List<String> dictTypeCodeList) {
        Map<String, DictGroupRedisVo> map = new HashMap<>(16);
        try {
            if (dictTypeCodeList != null && dictTypeCodeList.size() > 0) {
                List<String> unlockCodeList = dictTypeCodeList.stream().filter(
                        dictTypeCode -> !checkLock(dictTypeCode)
                ).collect(Collectors.toList());
                List<String> notCacheCodeList = new ArrayList<>();
                for (String dictTypeCode :
                        unlockCodeList) {
                    Object o = redisService.get(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
                    if (o == null) {
                        notCacheCodeList.add(dictTypeCode);
                    } else {
                        map.put(dictTypeCode, (DictGroupRedisVo) o);
                    }
                }
                if (notCacheCodeList.size() > 0) {
                    Result<List<DictGroupRedisVo>> result = mdmDictFeign.getDictGroupRedisList(dictTypeCodeList);
                    if (result.isSuccess()) {
                        List<DictGroupRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, DictGroupRedisVo> collect = list.stream().collect(Collectors.toMap(DictGroupRedisVo::getDictTypeCode, v -> v));
                            for (String dictTypeCode :
                                    notCacheCodeList) {
                                if (!collect.containsKey(dictTypeCode)) {
                                    lock(dictTypeCode);
                                } else {
                                    addRedisCache(collect.get(dictTypeCode));
                                    map.put(dictTypeCode, collect.get(dictTypeCode));
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            log.error("数据字典工具类异常：{}", e);
        }
        return map;
    }

    private static List<DictGroupRedisVo> getDictGroupList(List<String> dictTypeCodeList) {
        Result<List<DictGroupRedisVo>> result = mdmDictFeign.getDictGroupRedisList(dictTypeCodeList);
        if (result.isSuccess()) {
            List<DictGroupRedisVo> list = result.getResult();
            if (list != null && list.size() > 0) {
                return list;
            }
        } else {
            log.error("远程查询数据字典失败:{}", result.getMessage());
        }
        return new ArrayList<>();
    }

    /**
     * 删除数据字典缓存
     *
     * @param dictTypeCode
     */
    public static void deleteRedisCache(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            redisService.del(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
            redisService.del(RedisParam.DICT_TYPE_LOCK + dictTypeCode);
        }
    }

    /**
     * 添加数据字典缓存
     *
     * @param group
     */
    private static void addRedisCache(DictGroupRedisVo group) {
        if (group != null) {
            redisService.set(RedisParam.DICT_TYPE_GROUP + group.getDictTypeCode(), group);
        }
    }

    /**
     * 加锁
     *
     * @param dictTypeCode
     */
    private static void lock(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            redisService.setSeconds(RedisParam.DICT_TYPE_LOCK + dictTypeCode, dictTypeCode, RedisParam.TIME5);
        }
    }

    /**
     * 检查是否被锁
     *
     * @param dictTypeCode
     * @return
     */
    private static boolean checkLock(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            return redisService.get(RedisParam.DICT_TYPE_LOCK + dictTypeCode) != null;
        }
        return false;
    }

    private static List<DictDataVo> generateTree(List<DictDataVo> totalList) {

        //构建树list
        List<DictDataVo> treeList = new ArrayList<>();
        //当前操作层级数据
        List<DictDataVo> curLevelList = new ArrayList<>();
        //未操作数据
        List<DictDataVo> restList = new ArrayList<>();

        //key:id
        Map<String, DictDataVo> totalMap = totalList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, v -> v));

        //查找第一层
        for (DictDataVo item : totalList) {
            if (StringUtils.isEmpty(item.getParentDictCode()) || !totalMap.containsKey(item.getParentDictCode())) {
                treeList.add(item);
                curLevelList.add(item);
            } else {
                restList.add(item);
            }
        }

        //构建数据，从第二层开始
        while (curLevelList.size() > 0 && restList.size() > 0) {
            List<DictDataVo> restTempList = new ArrayList<>();
            List<DictDataVo> curLevelTempList = new ArrayList<>();
            Map<String, String> curLevelMap = curLevelList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictCode));
            Map<String, List<DictDataVo>> curLevelChildrenMap = new HashMap<>();

            for (DictDataVo item : restList) {
                if (curLevelMap.containsKey(item.getParentDictCode())) {
                    curLevelTempList.add(item);

                    List<DictDataVo> childrenList = new ArrayList<>();
                    if (curLevelChildrenMap.containsKey(item.getParentDictCode())) {
                        childrenList.addAll(curLevelChildrenMap.get(item.getParentDictCode()));
                    }
                    childrenList.add(item);
                    curLevelChildrenMap.put(item.getParentDictCode(), childrenList);
                } else {
                    restTempList.add(item);
                }
            }

            for (DictDataVo item : curLevelList) {
                if (curLevelChildrenMap.containsKey(item.getDictCode())) {
                    item.setChildren(curLevelChildrenMap.get(item.getDictCode()));
                }
            }

            curLevelList.clear();
            curLevelList.addAll(curLevelTempList);
            restList.clear();
            restList.addAll(restTempList);
        }

        return treeList;
    }
}
