package com.biz.crm.util;

import com.alibaba.fastjson.JSON;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.nebular.sfa.worksign.req.SfaApplyTimeInfoReqVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignRuleRespVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignSpecialRespVo;
import org.apache.commons.lang.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author ren.gang
 * @ClassName SfaSignUtils.java
 * @Description Sfa系统-考勤管理相关工具类
 * @createTime 2020年12月02日 14:08:00
 */
public class SfaSignUtils {

    /**
     * 验证开始时间和结束时间
     * 1、空校验
     * 2、日期顺序校验
     * @param startDate
     * @param endDate
     * @return 两个日期的间隔
     */
    public static long verifyStartAndEndDate(String startDate, String endDate) {
        // 校验开始时间格式是否正确
        try {
            LocalDate.parse(startDate);
        } catch (Exception e) {
            throw new BusinessException("开始时间错误。日期格式为：yyyy-MM-dd");
        }
        // 校验结束时间格式
        try {
            LocalDate.parse(endDate);
        } catch (Exception e) {
            throw new BusinessException("结束时间错误。日期格式为：yyyy-MM-dd");
        }
        // 开始时间到结束时间的间隔数。开始和结束一致为0
        long intervalNum = LocalDate.parse(startDate).until(LocalDate.parse(endDate), ChronoUnit.DAYS);
        // 校验开始时间不能超过结束时间
        if(intervalNum < 0) {
            throw new BusinessException("开始时间超过结束时间");
        }
        return intervalNum;
    }

    // 详情日期 日期类型全字段
    public static final String TIME_TYPE_ALL_STR = "1,2,3";

    /**
     * 验证考勤相关申请功能，数据信息
     * 1、当前方法参数不能为空，或者和正确参数不匹配
     * 2、开始-结束时间 和规则
     * 3、日期明细列表
     * 4、申请天数
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @param applyList 日期明细列表
     * @param applyDays 申请时长
     * @return
     */
    public static void verifyApply(String startDate, String endDate, List<SfaApplyTimeInfoReqVo> applyList, String applyDays) {
        // 校验并返回 申请时长
        BigDecimal countApplyDays = countApplyDays(startDate, endDate, applyList);
        // 校验申请时间是否正确
        try {
            new BigDecimal(applyDays);
        }catch (Exception e) {
            throw new BusinessException("申请时长错误");
        }
        if(new BigDecimal(applyDays).compareTo(countApplyDays) != 0) {
            throw new BusinessException("申请时长和明细时间不一致");
        }
    }

    /**
     * 验证考勤相关申请功能，数据信息。返回申请时长
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @param applyList 日期明细列表
     * @return 申请时长
     */
    public static void verifyApply(String startDate, String endDate, List<SfaApplyTimeInfoReqVo> applyList) {
        // 校验开始结束日期
        long intervalNum = verifyStartAndEndDate(startDate, endDate);

        // 日期明细列表空检验
        if(applyList == null || applyList.size() == 0) {
            throw new BusinessException("日期明细不能为空");
        }
        // 数据条数判断。间隔=0时，包含一条数据;间隔>=1是，包含两条数据
        if(applyList.size() == 1) {
            if(intervalNum != 0) {
                throw new BusinessException("日期明细列表和 申请开始结束时间不一致。至少包含开始和结束日期的明细");
            }
        } else if(applyList.size() == 2) {
            if(intervalNum == 0) {
                throw new BusinessException("日期明细列表和 申请开始结束时间不一致。至少包含开始和结束日期的明细");
            }
        } else {
            throw new BusinessException("日期明细列表只能包含开始和结束日期的明细");
        }

        // 遍历校验日期明细
        for(int r =0; r < applyList.size(); r ++) {
            SfaApplyTimeInfoReqVo vo = applyList.get(r);
            // 日期校验
            try {
                LocalDate.parse(vo.getTimeStr());
            }catch (Exception e) {
                throw new BusinessException("日期明细列表中 日期 错误。日期格式为：yyyy-MM-dd");
            }
            // 日期类型校验
            if(StringUtils.isEmpty(vo.getTimeType())
                    || TIME_TYPE_ALL_STR.indexOf(vo.getTimeType()) == -1) {
                throw new BusinessException("日期明细列表中 日期类型 错误。日期类型(1全天，2上午，3下午)");
            }
            if(r == 0) { // 第一条数据是开始日期
                if(!vo.getTimeStr().equals(startDate)){
                    throw new BusinessException("日期明细列表和 申请开始时间不一致");
                }
            } else if(r == applyList.size()-1) { //最后一条数据是结束日期
                if(!vo.getTimeStr().equals(endDate)) {
                    throw new BusinessException("日期明细列表和 申请结束时间不一致");
                }
            } else { // 其他日期要按照顺序排列
                if(LocalDate.parse(startDate).plusDays(r).until(LocalDate.parse(vo.getTimeStr()), ChronoUnit.DAYS) != 0) {
                    throw new BusinessException("日期明细列表和 申请开始结束时间不一致");
                }
            }
        }
    }


    /**
     * 获取日期之间的时长
     * @param startDate 开始时间
     * @param endDate 结束时间
     * @param applyList 日期明细
     * @return
     */
    public static BigDecimal countApplyDays(String startDate, String endDate, List<SfaApplyTimeInfoReqVo> applyList) {
        // 补全数据
        List<SfaApplyTimeInfoReqVo> returnList = fillTimeInfo(applyList, startDate, endDate);
        // 统计申请时间(天)
        BigDecimal countApplyDays = new BigDecimal(0);
        for(SfaApplyTimeInfoReqVo vo : returnList){
            // 统计申请时长
            if(SfaCommonEnum.dataTimeType.ALL_DAY.getValue().equals(vo.getTimeType())) {
                countApplyDays = countApplyDays.add(new BigDecimal(1));
            } else {
                countApplyDays = countApplyDays.add(new BigDecimal(0.5));
            }
        };

        return countApplyDays;
    }

    /**
     * 补全日期明细
     * @param applyList 原明细列表
     * @param startDate 开始时间
     * @param endDate 结束时间
     * @return
     */
    public static List<SfaApplyTimeInfoReqVo> fillTimeInfo(List<SfaApplyTimeInfoReqVo> applyList, String startDate, String endDate) {
        verifyApply(startDate, endDate, applyList);
        // 补全数据
        List<SfaApplyTimeInfoReqVo> returnList = new ArrayList<>();
        // 获取间隔时间
        long intervalNum = LocalDate.parse(startDate).until(LocalDate.parse(endDate), ChronoUnit.DAYS);
        if(intervalNum <= 1 || applyList.size() == intervalNum+1) {
            returnList = applyList;
        } else {
            returnList.add(applyList.get(0));
            for(int r =1; r < intervalNum; r ++) {
                SfaApplyTimeInfoReqVo vo = new SfaApplyTimeInfoReqVo();
                vo.setTimeStr(LocalDate.parse(startDate).plusDays(r).toString());
                vo.setTimeType(SfaCommonEnum.dataTimeType.ALL_DAY.getValue());
                returnList.add(vo);
            }
            returnList.add(applyList.get(applyList.size()-1));
        }

        return returnList;
    }

    /**
     * 生成新的日期明细。明细的开始时间和结束时间置为 时间段内
     * @param startDate 开始时间
     * @param endDate 结束时间
     * @param applyList 原日期明细
     * @return
     */
    public static List<SfaApplyTimeInfoReqVo> createNewTimeInfo(String startDate, String endDate, List<SfaApplyTimeInfoReqVo> applyList) {
        SfaApplyTimeInfoReqVo startInfo = applyList.get(0);
        SfaApplyTimeInfoReqVo endInfo = applyList.get(applyList.size() - 1);

        // 开始时间设置
        if(StringUtils.isNotBlank(startDate)) {
            LocalDate start = LocalDate.parse(startDate);
            if(start.isAfter(LocalDate.parse(endInfo.getTimeStr()))) {
                return null;
            }
            if(start.isAfter(LocalDate.parse(startInfo.getTimeStr()))) {
                startInfo = new SfaApplyTimeInfoReqVo(startDate, SfaCommonEnum.dataTimeType.ALL_DAY.getValue());
            }
        }
        if(StringUtils.isNotBlank(endDate)) {
            LocalDate end = LocalDate.parse(endDate);
            // 排除时间在开始结束之外的
            if(end.isBefore(LocalDate.parse(startInfo.getTimeStr()))) {
                return null;
            }
            if(end.isBefore(LocalDate.parse(endInfo.getTimeStr()))) {
                endInfo = new SfaApplyTimeInfoReqVo(endDate, SfaCommonEnum.dataTimeType.ALL_DAY.getValue());
            }
        }
        List<SfaApplyTimeInfoReqVo> result = new ArrayList<>();
        result.add(startInfo);
        if(!startInfo.getTimeStr().equals(endInfo.getTimeStr())) {
            result.add(endInfo);
        }

        return result;
    }

    /**
     * 统计新时间段内的天数
     * @param startDate
     * @param endDate
     * @param applyList
     * @return
     */
    public static BigDecimal countCreateNewTimeInfo(String startDate, String endDate, List<SfaApplyTimeInfoReqVo> applyList) {
        List<SfaApplyTimeInfoReqVo> list = createNewTimeInfo(startDate, endDate, applyList);
        if(list == null) {
            return new BigDecimal(0);
        }
        return countApplyDays(list.get(0).getTimeStr(), list.get(list.size()-1).getTimeStr(), list);
    }

    /**
     * 验证申请时间重复
     * @param applyBeginTime 申请开始时间
     * @param applyEndTime 申请结束时间
     * @param beginTime 原开始时间
     * @param endTime 原结束时间
     * @param applyTimeInfoList 申请日期明细
     * @param timeInfoList 原日期明细
     */
    public static void verifyDateRepeat(String applyBeginTime, String applyEndTime, String beginTime, String endTime
            , List<SfaApplyTimeInfoReqVo> applyTimeInfoList, List<SfaApplyTimeInfoReqVo> timeInfoList) {
        // 开始时间在结束时间之外或者结束时间在开始时间之前
        if(LocalDate.parse(applyBeginTime).isAfter(LocalDate.parse(endTime)) ||
                LocalDate.parse(applyEndTime).isBefore(LocalDate.parse(applyBeginTime))) {
            return ;
        }
        String startDate = ""; // 冲突开始时间
        String endDate = ""; // 冲突结束时间
        // 标记冲突开始时间。申请时间在原时间之后，为 申请时间，否则为 原开始时间
        if(LocalDate.parse(applyBeginTime).isAfter(LocalDate.parse(beginTime))) {
            startDate = applyBeginTime;
        } else {
            startDate = beginTime;
        }
        // 标记冲突结束时间。申请时间在原时间之后，为 原结束时间，否则为 申请结束时间
        if(LocalDate.parse(applyEndTime).isAfter(LocalDate.parse(endTime))) {
            endDate = endTime;
        } else {
            endDate = applyEndTime;
        }
        // 如果冲突开始和结束时间一致时，需要根据明细中类型来判断是否冲突
        if(startDate.equals(endDate)) {
            timeInfoList = fillTimeInfo(timeInfoList, timeInfoList.get(0).getTimeStr(), timeInfoList.get(timeInfoList.size()-1).getTimeStr());
            for(SfaApplyTimeInfoReqVo info: timeInfoList) {
                if(info.getTimeStr().equals(startDate)) {
                    // 原请假为全天,直接判定为冲突
                    if (SfaCommonEnum.dataTimeType.ALL_DAY.getValue().equals(info.getTimeType())) {
                        throw new BusinessException("时间冲突:" + info.getTimeStr());
                    }
                    // 遍历当前申请时间判断当天是否冲突
                    for (SfaApplyTimeInfoReqVo nowInfo : applyTimeInfoList) {
                        if (info.getTimeStr().equals(nowInfo.getTimeStr())) {
                            // 新申请为全天或者，半天时间一致
                            if (SfaCommonEnum.dataTimeType.ALL_DAY.getValue().equals(nowInfo.getTimeType()) || info.getTimeType().equals(nowInfo.getTimeType())) {
                                throw new BusinessException("时间冲突:" + info.getTimeStr());
                            }
                        }
                    }
                }
            }
        } else {
            throw new BusinessException("时间冲突："+startDate + "至" + endDate);
        }
    }

    /**
     * 判断日期是否 考勤规则内的 工作日
     * @param rule
     * @param date
     * @return true 0工作日(工作日+特殊必须打卡)，1非工作日(非工作日+<特殊不用打卡(已注释)>)，2非工作日(节假日)
     */
    public static String  isWorkDay(SfaWorkSignRuleRespVo rule, String date) {
        // 匹配 法定假日
        if(YesNoEnum.yesNoEnum.YES.getValue().equals(rule.getHolidayWhether())
                && HoliDayUtils.isHoliday(date)) {
            return YesNoEnum.yesNoEnum.TWO.getValue();
        }
        // 匹配特殊日期-不打卡
//        if(rule.getSfaWorkSignSpecialNotRespVos() != null) {
//            for (SfaWorkSignSpecialRespVo vo : rule.getSfaWorkSignSpecialNotRespVos()) {
//                if (vo.getWssDate().equals(date)) {
//                    return YesNoEnum.yesNoEnum.ONE.getValue();
//                }
//            }
//        }
        // 匹配特殊日期-必须打卡
        if(rule.getSfaWorkSignSpecialMustRespVos() != null) {
            for (SfaWorkSignSpecialRespVo vo : rule.getSfaWorkSignSpecialMustRespVos()) {
                if (vo.getWssDate().equals(date)) {
                    return YesNoEnum.yesNoEnum.ZERO.getValue();
                }
            }
        }
        // 匹配工作日
        LocalDate localDate = LocalDate.parse(date);
        if(rule.getWorkingDay().indexOf(localDate.getDayOfWeek().getValue()+"") == -1) { // 非工作日
            return YesNoEnum.yesNoEnum.ONE.getValue();
        } else {
            return YesNoEnum.yesNoEnum.ZERO.getValue();
        }
    }

    /**
     * 增加 抵扣信息 [{"id":"多少天"},{"8273482":"0.5"}]
     * @param old 原抵扣信息
     * @param id 被抵扣id
     * @param days 抵扣天数
     * @return
     */
    public static String addDeductionIds(String old,String id, BigDecimal days){
        List<Map> list = new ArrayList<>();
        if(StringUtils.isNotBlank(old)) {
            list = JSON.parseArray(old, Map.class);
        }
        Map map = new HashMap();
        map.put(id, days);
        list.add(map);
        return JSON.toJSONString(list);
    }

    /**
     * 回滚抵扣信息。抵扣天数为0时，自动移除该条抵扣信息
     * @param deductionIds 原抵扣信息
     * @param id 被抵扣id
     * @param days 抵扣天数
     * @return
     */
    public static String backDeductionIds(String deductionIds,String id, BigDecimal days){
        List<Map> result = new ArrayList<>();
        List<Map> list = decodeDeductionIds(deductionIds);
        for(Map map : list) {
            String oldId = getDeductionId(map);
            BigDecimal oldDays = getDeductionDays(map);
            if(oldId.equals(id)) {
                days = oldDays.subtract(days);
                if(days.compareTo(BigDecimal.ZERO) == 0) {
                    continue;
                }
                map.put(id, days);
            }

            result.add(map);
        }
        return JSON.toJSONString(list);
    }


    /**
     * 解码 已使用ids。[{"id":"多少天"},{"8273482":"0.5"}]
     * @param deductionIds
     * @return
     */
    public static List<Map> decodeDeductionIds(String deductionIds) {
        return JSON.parseArray(deductionIds, Map.class);
    }

    /**
     * 获取抵扣的天数
     * @param map
     * @return
     */
    public static BigDecimal getDeductionDays(Map map) {
        for(Object value : map.values()) {
            return new BigDecimal(value.toString());
        }
        return new BigDecimal(0);
    }

    /**
     * 获取抵扣的id
     * @param map
     * @return
     */
    public static String getDeductionId(Map map) {
        for(Object key : map.keySet()) {
            return key.toString();
        }
        return "";
    }

}
