package com.biz.crm.util;

import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mdm.org.MdmOrgRedisFeign;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgParentChildrenRedisVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRedisVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgReloadRedisVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.service.RedisService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 企业组织查询工具类
 */
@Slf4j
public class OrgUtil {

    private static RedisService redisService;

    private static MdmOrgRedisFeign mdmOrgRedisFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmOrgRedisFeign(MdmOrgRedisFeign orgRedisFeign) {
        mdmOrgRedisFeign = orgRedisFeign;
    }

    /**
     * 根据组织编码获取组织详情
     *
     * @param orgCode 组织编码
     * @return
     */
    public static MdmOrgRespVo getOrgByCode(String orgCode) {
        if (orgCode != null && !"".equals(orgCode)) {
            Map<String, MdmOrgRedisVo> orgRedisMap = getOrgSelfRedisMapFromLocalFirst(Collections.singletonList(orgCode));
            if (orgRedisMap.containsKey(orgCode)) {
                MdmOrgRedisVo mdmOrgRedisVo = orgRedisMap.get(orgCode);
                return CrmBeanUtil.copy(mdmOrgRedisVo, MdmOrgRespVo.class);
            }
        }
        return null;
    }

    /**
     * 根据组织编码集合获取组织详情
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getOrgByCodeList(List<String> orgCodeList) {
        List<MdmOrgRespVo> list = new ArrayList<>();
        if (orgCodeList != null && orgCodeList.size() > 0) {
            Map<String, MdmOrgRedisVo> orgRedisMap = getOrgSelfRedisMapFromLocalFirst(orgCodeList);
            if (!orgRedisMap.isEmpty()) {
                list.addAll(orgRedisMap.values().stream().map(item -> CrmBeanUtil.copy(item, MdmOrgRespVo.class)).collect(Collectors.toList()));
            }
        }
        return list;
    }


    /**
     * 根据组织编码获取组织及组织全部上级
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListIncludeSelf(String orgCode) {
        if (orgCode != null && !"".equals(orgCode)) {
            Map<String, MdmOrgParentChildrenRedisVo> map = getOrgParentRedisMapFromLocalFirst(Collections.singletonList(orgCode));
            if (!map.isEmpty()) {
                MdmOrgParentChildrenRedisVo redisVo = map.get(orgCode);
                if (redisVo != null) {
                    List<MdmOrgRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, MdmOrgRespVo.class);
                    }
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码集合获取组织及组织全部上级
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListIncludeSelf(List<String> orgCodeList) {
        if (orgCodeList != null && orgCodeList.size() > 0) {
            Map<String, MdmOrgParentChildrenRedisVo> map = getOrgParentRedisMapFromLocalFirst(orgCodeList);
            Set<MdmOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String orgCode :
                        orgCodeList) {
                    if (!map.containsKey(orgCode)) {
                        continue;
                    }
                    MdmOrgParentChildrenRedisVo redisVo = map.get(orgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmOrgRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码获取组织（不含自己）全部上级
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListExcludeSelf(String orgCode) {
        if (orgCode != null && !"".equals(orgCode)) {
            List<MdmOrgRespVo> parentOrgListIncludeSelf = getParentOrgListIncludeSelf(orgCode);
            if (parentOrgListIncludeSelf != null && parentOrgListIncludeSelf.size() > 0) {
                return parentOrgListIncludeSelf.stream().filter(item -> !orgCode.equals(item.getOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的组织
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListExcludeSelf(List<String> orgCodeList) {
        if (orgCodeList != null && orgCodeList.size() > 0) {
            Map<String, MdmOrgParentChildrenRedisVo> map = getOrgParentRedisMapFromLocalFirst(orgCodeList);
            Set<MdmOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String orgCode :
                        orgCodeList) {
                    if (!map.containsKey(orgCode)) {
                        continue;
                    }
                    MdmOrgParentChildrenRedisVo redisVo = map.get(orgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !orgCode.equals(item.getOrgCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmOrgRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码集合获取组织（不含参数中的任何一个）全部上级
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListExcludeAnySelf(List<String> orgCodeList) {
        if (orgCodeList != null && orgCodeList.size() > 0) {
            List<MdmOrgRespVo> parentOrgListExcludeSelf = getParentOrgListExcludeSelf(orgCodeList);
            if (parentOrgListExcludeSelf != null && parentOrgListExcludeSelf.size() > 0) {
                final Set<String> orgCodeSet = new HashSet<>(orgCodeList);
                return parentOrgListExcludeSelf.stream().filter(item -> !orgCodeSet.contains(item.getOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }


    /**
     * 根据组织编码获取组织及组织全部下级
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListIncludeSelf(String orgCode) {
        if (orgCode != null && !"".equals(orgCode)) {
            Map<String, MdmOrgParentChildrenRedisVo> map = getOrgChildrenRedisMapFromLocalFirst(Collections.singletonList(orgCode));
            if (!map.isEmpty()) {
                MdmOrgParentChildrenRedisVo redisVo = map.get(orgCode);
                if (redisVo != null) {
                    List<MdmOrgRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, MdmOrgRespVo.class);
                    }
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码集合获取组织及组织全部下级
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListIncludeSelf(List<String> orgCodeList) {
        if (orgCodeList != null && orgCodeList.size() > 0) {
            Map<String, MdmOrgParentChildrenRedisVo> map = getOrgChildrenRedisMapFromLocalFirst(orgCodeList);
            Set<MdmOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String orgCode :
                        orgCodeList) {
                    if (!map.containsKey(orgCode)) {
                        continue;
                    }
                    MdmOrgParentChildrenRedisVo redisVo = map.get(orgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmOrgRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码获取组织（不含自己）全部下级
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListExcludeSelf(String orgCode) {
        if (orgCode != null && !"".equals(orgCode)) {
            List<MdmOrgRespVo> parentOrgListIncludeSelf = getChildrenOrgListIncludeSelf(orgCode);
            if (parentOrgListIncludeSelf != null && parentOrgListIncludeSelf.size() > 0) {
                return parentOrgListIncludeSelf.stream().filter(item -> !orgCode.equals(item.getOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的组织
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListExcludeSelf(List<String> orgCodeList) {
        if (orgCodeList != null && orgCodeList.size() > 0) {
            Map<String, MdmOrgParentChildrenRedisVo> map = getOrgChildrenRedisMapFromLocalFirst(orgCodeList);
            Set<MdmOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String orgCode :
                        orgCodeList) {
                    if (!map.containsKey(orgCode)) {
                        continue;
                    }
                    MdmOrgParentChildrenRedisVo redisVo = map.get(orgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !orgCode.equals(item.getOrgCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmOrgRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据组织编码集合获取组织（不含参数中任何一个）全部下级
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListExcludeAnySelf(List<String> orgCodeList) {
        if (orgCodeList != null && orgCodeList.size() > 0) {
            List<MdmOrgRespVo> parentOrgListExcludeSelf = getChildrenOrgListExcludeSelf(orgCodeList);
            if (parentOrgListExcludeSelf != null && parentOrgListExcludeSelf.size() > 0) {
                final Set<String> orgCodeSet = new HashSet<>(orgCodeList);
                return parentOrgListExcludeSelf.stream().filter(item -> !orgCodeSet.contains(item.getOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }


    /**
     * 根据组织编码获取组织及组织全部上级（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListIncludeSelfEnable(String orgCode) {
        return getParentOrgListIncludeSelf(orgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织及组织全部上级（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListIncludeSelfEnable(List<String> orgCodeList) {
        return getParentOrgListIncludeSelf(orgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码获取组织（不含自己）全部上级（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListExcludeSelfEnable(String orgCode) {
        return getParentOrgListExcludeSelf(orgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的组织（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListExcludeSelfEnable(List<String> orgCodeList) {
        return getParentOrgListExcludeSelf(orgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含参数中的任何一个）全部上级（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getParentOrgListExcludeAnySelfEnable(List<String> orgCodeList) {
        return getParentOrgListExcludeAnySelf(orgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据组织编码获取组织及组织全部下级（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListIncludeSelfEnable(String orgCode) {
        return getChildrenOrgListIncludeSelf(orgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织及组织全部下级（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListIncludeSelfEnable(List<String> orgCodeList) {
        return getChildrenOrgListIncludeSelf(orgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码获取组织（不含自己）全部下级（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListExcludeSelfEnable(String orgCode) {
        return getChildrenOrgListExcludeSelf(orgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的组织（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListExcludeSelfEnable(List<String> orgCodeList) {
        return getChildrenOrgListExcludeSelf(orgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含参数中任何一个）全部下级（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<MdmOrgRespVo> getChildrenOrgListExcludeAnySelfEnable(List<String> orgCodeList) {
        return getChildrenOrgListExcludeAnySelf(orgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据组织编码获取组织及组织全部上级组织编码
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getParentOrgCodeListIncludeSelf(String orgCode) {
        return getParentOrgListIncludeSelf(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织及组织全部上级组织编码
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getParentOrgCodeListIncludeSelf(List<String> orgCodeList) {
        return getParentOrgListIncludeSelf(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码获取组织（不含自己）全部上级组织编码
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getParentOrgCodeListExcludeSelf(String orgCode) {
        return getParentOrgListExcludeSelf(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部上级组织编码，如果本身有上下级关系，则会返回处于上级的组织
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getParentOrgCodeListExcludeSelf(List<String> orgCodeList) {
        return getParentOrgListExcludeSelf(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含参数中的任何一个）全部上级组织编码
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getParentOrgCodeListExcludeAnySelf(List<String> orgCodeList) {
        return getParentOrgListExcludeAnySelf(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }


    /**
     * 根据组织编码获取组织及组织全部下级组织编码
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getChildrenOrgCodeListIncludeSelf(String orgCode) {
        return getChildrenOrgListIncludeSelf(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织及组织全部下级组织编码
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getChildrenOrgCodeListIncludeSelf(List<String> orgCodeList) {
        return getChildrenOrgListIncludeSelf(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码获取组织（不含自己）全部下级组织编码
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getChildrenOrgCodeListExcludeSelf(String orgCode) {
        return getChildrenOrgListExcludeSelf(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部下级组织编码，如果参数本身有上下级关系，则会返回处于下级的组织
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getChildrenOrgCodeListExcludeSelf(List<String> orgCodeList) {
        return getChildrenOrgListExcludeSelf(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含参数中任何一个）全部下级组织编码
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getChildrenOrgCodeListExcludeAnySelf(List<String> orgCodeList) {
        return getChildrenOrgListExcludeAnySelf(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }


    /**
     * 根据组织编码获取组织及组织全部上级组织编码（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getParentOrgCodeListIncludeSelfEnable(String orgCode) {
        return getParentOrgListIncludeSelfEnable(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织及组织全部上级组织编码（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getParentOrgCodeListIncludeSelfEnable(List<String> orgCodeList) {
        return getParentOrgListIncludeSelfEnable(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码获取组织（不含自己）全部上级组织编码（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getParentOrgCodeListExcludeSelfEnable(String orgCode) {
        return getParentOrgListExcludeSelfEnable(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部上级组织编码，如果本身有上下级关系，则会返回处于上级的组织（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getParentOrgCodeListExcludeSelfEnable(List<String> orgCodeList) {
        return getParentOrgListExcludeSelfEnable(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含参数中的任何一个）全部上级组织编码（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getParentOrgCodeListExcludeAnySelfEnable(List<String> orgCodeList) {
        return getParentOrgListExcludeAnySelfEnable(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }


    /**
     * 根据组织编码获取组织及组织全部下级组织编码（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getChildrenOrgCodeListIncludeSelfEnable(String orgCode) {
        return getChildrenOrgListIncludeSelfEnable(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织及组织全部下级组织编码（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getChildrenOrgCodeListIncludeSelfEnable(List<String> orgCodeList) {
        return getChildrenOrgListIncludeSelfEnable(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码获取组织（不含自己）全部下级组织编码（只查启用）
     *
     * @param orgCode 组织编码
     * @return
     */
    public static List<String> getChildrenOrgCodeListExcludeSelfEnable(String orgCode) {
        return getChildrenOrgListExcludeSelfEnable(orgCode).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含自己）全部下级组织编码，如果参数本身有上下级关系，则会返回处于下级的组织（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getChildrenOrgCodeListExcludeSelfEnable(List<String> orgCodeList) {
        return getChildrenOrgListExcludeSelfEnable(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据组织编码集合获取组织（不含参数中任何一个）全部下级组织编码（只查启用）
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    public static List<String> getChildrenOrgCodeListExcludeAnySelfEnable(List<String> orgCodeList) {
        return getChildrenOrgListExcludeAnySelfEnable(orgCodeList).stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
    }


    /**
     * 清除全部组织Redis缓存和本地缓存
     */
    public static void deleteAllCache() {
        redisService.del(RedisParam.MDM_ORG_LOCK_HASH);
        redisService.del(RedisParam.MDM_ORG_SELF_HASH);
        redisService.del(RedisParam.MDM_ORG_PARENT_HASH);
        redisService.del(RedisParam.MDM_ORG_CHILDREN_HASH);
        ThreadLocalUtil.delOrg();
    }

    /**
     * 重新加载redis缓存
     */
    public static void reloadAllCache() {
        deleteAllCache();
        Result<List<MdmOrgReloadRedisVo>> result = mdmOrgRedisFeign.findRedisReloadList();
        if (result.isSuccess()) {
            List<MdmOrgReloadRedisVo> list = result.getResult();
            if (list != null && list.size() > 0) {
                List<List<MdmOrgReloadRedisVo>> partition = Lists.partition(list, 500);
                for (List<MdmOrgReloadRedisVo> item :
                        partition) {
                    Map<String, MdmOrgRedisVo> selfMap = item.stream().collect(Collectors.toMap(MdmOrgReloadRedisVo::getOrgCode, MdmOrgReloadRedisVo::getSelf));
                    Map<String, List<MdmOrgRedisVo>> parentMap = item.stream().collect(Collectors.toMap(MdmOrgReloadRedisVo::getOrgCode, MdmOrgReloadRedisVo::getParentList));
                    Map<String, List<MdmOrgRedisVo>> childrenMap = item.stream().collect(Collectors.toMap(MdmOrgReloadRedisVo::getOrgCode, MdmOrgReloadRedisVo::getChildrenList));
                    redisService.hmset(RedisParam.MDM_ORG_SELF_HASH, new HashMap<Object, Object>(selfMap));
                    redisService.hmset(RedisParam.MDM_ORG_PARENT_HASH, new HashMap<Object, Object>(parentMap));
                    redisService.hmset(RedisParam.MDM_ORG_CHILDREN_HASH, new HashMap<Object, Object>(childrenMap));
                }
            }
        }
    }

    /*--------------------------------------------------以下是私有方法--------------------------------------------------*/

    /**
     * 获取组织缓存数据，优先从本地缓存取，再从redis取
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    private static Map<String, MdmOrgRedisVo> getOrgSelfRedisMapFromLocalFirst(List<String> orgCodeList) {
        Map<String, MdmOrgRedisVo> map = new HashMap<>(16);
        try {
            if (orgCodeList != null && orgCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String orgCode :
                        orgCodeList) {
                    if (ThreadLocalUtil.orgSelfContains(orgCode)) {
                        MdmOrgRedisVo fromOrgSelf = ThreadLocalUtil.getFromOrgSelf(orgCode);
                        map.put(orgCode, fromOrgSelf);
                    } else {
                        getFromRedisCodeSet.add(orgCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmOrgRedisVo> mapFromRedisFirst = getOrgSelfRedisMapFromRedisFirst(new ArrayList<>(orgCodeList));

                    for (String orgCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(orgCode)) {
                            map.put(orgCode, mapFromRedisFirst.get(orgCode));
                        }
                        ThreadLocalUtil.addToOrgSelf(orgCode, mapFromRedisFirst.get(orgCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取组织缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    private static Map<String, MdmOrgRedisVo> getOrgSelfRedisMapFromRedisFirst(List<String> orgCodeList) {
        Map<String, MdmOrgRedisVo> map = new HashMap<>(16);
        try {
            if (orgCodeList != null && orgCodeList.size() > 0) {
                List<MdmOrgRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_ORG_LOCK_HASH, new HashSet<Object>(orgCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = orgCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> orgListFromRedis = redisService.hmget(RedisParam.MDM_ORG_SELF_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (orgListFromRedis != null && orgListFromRedis.size() > 0) {
                        List<MdmOrgRedisVo> collect = orgListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmOrgRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmOrgRedisVo::getOrgCode).collect(Collectors.toSet()));
                        }
                    }
                }

                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmOrgRedisVo>> result = mdmOrgRedisFeign.findRedisListByOrgCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmOrgRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmOrgRedisVo> collect = list.stream().collect(Collectors.toMap(MdmOrgRedisVo::getOrgCode, v -> v));

                            Map<String, MdmOrgRedisVo> cacheMap = new HashMap<>(16);
                            for (String orgCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(orgCode)) {
                                    lockCodeMap.put(orgCode, orgCode);
                                } else {
                                    redisVoList.add(collect.get(orgCode));
                                    cacheMap.put(orgCode, collect.get(orgCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_ORG_SELF_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_ORG_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmOrgRedisVo::getOrgCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取组织上级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    private static Map<String, MdmOrgParentChildrenRedisVo> getOrgParentRedisMapFromLocalFirst(List<String> orgCodeList) {
        Map<String, MdmOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (orgCodeList != null && orgCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String orgCode :
                        orgCodeList) {
                    if (ThreadLocalUtil.orgParentContains(orgCode)) {
                        MdmOrgParentChildrenRedisVo fromOrgParent = ThreadLocalUtil.getFromOrgParent(orgCode);
                        map.put(orgCode, fromOrgParent);
                    } else {
                        getFromRedisCodeSet.add(orgCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmOrgParentChildrenRedisVo> mapFromRedisFirst = getOrgParentRedisMapFromRedisFirst(new ArrayList<>(orgCodeList));

                    for (String orgCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(orgCode)) {
                            map.put(orgCode, mapFromRedisFirst.get(orgCode));
                        }
                        ThreadLocalUtil.addToOrgParent(orgCode, mapFromRedisFirst.get(orgCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取组织上级缓存数据，优先从redis缓存存取，再从数据库取
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    private static Map<String, MdmOrgParentChildrenRedisVo> getOrgParentRedisMapFromRedisFirst(List<String> orgCodeList) {
        Map<String, MdmOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (orgCodeList != null && orgCodeList.size() > 0) {
                List<MdmOrgParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_ORG_LOCK_HASH, new HashSet<Object>(orgCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = orgCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> orgListFromRedis = redisService.hmget(RedisParam.MDM_ORG_PARENT_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (orgListFromRedis != null && orgListFromRedis.size() > 0) {
                        List<MdmOrgParentChildrenRedisVo> collect = orgListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmOrgParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmOrgParentChildrenRedisVo::getOrgCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmOrgParentChildrenRedisVo>> result = mdmOrgRedisFeign.findRedisOrgParentListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmOrgParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmOrgParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmOrgParentChildrenRedisVo::getOrgCode, v -> v));

                            Map<String, MdmOrgParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String orgCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(orgCode)) {
                                    lockCodeMap.put(orgCode, orgCode);
                                } else {
                                    redisVoList.add(collect.get(orgCode));
                                    cacheMap.put(orgCode, collect.get(orgCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_ORG_PARENT_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_ORG_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmOrgParentChildrenRedisVo::getOrgCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取组织下级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    private static Map<String, MdmOrgParentChildrenRedisVo> getOrgChildrenRedisMapFromLocalFirst(List<String> orgCodeList) {
        Map<String, MdmOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (orgCodeList != null && orgCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String orgCode :
                        orgCodeList) {
                    if (ThreadLocalUtil.orgChildrenContains(orgCode)) {
                        MdmOrgParentChildrenRedisVo fromOrgChildren = ThreadLocalUtil.getFromOrgChildren(orgCode);
                        map.put(orgCode, fromOrgChildren);
                    } else {
                        getFromRedisCodeSet.add(orgCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmOrgParentChildrenRedisVo> mapFromRedisFirst = getOrgChildrenRedisMapFromRedisFirst(new ArrayList<>(orgCodeList));

                    for (String orgCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(orgCode)) {
                            map.put(orgCode, mapFromRedisFirst.get(orgCode));
                        }
                        ThreadLocalUtil.addToOrgChildren(orgCode, mapFromRedisFirst.get(orgCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取组织下级缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param orgCodeList 组织编码集合
     * @return
     */
    private static Map<String, MdmOrgParentChildrenRedisVo> getOrgChildrenRedisMapFromRedisFirst(List<String> orgCodeList) {
        Map<String, MdmOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (orgCodeList != null && orgCodeList.size() > 0) {
                List<MdmOrgParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_ORG_LOCK_HASH, new HashSet<Object>(orgCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = orgCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> orgListFromRedis = redisService.hmget(RedisParam.MDM_ORG_CHILDREN_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (orgListFromRedis != null && orgListFromRedis.size() > 0) {
                        List<MdmOrgParentChildrenRedisVo> collect = orgListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmOrgParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmOrgParentChildrenRedisVo::getOrgCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmOrgParentChildrenRedisVo>> result = mdmOrgRedisFeign.findRedisOrgChildrenListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmOrgParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmOrgParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmOrgParentChildrenRedisVo::getOrgCode, v -> v));

                            Map<String, MdmOrgParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String orgCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(orgCode)) {
                                    lockCodeMap.put(orgCode, orgCode);
                                } else {
                                    redisVoList.add(collect.get(orgCode));
                                    cacheMap.put(orgCode, collect.get(orgCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_ORG_CHILDREN_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_ORG_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmOrgParentChildrenRedisVo::getOrgCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

}
