package com.biz.crm.util.fee;

import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.fee.FeePoolOperationTypeExtendEnum;
import com.biz.crm.eunm.fee.FeePoolOperationTypeGroupEnum;
import com.biz.crm.eunm.fee.FeePoolTypeEnum;
import com.biz.crm.nebular.fee.pool.resp.FeeSelectOptionVo;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.util.DictUtil;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 费用池配置查询校验工具类
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
public class FeePoolConfigUtil {

    /**e
     * 获取折扣类型下拉框
     *
     * @return
     */
    public static List<FeeSelectOptionVo> getDiscountSelectList() {
        return DictUtil.list(DictConstant.FEE_POOL_DISCOUNT_TYPE).stream().map(item -> {
            FeeSelectOptionVo option = new FeeSelectOptionVo();
            option.setValue(item.getDictCode());
            option.setText(item.getDictValue());
            return option;
        }).collect(Collectors.toList());
    }

    /**
     * 获取操作类型编码集合
     *
     * @return
     */
    public static List<String> getOperationTypeList() {
        return DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE)
                .stream()
                .map(DictDataVo::getDictCode).collect(Collectors.toList());
    }

    /**
     * 获取操作类型编码集合，根据扩展字段枚举
     *
     * @param extendEnum
     * @return
     */
    public static List<String> getOperationTypeList(FeePoolOperationTypeExtendEnum extendEnum) {
        return DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE)
                .stream()
                .filter(item -> extendEnum == null || YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getExtendMap().get(extendEnum.getValue())))
                .map(DictDataVo::getDictCode).collect(Collectors.toList());
    }

    /**
     * 获取操作类型编码集合，根据分组枚举
     *
     * @param groupEnum
     * @return
     */
    public static List<String> getOperationTypeList(FeePoolOperationTypeGroupEnum groupEnum) {
        return DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE)
                .stream()
                .filter(item -> groupEnum == null || groupEnum.getValue().equals(item.getExtendMap().get(FeePoolOperationTypeExtendEnum.GROUP.getValue())))
                .map(DictDataVo::getDictCode).collect(Collectors.toList());
    }

    /**
     * 获取操作类型选项集合，根据扩展字段
     *
     * @param filterExtend
     * @return
     */
    public static List<FeeSelectOptionVo> getOperationTypeSelectList(String filterExtend) {
        return DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE)
                .stream()
                .filter(item -> YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getExtendMap().get(filterExtend)))
                .map(item -> {
                    FeeSelectOptionVo option = new FeeSelectOptionVo();
                    option.setValue(item.getDictCode());
                    option.setText(item.getDictValue());
                    return option;
                }).collect(Collectors.toList());
    }

    public static List<FeeSelectOptionVo> getOperationTypeSelectList(FeePoolOperationTypeExtendEnum extendEnum) {
        return DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE)
                .stream()
                .filter(item -> extendEnum == null || YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getExtendMap().get(extendEnum.getValue())))
                .map(item -> {
                    FeeSelectOptionVo option = new FeeSelectOptionVo();
                    option.setValue(item.getDictCode());
                    option.setText(item.getDictValue());
                    return option;
                }).collect(Collectors.toList());
    }

    /**
     * 获取支付方式选项
     *
     * @return
     */
    public static List<FeeSelectOptionVo> getPayTypeSelectList() {
        return DictUtil.list(DictConstant.FEE_POOL_PAY_TYPE)
                .stream()
                .map(item -> {
                    FeeSelectOptionVo option = new FeeSelectOptionVo();
                    option.setValue(item.getDictCode());
                    option.setText(item.getDictValue());
                    return option;
                }).collect(Collectors.toList());
    }

    public static boolean checkOperationTypeUsable(String operationType, FeePoolOperationTypeExtendEnum feePoolOperationTypeExtendEnum) {
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.hasText(operationTypeValue, "无效的操作类型");
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.isTrue(extendMap.containsKey(feePoolOperationTypeExtendEnum.getValue()), "操作类型" + operationTypeValue + "未配置" + feePoolOperationTypeExtendEnum.getDesc());
        return YesNoEnum.yesNoEnum.ONE.getValue().equals(extendMap.get(feePoolOperationTypeExtendEnum.getValue()));
    }

    /**
     * 操作类型与操作类型分组匹配
     *
     * @param operationType 操作类型
     * @param groupEnum     操作类型分组枚举
     * @return
     */
    public static boolean checkOperationTypeGroup(String operationType, FeePoolOperationTypeGroupEnum groupEnum) {
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.hasText(operationTypeValue, "无效的操作类型");
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.isTrue(extendMap.containsKey(FeePoolOperationTypeExtendEnum.GROUP.getValue()), "操作类型" + operationTypeValue + "未配置" + FeePoolOperationTypeExtendEnum.GROUP.getDesc());
        return groupEnum.getValue().equals(extendMap.get(FeePoolOperationTypeExtendEnum.GROUP.getValue()));
    }

    /**
     * 根据费用池类型获取支付方式
     *
     * @param poolType 费用池类型
     * @return
     */
    public static String getPayType(String poolType) {
        String poolTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_TYPE, poolType);
        Assert.hasText(poolTypeValue, "无效的费用池类型");
        Map<String, String> discountExtendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_TYPE, poolType);
        String payType = discountExtendMap.get(DictConstant.FEE_POOL_TYPE_PAY_TYPE_EXTEND_FIELD);
        Assert.hasText(payType, poolTypeValue + "未配置支付方式");
        return payType;
    }

    /**
     * 根据操作类型获取操作类型分组
     *
     * @param operationType 操作类型
     * @return
     */
    public static FeePoolOperationTypeGroupEnum getOperationTypeGroup(String operationType) {
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.hasText(operationTypeValue, "无效的操作类型");
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        String group = extendMap.get(FeePoolOperationTypeExtendEnum.GROUP.getValue());
        Assert.hasText(group, "操作类型" + operationTypeValue + "未配置" + FeePoolOperationTypeExtendEnum.GROUP.getDesc());
        FeePoolOperationTypeGroupEnum anEnum = FeePoolOperationTypeGroupEnum.getEnum(group);
        Assert.notNull(anEnum, "操作类型" + operationTypeValue + "配置分组无效");
        return anEnum;
    }

    /**
     * 根据支付方式反向获取费用池类型
     *
     * @param payType 支付方式
     * @return
     */
    public static FeePoolTypeEnum getPoolTypeByPayType(String payType) {
        String payTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_PAY_TYPE, payType);
        Assert.hasText(payTypeValue, "无效的支付方式");

        List<DictDataVo> list = DictUtil.list(DictConstant.FEE_POOL_TYPE);
        Assert.notEmpty(list, "费用池类型数据字典配置有误");
        List<DictDataVo> collect = list.stream().filter(item -> payType.equals(item.getExtendMap().get(DictConstant.FEE_POOL_TYPE_PAY_TYPE_EXTEND_FIELD))).collect(Collectors.toList());
        Assert.notEmpty(collect, "未找到支付方式" + payTypeValue + "对应的费用池类型");
        Assert.isTrue(collect.size() == 1, "支付方式" + payTypeValue + "关联了多个费用池类型");

        String dictCode = collect.get(0).getDictCode();

        FeePoolTypeEnum poolTypeEnum = FeePoolTypeEnum.getEnum(dictCode);
        Assert.notNull(poolTypeEnum, "费用池类型数据字典有误");

        return poolTypeEnum;
    }

}
