package com.biz.crm.util;

import com.biz.crm.common.param.ParameterParam;
import lombok.extern.slf4j.Slf4j;

import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 树形结构降维编码工具类
 */
@Slf4j
public class TreeRuleCodeUtil {

    /**
     * 数字转换成一层降维编码
     * 如（3->003）
     *
     * @param num
     * @return
     */
    public static String numToSingleCode(Integer num) {
        int length = getRuleCodeSplitLength();
        return String.format("%0" + length + "d", num);
    }

    /**
     * 根据降维编码分离出上级降维编码集合
     *
     * @param ruleCode
     * @return
     */
    public static Set<String> splitParentRuleCodes(String ruleCode) {
        int length = getRuleCodeSplitLength();
        Set<String> set = new LinkedHashSet<>();
        if (StringUtils.isNotEmpty(ruleCode)) {
            set.add(ruleCode);
            while (ruleCode.length() > length) {
                ruleCode = ruleCode.substring(0, ruleCode.length() - length);
                set.add(ruleCode);
            }
        }
        return set;
    }

    /**
     * 根据降维编码分离出上级降维编码集合
     *
     * @param ruleCodeList
     * @return
     */
    public static Set<String> splitParentRuleCodes(List<String> ruleCodeList) {
        Set<String> set = new LinkedHashSet<>();
        for (String ruleCode :
                ruleCodeList) {
            set.addAll(splitParentRuleCodes(ruleCode));
        }
        return set;
    }

    /**
     * 根据降维编码分离出上级降维编码集合(不包含自己)
     *
     * @param ruleCode
     * @return
     */
    public static Set<String> splitParentRuleCodesExcludeSelf(String ruleCode) {
        int length = getRuleCodeSplitLength();
        Set<String> set = new LinkedHashSet<>();
        if (StringUtils.isNotEmpty(ruleCode)) {
            while (ruleCode.length() > length) {
                ruleCode = ruleCode.substring(0, ruleCode.length() - length);
                set.add(ruleCode);
            }
        }
        return set;
    }

    /**
     * 根据降维编码分离出上级降维编码集合(不包含自己，如果本身编码集合已经有包含关系也会返回)
     *
     * @param ruleCodeList
     * @return
     */
    public static Set<String> splitParentRuleCodesExcludeSelf(List<String> ruleCodeList) {
        Set<String> set = new LinkedHashSet<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(ruleCodeList)) {
            for (String ruleCode :
                    ruleCodeList) {
                set.addAll(splitParentRuleCodesExcludeSelf(ruleCode));
            }
        }
        return set;
    }

    /**
     * 根据降维编码分离出上级降维编码集合(返回值不包含任何一个参数里的编码)
     *
     * @param ruleCodeList
     * @return
     */
    public static Set<String> splitParentRuleCodesExcludeAnySelf(List<String> ruleCodeList) {
        Set<String> set = new LinkedHashSet<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(ruleCodeList)) {
            final Set<String> collect = new HashSet<>(ruleCodeList);
            for (String ruleCode :
                    collect) {
                set.addAll(splitParentRuleCodesExcludeSelf(ruleCode));
            }
            return set.stream().filter(item -> !collect.contains(item)).collect(Collectors.toSet());
        }
        return set;
    }

    /**
     * 分离出当前降维最后一层对应的数（即降维编码在当前层级的序号）
     *
     * @param ruleCode
     * @return
     */
    public static int getCurLevelCodeValue(String ruleCode) {
        int length = getRuleCodeSplitLength();
        if (ruleCode != null && !"".equals(ruleCode) && ruleCode.length() >= length) {
            return Integer.valueOf(ruleCode.substring(ruleCode.length() - length));
        }
        return 0;
    }

    /**
     * 获取降维编码单层长度
     *
     * @return
     */
    public static int getRuleCodeSplitLength() {
        int length = 3;
        try {
            String parameterValue = ParamUtil.getParameterValue(ParameterParam.TREE_REDUCTION_RULE_CODE_SPLIT_LENGTH);
            length = Integer.valueOf(parameterValue);
        } catch (Exception e) {
            log.error("获取降维编码长度失败：{}", e);
        }
        return length;
    }

    /**
     * 获取降维编码单层长度
     *
     * @return
     */
    public static int getMaxRuleCodeValueInLevel() {
        int ruleCodeSplitLength = getRuleCodeSplitLength();
        int max = 1;
        for (int i=1;i<ruleCodeSplitLength;i++){
            max *= 10;
        }
        return (max - 1);
    }
}
