package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.mdm.MdmCommonUserFeign;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.service.RedisService;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.util
 * @Description: 登录信息
 * @date 2020/8/26 下午3:04
 */
@Slf4j
public class UserUtils {
    private static UserAutoConfig userAutoConfig = null;
    private static RedisService redisService;
    private static MdmCommonUserFeign mdmCommonUserFeign;

    private static void initUserFeign() {
        if (mdmCommonUserFeign == null) {
            mdmCommonUserFeign = SpringApplicationContextUtil.getApplicationContext().getBean(MdmCommonUserFeign.class);
        }
    }

    private static void initRedis() {
        if (redisService == null) {
            redisService = SpringApplicationContextUtil.getApplicationContext().getBean(RedisService.class);
        }
    }

    /**
     * 获取当前登录人信息
     *
     * @return
     */
    public static UserRedis getUser() {
        UserRedis user = ThreadLocalUtil.getUser();
        if (user != null&&StringUtils.isNotEmpty(user.getUsername())) {
            return user;
        }
        UserRedis userRedis = null;
        try {
            String token = getToken();
            if (StringUtils.isEmpty(token)) {
                log.info("未获取登录用户token信息");
            } else {
                initRedis();
                Object o = redisService.get(RedisParam.TOKEN+token);
                if (o == null) {
                    redisService.setSeconds(RedisParam.TOKEN+token, null, RedisParam.TIME10);
                } else {
                    userRedis = (UserRedis) o;
                    ThreadLocalUtil.setUser(userRedis);
                }
            }
        } catch (Exception e) {
            log.error("获取当前用户:", e);
        }
        return userRedis;
    }

    /**
     * 获取当前token
     *
     * @return
     */
    public static String getToken() {
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        Object tokenLock = ThreadLocalUtil.getObj(GlobalParam.TOKEN);
        String token = tokenLock == null ? "" : tokenLock.toString();
        if (StringUtils.isNotEmpty(token) || request == null) {
            return token;
        }
        if(request!=null){
            token = request.getHeader(GlobalParam.TOKEN);
            if(StringUtils.isEmpty(token)){
                Cookie cookie =  CookiesUtil.getCookieByName(request,GlobalParam.TOKEN);
                if(cookie!=null){
                    token = cookie.getValue();
                }
            }
        }
        return token;
    }

    public static String getToken(String key) {
        initRedis();
        return redisService.getAndSet(RedisParam.TOKEN+key);
    }

    /**
     * 将登陆token添加到本地线程变量
     * @param token
     */
    public static void setToken(String token) {
        ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
        map.put(GlobalParam.TOKEN, token);
        ThreadLocalUtil.stObj(map);
    }
    /**
     * 默认设置当前登录权限，如果没有登录情况
     */
    public static void doTokenForNull() {
        initConfig();
        if (StringUtils.isEmpty(userAutoConfig.getUsername())) {
            return;
        }
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            String token = UUID.randomUUID().toString().replaceAll("-", "");
            map.put(GlobalParam.TOKEN, token);
//            UserRedis userRedis = new UserRedis();
//            userRedis.setUsername(userAutoConfig.getUsername());
            ThreadLocalUtil.stObj(map);
            setUser(token, userAutoConfig.getUsername());
//            userRedis.setId("admin");
//            redisService.setHours(token, userRedis, RedisParam.TIME1);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
    }

    private static void initConfig() {
        if (userAutoConfig == null) {
            userAutoConfig = SpringApplicationContextUtil.getApplicationContext().getBean(UserAutoConfig.class);
        }
    }

    public static void setUser(String token, UserRedis userRedis) {
        Assert.hasText(token, "token不能为空");
        Assert.notNull(userRedis, "userRedis不能为空");
        initRedis();
        String username = userRedis.getUsername();
        String fromtype = userRedis.getFromtype();
        Assert.hasText(username, "用户帐号不能为空");
        Assert.hasText(fromtype, "登录类型不能为空");
        String key = RedisParam.USER_GROUP_PREFIX + username + ":" + fromtype + ":" + token;
        redisService.setDays(key, token, RedisParam.TIME1);
        setDaysUser(token, userRedis);
    }

    /**
     * 强制下线（全平台）
     *
     * @param username 用户登录名
     */
    public static void deleteUser(String username) {
        deleteUser(username, null);
    }

    /**
     * 强制下线，指定来源
     *
     * @param username 用户登录名
     * @param fromtypeList 来源类型编码
     */
    public static void deleteUser(String username, List<String> fromtypeList) {
        if (StringUtils.isEmpty(username)) {
            return;
        }
        Set<String> keys = new HashSet<>(16);
        if (fromtypeList != null && fromtypeList.size() > 0) {
            for (String fromtype :
                    fromtypeList) {
                keys.addAll(redisService.keys(RedisParam.USER_GROUP_PREFIX + username + ":" + fromtype + ":" + "*"));
            }
        } else {
            keys.addAll(redisService.keys(RedisParam.USER_GROUP_PREFIX + username + ":" + "*" + ":" + "*"));
        }
        if (!keys.isEmpty()) {
            Set<String> tokens = keys.stream().map(x -> {
                Object o = redisService.get(x);
                if (o != null) {
                    return o.toString();
                }
                return "";
            }).filter(StringUtils::isNotEmpty).collect(Collectors.toSet());
            deleteToken(tokens);
            redisService.del(keys.toArray(new String[keys.size()]));
        }
    }

    /**
     * 从redis移除token
     *
     * @param token
     */
    protected static void deleteToken(String token) {
        redisService.del(RedisParam.TOKEN + token);
    }

    /**
     * 从redis批量移除token
     *
     * @param tokens
     */
    protected static void deleteToken(Set<String> tokens) {
        String[] strings = tokens.stream().map(x -> RedisParam.TOKEN + x).toArray(String[]::new);
        redisService.del(strings);
    }

    /**
     * 强制用户下线，全平台
     *
     * @param usernameList
     */
    public static void deleteUser(List<String> usernameList) {
        if(CollectionUtils.isEmpty(usernameList)){
            return;
        }
        for (String username : usernameList) {
            deleteUser(username);
        }
    }

    /**
     * 批量用户强制下线，指定登录类型
     *
     * @param usernameList
     * @param fromtypeList
     */
    public static void deleteUser(List<String> usernameList, List<String> fromtypeList) {
        if (CollectionUtils.isEmpty(usernameList)) {
            return;
        }
        for (String username : usernameList) {
            deleteUser(username, fromtypeList);
        }
    }

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userName
     */
    public static void setUser(String token, String userName) {
        initRedis();
        Assert.hasText(token, "token不能为空");
        Assert.hasText(userName, "用户帐号不能为空");
        initUserFeign();
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(userName);
        //预先缓存会话信息，给下面的feign请求使用
        setDaysUser(token, userRedis);
        Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
        MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                .map(Result::getResult)
                .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
        userRedis.setRealname(mdmUserRespVo.getFullName());
        userRedis.setPoscode(mdmUserRespVo.getPositionCode());
        userRedis.setPosname(mdmUserRespVo.getPositionName());
        userRedis.setOrgname(mdmUserRespVo.getOrgName());
        userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
        setDaysUser(token, userRedis);
    }

    protected static void setDaysUser(String token, UserRedis userRedis) {
        redisService.setDays(RedisParam.TOKEN + token, userRedis, RedisParam.TIME1);
    }

    public static void remove() {
        initRedis();
        String username = Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getUsername)
                .orElse(null);
        if(!StringUtils.isEmpty(username)){
            deleteUser(username);
        }
        String token = getToken();
        if (!StringUtils.isEmpty(token)) {
            deleteToken(token);
        }
    }

    public static void remove(String fromtype) {
        initRedis();
        String username = Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getUsername)
                .orElse(null);
        if(!StringUtils.isEmpty(username)){
            deleteUser(username, Collections.singletonList(fromtype));
        }
        String token = getToken();
        if (!StringUtils.isEmpty(token)) {
            deleteToken(token);
        }
    }

    public static void remove(List<String> fromtypeList) {
        if (fromtypeList != null && fromtypeList.size() > 0) {
            fromtypeList.forEach(UserUtils::remove);
        }
    }

    /**
     * 处理多组织问题
     * @param orgCode
     * @return
     */
    public static List<String> handleOrgCodes(String orgCode){
        if(StringUtils.isEmpty(orgCode)){
            return null;
        }
        return Arrays.asList(orgCode.split(","));
    }
}
