package com.biz.crm.util;

import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mdm.productlevel.MdmProductLevelRedisFeign;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelParentChildrenRedisVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelRedisVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelReloadRedisVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelRespVo;
import com.biz.crm.service.RedisService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 产品层级查询工具类
 */
@Slf4j
public class ProductLevelUtil {

    private static RedisService redisService;

    private static MdmProductLevelRedisFeign mdmProductLevelRedisFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmProductLevelRedisFeign(MdmProductLevelRedisFeign productLevelRedisFeign) {
        mdmProductLevelRedisFeign = productLevelRedisFeign;
    }

    /**
     * 根据产品层级编码获取产品层级详情
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static MdmProductLevelRespVo getProductLevelByCode(String productLevelCode) {
        if (productLevelCode != null && !"".equals(productLevelCode)) {
            Map<String, MdmProductLevelRedisVo> productLevelRedisMap = getProductLevelSelfRedisMapFromLocalFirst(Collections.singletonList(productLevelCode));
            if (productLevelRedisMap.containsKey(productLevelCode)) {
                MdmProductLevelRedisVo mdmProductLevelRedisVo = productLevelRedisMap.get(productLevelCode);
                return CrmBeanUtil.copy(mdmProductLevelRedisVo, MdmProductLevelRespVo.class);
            }
        }
        return null;
    }

    /**
     * 根据产品层级编码集合获取产品层级详情
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getProductLevelByCodeList(List<String> productLevelCodeList) {
        List<MdmProductLevelRespVo> list = new ArrayList<>();
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            Map<String, MdmProductLevelRedisVo> productLevelRedisMap = getProductLevelSelfRedisMapFromLocalFirst(productLevelCodeList);
            if (!productLevelRedisMap.isEmpty()) {
                list.addAll(productLevelRedisMap.values().stream().map(item -> CrmBeanUtil.copy(item, MdmProductLevelRespVo.class)).collect(Collectors.toList()));
            }
        }
        return list;
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部上级
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListIncludeSelf(String productLevelCode) {
        if (productLevelCode != null && !"".equals(productLevelCode)) {
            Map<String, MdmProductLevelParentChildrenRedisVo> map = getProductLevelParentRedisMapFromLocalFirst(Collections.singletonList(productLevelCode));
            if (!map.isEmpty()) {
                MdmProductLevelParentChildrenRedisVo redisVo = map.get(productLevelCode);
                if (redisVo != null) {
                    List<MdmProductLevelRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, MdmProductLevelRespVo.class);
                    }
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部上级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListIncludeSelf(List<String> productLevelCodeList) {
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            Map<String, MdmProductLevelParentChildrenRedisVo> map = getProductLevelParentRedisMapFromLocalFirst(productLevelCodeList);
            Set<MdmProductLevelRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (!map.containsKey(productLevelCode)) {
                        continue;
                    }
                    MdmProductLevelParentChildrenRedisVo redisVo = map.get(productLevelCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmProductLevelRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部上级
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListExcludeSelf(String productLevelCode) {
        if (productLevelCode != null && !"".equals(productLevelCode)) {
            List<MdmProductLevelRespVo> parentProductLevelListIncludeSelf = getParentProductLevelListIncludeSelf(productLevelCode);
            if (parentProductLevelListIncludeSelf != null && parentProductLevelListIncludeSelf.size() > 0) {
                return parentProductLevelListIncludeSelf.stream().filter(item -> !productLevelCode.equals(item.getProductLevelCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的产品层级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListExcludeSelf(List<String> productLevelCodeList) {
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            Map<String, MdmProductLevelParentChildrenRedisVo> map = getProductLevelParentRedisMapFromLocalFirst(productLevelCodeList);
            Set<MdmProductLevelRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (!map.containsKey(productLevelCode)) {
                        continue;
                    }
                    MdmProductLevelParentChildrenRedisVo redisVo = map.get(productLevelCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !productLevelCode.equals(item.getProductLevelCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmProductLevelRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中的任何一个）全部上级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListExcludeAnySelf(List<String> productLevelCodeList) {
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            List<MdmProductLevelRespVo> parentProductLevelListExcludeSelf = getParentProductLevelListExcludeSelf(productLevelCodeList);
            if (parentProductLevelListExcludeSelf != null && parentProductLevelListExcludeSelf.size() > 0) {
                final Set<String> productLevelCodeSet = new HashSet<>(productLevelCodeList);
                return parentProductLevelListExcludeSelf.stream().filter(item -> !productLevelCodeSet.contains(item.getProductLevelCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部下级
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListIncludeSelf(String productLevelCode) {
        if (productLevelCode != null && !"".equals(productLevelCode)) {
            Map<String, MdmProductLevelParentChildrenRedisVo> map = getProductLevelChildrenRedisMapFromLocalFirst(Collections.singletonList(productLevelCode));
            if (!map.isEmpty()) {
                MdmProductLevelParentChildrenRedisVo redisVo = map.get(productLevelCode);
                if (redisVo != null) {
                    List<MdmProductLevelRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, MdmProductLevelRespVo.class);
                    }
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部下级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListIncludeSelf(List<String> productLevelCodeList) {
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            Map<String, MdmProductLevelParentChildrenRedisVo> map = getProductLevelChildrenRedisMapFromLocalFirst(productLevelCodeList);
            Set<MdmProductLevelRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (!map.containsKey(productLevelCode)) {
                        continue;
                    }
                    MdmProductLevelParentChildrenRedisVo redisVo = map.get(productLevelCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmProductLevelRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部下级
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListExcludeSelf(String productLevelCode) {
        if (productLevelCode != null && !"".equals(productLevelCode)) {
            List<MdmProductLevelRespVo> parentProductLevelListIncludeSelf = getChildrenProductLevelListIncludeSelf(productLevelCode);
            if (parentProductLevelListIncludeSelf != null && parentProductLevelListIncludeSelf.size() > 0) {
                return parentProductLevelListIncludeSelf.stream().filter(item -> !productLevelCode.equals(item.getProductLevelCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的产品层级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListExcludeSelf(List<String> productLevelCodeList) {
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            Map<String, MdmProductLevelParentChildrenRedisVo> map = getProductLevelChildrenRedisMapFromLocalFirst(productLevelCodeList);
            Set<MdmProductLevelRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (!map.containsKey(productLevelCode)) {
                        continue;
                    }
                    MdmProductLevelParentChildrenRedisVo redisVo = map.get(productLevelCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !productLevelCode.equals(item.getProductLevelCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmProductLevelRespVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中任何一个）全部下级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListExcludeAnySelf(List<String> productLevelCodeList) {
        if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
            List<MdmProductLevelRespVo> parentProductLevelListExcludeSelf = getChildrenProductLevelListExcludeSelf(productLevelCodeList);
            if (parentProductLevelListExcludeSelf != null && parentProductLevelListExcludeSelf.size() > 0) {
                final Set<String> productLevelCodeSet = new HashSet<>(productLevelCodeList);
                return parentProductLevelListExcludeSelf.stream().filter(item -> !productLevelCodeSet.contains(item.getProductLevelCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部上级（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListIncludeSelfEnable(String productLevelCode) {
        return getParentProductLevelListIncludeSelf(productLevelCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部上级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListIncludeSelfEnable(List<String> productLevelCodeList) {
        return getParentProductLevelListIncludeSelf(productLevelCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部上级（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListExcludeSelfEnable(String productLevelCode) {
        return getParentProductLevelListExcludeSelf(productLevelCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的产品层级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListExcludeSelfEnable(List<String> productLevelCodeList) {
        return getParentProductLevelListExcludeSelf(productLevelCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中的任何一个）全部上级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getParentProductLevelListExcludeAnySelfEnable(List<String> productLevelCodeList) {
        return getParentProductLevelListExcludeAnySelf(productLevelCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部下级（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListIncludeSelfEnable(String productLevelCode) {
        return getChildrenProductLevelListIncludeSelf(productLevelCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部下级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListIncludeSelfEnable(List<String> productLevelCodeList) {
        return getChildrenProductLevelListIncludeSelf(productLevelCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部下级（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListExcludeSelfEnable(String productLevelCode) {
        return getChildrenProductLevelListExcludeSelf(productLevelCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的产品层级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListExcludeSelfEnable(List<String> productLevelCodeList) {
        return getChildrenProductLevelListExcludeSelf(productLevelCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中任何一个）全部下级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<MdmProductLevelRespVo> getChildrenProductLevelListExcludeAnySelfEnable(List<String> productLevelCodeList) {
        return getChildrenProductLevelListExcludeAnySelf(productLevelCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部上级产品层级编码
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getParentProductLevelCodeListIncludeSelf(String productLevelCode) {
        return getParentProductLevelListIncludeSelf(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部上级产品层级编码
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getParentProductLevelCodeListIncludeSelf(List<String> productLevelCodeList) {
        return getParentProductLevelListIncludeSelf(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部上级产品层级编码
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getParentProductLevelCodeListExcludeSelf(String productLevelCode) {
        return getParentProductLevelListExcludeSelf(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部上级产品层级编码，如果本身有上下级关系，则会返回处于上级的产品层级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getParentProductLevelCodeListExcludeSelf(List<String> productLevelCodeList) {
        return getParentProductLevelListExcludeSelf(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中的任何一个）全部上级产品层级编码
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getParentProductLevelCodeListExcludeAnySelf(List<String> productLevelCodeList) {
        return getParentProductLevelListExcludeAnySelf(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部下级产品层级编码
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListIncludeSelf(String productLevelCode) {
        return getChildrenProductLevelListIncludeSelf(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部下级产品层级编码
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListIncludeSelf(List<String> productLevelCodeList) {
        return getChildrenProductLevelListIncludeSelf(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部下级产品层级编码
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListExcludeSelf(String productLevelCode) {
        return getChildrenProductLevelListExcludeSelf(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部下级产品层级编码，如果参数本身有上下级关系，则会返回处于下级的产品层级
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListExcludeSelf(List<String> productLevelCodeList) {
        return getChildrenProductLevelListExcludeSelf(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中任何一个）全部下级产品层级编码
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListExcludeAnySelf(List<String> productLevelCodeList) {
        return getChildrenProductLevelListExcludeAnySelf(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部上级产品层级编码（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getParentProductLevelCodeListIncludeSelfEnable(String productLevelCode) {
        return getParentProductLevelListIncludeSelfEnable(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部上级产品层级编码（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getParentProductLevelCodeListIncludeSelfEnable(List<String> productLevelCodeList) {
        return getParentProductLevelListIncludeSelfEnable(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部上级产品层级编码（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getParentProductLevelCodeListExcludeSelfEnable(String productLevelCode) {
        return getParentProductLevelListExcludeSelfEnable(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部上级产品层级编码，如果本身有上下级关系，则会返回处于上级的产品层级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getParentProductLevelCodeListExcludeSelfEnable(List<String> productLevelCodeList) {
        return getParentProductLevelListExcludeSelfEnable(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中的任何一个）全部上级产品层级编码（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getParentProductLevelCodeListExcludeAnySelfEnable(List<String> productLevelCodeList) {
        return getParentProductLevelListExcludeAnySelfEnable(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }


    /**
     * 根据产品层级编码获取产品层级及产品层级全部下级产品层级编码（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListIncludeSelfEnable(String productLevelCode) {
        return getChildrenProductLevelListIncludeSelfEnable(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级及产品层级全部下级产品层级编码（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListIncludeSelfEnable(List<String> productLevelCodeList) {
        return getChildrenProductLevelListIncludeSelfEnable(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码获取产品层级（不含自己）全部下级产品层级编码（只查启用）
     *
     * @param productLevelCode 产品层级编码
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListExcludeSelfEnable(String productLevelCode) {
        return getChildrenProductLevelListExcludeSelfEnable(productLevelCode).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含自己）全部下级产品层级编码，如果参数本身有上下级关系，则会返回处于下级的产品层级（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListExcludeSelfEnable(List<String> productLevelCodeList) {
        return getChildrenProductLevelListExcludeSelfEnable(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }

    /**
     * 根据产品层级编码集合获取产品层级（不含参数中任何一个）全部下级产品层级编码（只查启用）
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    public static List<String> getChildrenProductLevelCodeListExcludeAnySelfEnable(List<String> productLevelCodeList) {
        return getChildrenProductLevelListExcludeAnySelfEnable(productLevelCodeList).stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList());
    }


    /**
     * 清除全部产品层级Redis缓存和本地缓存
     */
    public static void deleteAllCache() {
        redisService.del(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH);
        redisService.del(RedisParam.MDM_PRODUCT_LEVEL_SELF_HASH);
        redisService.del(RedisParam.MDM_PRODUCT_LEVEL_PARENT_HASH);
        redisService.del(RedisParam.MDM_PRODUCT_LEVEL_CHILDREN_HASH);
        ThreadLocalUtil.delProductLevel();
    }

    /**
     * 重新加载redis缓存
     */
    public static void reloadAllCache() {
        deleteAllCache();
        Result<List<MdmProductLevelReloadRedisVo>> result = mdmProductLevelRedisFeign.findRedisReloadList();
        if (result.isSuccess()) {
            List<MdmProductLevelReloadRedisVo> list = result.getResult();
            if (list != null && list.size() > 0) {
                List<List<MdmProductLevelReloadRedisVo>> partition = Lists.partition(list, 500);
                for (List<MdmProductLevelReloadRedisVo> item :
                        partition) {
                    Map<String, MdmProductLevelRedisVo> selfMap = item.stream().collect(Collectors.toMap(MdmProductLevelReloadRedisVo::getProductLevelCode, MdmProductLevelReloadRedisVo::getSelf));
                    Map<String, List<MdmProductLevelRedisVo>> parentMap = item.stream().collect(Collectors.toMap(MdmProductLevelReloadRedisVo::getProductLevelCode, MdmProductLevelReloadRedisVo::getParentList));
                    Map<String, List<MdmProductLevelRedisVo>> childrenMap = item.stream().collect(Collectors.toMap(MdmProductLevelReloadRedisVo::getProductLevelCode, MdmProductLevelReloadRedisVo::getChildrenList));
                    redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_SELF_HASH, new HashMap<Object, Object>(selfMap));
                    redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_PARENT_HASH, new HashMap<Object, Object>(parentMap));
                    redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_CHILDREN_HASH, new HashMap<Object, Object>(childrenMap));
                }
            }
        }
    }

    /*--------------------------------------------------以下是私有方法--------------------------------------------------*/

    /**
     * 获取产品层级缓存数据，优先从本地缓存取，再从redis取
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    private static Map<String, MdmProductLevelRedisVo> getProductLevelSelfRedisMapFromLocalFirst(List<String> productLevelCodeList) {
        Map<String, MdmProductLevelRedisVo> map = new HashMap<>(16);
        try {
            if (productLevelCodeList != null && productLevelCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (ThreadLocalUtil.productLevelSelfContains(productLevelCode)) {
                        MdmProductLevelRedisVo fromProductLevelSelf = ThreadLocalUtil.getFromProductLevelSelf(productLevelCode);
                        map.put(productLevelCode, fromProductLevelSelf);
                    } else {
                        getFromRedisCodeSet.add(productLevelCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmProductLevelRedisVo> mapFromRedisFirst = getProductLevelSelfRedisMapFromRedisFirst(new ArrayList<>(productLevelCodeList));

                    for (String productLevelCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(productLevelCode)) {
                            map.put(productLevelCode, mapFromRedisFirst.get(productLevelCode));
                        }
                        ThreadLocalUtil.addToProductLevelSelf(productLevelCode, mapFromRedisFirst.get(productLevelCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("产品层级查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取产品层级缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    private static Map<String, MdmProductLevelRedisVo> getProductLevelSelfRedisMapFromRedisFirst(List<String> productLevelCodeList) {
        Map<String, MdmProductLevelRedisVo> map = new HashMap<>(16);
        try {
            if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
                List<MdmProductLevelRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH, new HashSet<Object>(productLevelCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = productLevelCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> productLevelListFromRedis = redisService.hmget(RedisParam.MDM_PRODUCT_LEVEL_SELF_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (productLevelListFromRedis != null && productLevelListFromRedis.size() > 0) {
                        List<MdmProductLevelRedisVo> collect = productLevelListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmProductLevelRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmProductLevelRedisVo::getProductLevelCode).collect(Collectors.toSet()));
                        }
                    }
                }

                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmProductLevelRedisVo>> result = mdmProductLevelRedisFeign.findRedisListByProductLevelCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmProductLevelRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmProductLevelRedisVo> collect = list.stream().collect(Collectors.toMap(MdmProductLevelRedisVo::getProductLevelCode, v -> v));

                            Map<String, MdmProductLevelRedisVo> cacheMap = new HashMap<>(16);
                            for (String productLevelCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(productLevelCode)) {
                                    lockCodeMap.put(productLevelCode, productLevelCode);
                                } else {
                                    redisVoList.add(collect.get(productLevelCode));
                                    cacheMap.put(productLevelCode, collect.get(productLevelCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_SELF_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmProductLevelRedisVo::getProductLevelCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("产品层级查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取产品层级上级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    private static Map<String, MdmProductLevelParentChildrenRedisVo> getProductLevelParentRedisMapFromLocalFirst(List<String> productLevelCodeList) {
        Map<String, MdmProductLevelParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (productLevelCodeList != null && productLevelCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (ThreadLocalUtil.productLevelParentContains(productLevelCode)) {
                        MdmProductLevelParentChildrenRedisVo fromProductLevelParent = ThreadLocalUtil.getFromProductLevelParent(productLevelCode);
                        map.put(productLevelCode, fromProductLevelParent);
                    } else {
                        getFromRedisCodeSet.add(productLevelCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmProductLevelParentChildrenRedisVo> mapFromRedisFirst = getProductLevelParentRedisMapFromRedisFirst(new ArrayList<>(productLevelCodeList));

                    for (String productLevelCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(productLevelCode)) {
                            map.put(productLevelCode, mapFromRedisFirst.get(productLevelCode));
                        }
                        ThreadLocalUtil.addToProductLevelParent(productLevelCode, mapFromRedisFirst.get(productLevelCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("产品层级查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取产品层级上级缓存数据，优先从redis缓存存取，再从数据库取
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    private static Map<String, MdmProductLevelParentChildrenRedisVo> getProductLevelParentRedisMapFromRedisFirst(List<String> productLevelCodeList) {
        Map<String, MdmProductLevelParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
                List<MdmProductLevelParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH, new HashSet<Object>(productLevelCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = productLevelCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> productLevelListFromRedis = redisService.hmget(RedisParam.MDM_PRODUCT_LEVEL_PARENT_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (productLevelListFromRedis != null && productLevelListFromRedis.size() > 0) {
                        List<MdmProductLevelParentChildrenRedisVo> collect = productLevelListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmProductLevelParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmProductLevelParentChildrenRedisVo::getProductLevelCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmProductLevelParentChildrenRedisVo>> result = mdmProductLevelRedisFeign.findRedisProductLevelParentListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmProductLevelParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmProductLevelParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmProductLevelParentChildrenRedisVo::getProductLevelCode, v -> v));

                            Map<String, MdmProductLevelParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String productLevelCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(productLevelCode)) {
                                    lockCodeMap.put(productLevelCode, productLevelCode);
                                } else {
                                    redisVoList.add(collect.get(productLevelCode));
                                    cacheMap.put(productLevelCode, collect.get(productLevelCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_PARENT_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmProductLevelParentChildrenRedisVo::getProductLevelCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("产品层级查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取产品层级下级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    private static Map<String, MdmProductLevelParentChildrenRedisVo> getProductLevelChildrenRedisMapFromLocalFirst(List<String> productLevelCodeList) {
        Map<String, MdmProductLevelParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (productLevelCodeList != null && productLevelCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String productLevelCode :
                        productLevelCodeList) {
                    if (ThreadLocalUtil.productLevelChildrenContains(productLevelCode)) {
                        MdmProductLevelParentChildrenRedisVo fromProductLevelChildren = ThreadLocalUtil.getFromProductLevelChildren(productLevelCode);
                        map.put(productLevelCode, fromProductLevelChildren);
                    } else {
                        getFromRedisCodeSet.add(productLevelCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmProductLevelParentChildrenRedisVo> mapFromRedisFirst = getProductLevelChildrenRedisMapFromRedisFirst(new ArrayList<>(productLevelCodeList));

                    for (String productLevelCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(productLevelCode)) {
                            map.put(productLevelCode, mapFromRedisFirst.get(productLevelCode));
                        }
                        ThreadLocalUtil.addToProductLevelChildren(productLevelCode, mapFromRedisFirst.get(productLevelCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("产品层级查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取产品层级下级缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param productLevelCodeList 产品层级编码集合
     * @return
     */
    private static Map<String, MdmProductLevelParentChildrenRedisVo> getProductLevelChildrenRedisMapFromRedisFirst(List<String> productLevelCodeList) {
        Map<String, MdmProductLevelParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (productLevelCodeList != null && productLevelCodeList.size() > 0) {
                List<MdmProductLevelParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH, new HashSet<Object>(productLevelCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = productLevelCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> productLevelListFromRedis = redisService.hmget(RedisParam.MDM_PRODUCT_LEVEL_CHILDREN_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (productLevelListFromRedis != null && productLevelListFromRedis.size() > 0) {
                        List<MdmProductLevelParentChildrenRedisVo> collect = productLevelListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmProductLevelParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmProductLevelParentChildrenRedisVo::getProductLevelCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmProductLevelParentChildrenRedisVo>> result = mdmProductLevelRedisFeign.findRedisProductLevelChildrenListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmProductLevelParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmProductLevelParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmProductLevelParentChildrenRedisVo::getProductLevelCode, v -> v));

                            Map<String, MdmProductLevelParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String productLevelCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(productLevelCode)) {
                                    lockCodeMap.put(productLevelCode, productLevelCode);
                                } else {
                                    redisVoList.add(collect.get(productLevelCode));
                                    cacheMap.put(productLevelCode, collect.get(productLevelCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_CHILDREN_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_PRODUCT_LEVEL_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmProductLevelParentChildrenRedisVo::getProductLevelCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("产品层级查询工具类异常：{}", e);
        }
        return map;
    }

}
