package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.ApprovalType;
import com.biz.crm.eunm.mdm.MdmApprovalConfig;
import com.biz.crm.eunm.mdm.MdmBpmStatusEnum;
import com.biz.crm.mdm.approval.MdmApprovalConfigFeign;
import com.biz.crm.nebular.mdm.approval.resp.MdmApprovalConfigConditionRedisVo;
import com.biz.crm.nebular.mdm.approval.resp.MdmApprovalConfigRedisVo;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
public class ApprovalConfigUtil {

    private static RedisService redisService;

    private static MdmApprovalConfigFeign mdmApprovalConfigFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmApprovalConfigFeign(MdmApprovalConfigFeign approvalConfigFeign) {
        mdmApprovalConfigFeign = approvalConfigFeign;
    }

    /**
     * 获取审批配置
     *
     * @param menuCode     菜单编码
     * @param businessCode 业务编码
     * @return
     */
    public static MdmApprovalConfigRedisVo getConfig(String menuCode, String businessCode) {
        return getConfig(menuCode, businessCode, false);
    }

    /**
     * 获取审批配置
     *
     * @param menuCode     菜单编码
     * @param businessCode 业务编码
     * @param notNull      是否允许结果为空，如果不允许，则没有查询到结果会抛异常
     * @return
     */
    public static MdmApprovalConfigRedisVo getConfig(String menuCode, String businessCode, boolean notNull) {
        Assert.hasText(menuCode, "缺失菜单编码");
        Assert.hasText(businessCode, "缺失业务编码");
        if (!StringUtils.isEmpty(menuCode) && !StringUtils.isEmpty(businessCode)) {
            MdmApprovalConfigRedisVo redisVo = get(menuCode, businessCode);
            if (notNull) {
                Assert.notNull(redisVo, "未找到配置");
            }
            return redisVo;
        }
        return null;
    }

    /**
     * 校验审批配置并且返回需要审批的方式
     *
     * @param menuCode     菜单编码
     * @param businessCode 业务编码
     * @param paramObj     参数对象，字段从对象第一层中反射获取
     * @return
     */
    public static ApprovalType checkAndGet(String menuCode, String businessCode, Object paramObj) {
        MdmApprovalConfigRedisVo redisVo = get(menuCode, businessCode);
        if (redisVo == null) {
            return ApprovalType.NULL;
        }
        if (YesNoEnum.yesNoEnum.ZERO.getValue().equals(redisVo.getControlFlag())) {
            return ApprovalType.NULL;
        }
        Assert.hasText(redisVo.getApprovalType(), "审批配置【" + redisVo.getBusinessName() + "】不完整，缺失审批类型");
        Assert.hasText(redisVo.getControlScope(), "审批配置【" + redisVo.getBusinessName() + "】不完整，缺失控制范围");
        if (MdmApprovalConfig.ControlScope.ALL.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.getEnum(redisVo.getApprovalType());
        }
        Assert.hasText(redisVo.getConditionLogicalType(), "审批配置【" + redisVo.getBusinessName() + "】不完整，缺失条件运算方式");
        List<MdmApprovalConfigConditionRedisVo> conditionList = redisVo.getConditionList();
        Assert.notEmpty(conditionList, "审批配置【" + redisVo.getBusinessName() + "】不完整，缺失审批条件");
        Map<String, Boolean> conditionResultMap = new HashMap<>(16);
        for (MdmApprovalConfigConditionRedisVo condition :
                conditionList) {
            String fieldCode = condition.getFieldCode();
            if (!ReflectUtil.checkFieldExist(paramObj, fieldCode)) {
                throw new BusinessException("参数未找到字段" + condition.getFieldName() + "[" + condition.getFieldCode() + "]");
            }
            Object filedValueObj = ReflectUtil.getFiledValueByName(paramObj, fieldCode);
            boolean b = checkCondition(filedValueObj == null ? "" : filedValueObj.toString(), condition);
            if (MdmApprovalConfig.ConditionLogicalType.OR.getValue().equals(redisVo.getConditionLogicalType())) {
                if (b) {
                    return ApprovalType.getEnum(redisVo.getApprovalType());
                }
            }
            if (MdmApprovalConfig.ConditionLogicalType.AND.getValue().equals(redisVo.getConditionLogicalType())) {
                if (!b) {
                    return ApprovalType.NULL;
                }
            }
            conditionResultMap.put(condition.getFieldCode(), b);
        }
        if (MdmApprovalConfig.ConditionLogicalType.OR.getValue().equals(redisVo.getConditionLogicalType())) {
            List<Boolean> collect = conditionResultMap.values().stream().filter(x -> x).collect(Collectors.toList());
            if (!collect.isEmpty()) {
                return ApprovalType.getEnum(redisVo.getApprovalType());
            }
        } else if (MdmApprovalConfig.ConditionLogicalType.AND.getValue().equals(redisVo.getConditionLogicalType())) {
            List<Boolean> collect = conditionResultMap.values().stream().filter(x -> !x).collect(Collectors.toList());
            if (collect.isEmpty()) {
                return ApprovalType.getEnum(redisVo.getApprovalType());
            } else {
                return ApprovalType.NULL;
            }
        } else {
            //自定义
        }
        return ApprovalType.NULL;
    }

    /**
     * 校验数据是否可以编辑
     *
     * @param approvalType   审批类型
     * @param approvalStatus 当前审批状态
     * @return
     */
    public static boolean editableCheck(String approvalType, String approvalStatus) {
        if (approvalType != null && !"".equals(approvalType) && approvalStatus != null && !"".equals(approvalStatus)) {
            if (ApprovalType.BPM.getValue().equals(approvalType) || ApprovalType.STATUS.getValue().equals(approvalType)) {
                if (MdmBpmStatusEnum.PASS.getValue().equals(approvalStatus) || MdmBpmStatusEnum.APPROVAL.getValue().equals(approvalStatus)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 根据菜单编码清除缓存
     *
     * @param menuCode 菜单编码
     */
    public static void deleteCache(String menuCode) {
        if (StringUtils.isEmpty(menuCode)) {
            return;
        }
        deleteCache(Collections.singletonList(menuCode));
    }

    /**
     * 根据菜单编码清除缓存
     *
     * @param menuCodeList 菜单编码集合
     */
    public static void deleteCache(List<String> menuCodeList) {
        if (menuCodeList == null || menuCodeList.size() == 0) {
            return;
        }
        redisService.del(menuCodeList.stream().map(x -> connectHashKey(RedisParam.MDM_APPROVAL_CONFIG, x)).toArray(String[]::new));
        redisService.del(menuCodeList.stream().map(x -> connectHashKey(RedisParam.MDM_APPROVAL_CONFIG_LOCK, x)).toArray(String[]::new));
    }

    /**
     * 清除缓存
     *
     * @param menuCode     菜单编码
     * @param businessCode 业务编码
     */
    public static void deleteCache(String menuCode, String businessCode) {
        redisService.hdel(connectHashKey(RedisParam.MDM_APPROVAL_CONFIG, menuCode), businessCode);
        redisService.hdel(connectHashKey(RedisParam.MDM_APPROVAL_CONFIG_LOCK, menuCode), businessCode);
    }

    /*-------------------------------------------------以下为私有方法-------------------------------------------------*/

    private static boolean checkCondition(String fieldValue, MdmApprovalConfigConditionRedisVo condition) {
        if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(condition.getControlFlag())) {
            return true;
        }
        String fieldKeywords = condition.getFieldKeywords();
        if (StringUtils.isEmpty(fieldKeywords)) {
            throw new BusinessException("条件" + condition.getFieldName() + "缺失关键值");
        }
        String[] split = fieldKeywords.split(",");
        for (String keywords :
                split) {
            if (MdmApprovalConfig.FieldCompareType.EQ.getValue().equals(condition.getCompareType())) {
                if (fieldValue.equals(keywords)) {
                    return true;
                }
            }
        }
        return false;
    }

    private static MdmApprovalConfigRedisVo get(String menuCode, String businessCode) {
        MdmApprovalConfigRedisVo redisVo = null;
        if (!checkLock(menuCode, businessCode)) {
            String hashKey = connectHashKey(RedisParam.MDM_APPROVAL_CONFIG, menuCode);
            Object o = redisService.hget(hashKey, businessCode);
            if (o == null) {
                Result<MdmApprovalConfigRedisVo> result = mdmApprovalConfigFeign.queryForRedis(menuCode, businessCode);
                Assert.isTrue(result.isSuccess(), "获取审批配置失败");
                if (result.getResult() == null) {
                    lock(menuCode, businessCode);
                } else {
                    redisVo = result.getResult();
                    redisService.hset(hashKey, businessCode, redisVo, RedisParam.SECONDS_OF_MONTH);
                }
            } else {
                redisVo = (MdmApprovalConfigRedisVo) o;
            }
        }
        return redisVo;
    }

    private static void lock(String menuCode, String businessCode) {
        redisService.hset(connectHashKey(RedisParam.MDM_APPROVAL_CONFIG, menuCode), businessCode, businessCode, RedisParam.SECONDS_OF_MONTH);
    }

    private static boolean checkLock(String menuCode, String businessCode) {
        return redisService.hget(connectHashKey(RedisParam.MDM_APPROVAL_CONFIG_LOCK, menuCode), businessCode) != null;
    }

    private static String connectHashKey(String prefix, String menuCode) {
        return prefix + menuCode;
    }

}
