package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.mdm.MdmCommonUserFeign;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.service.RedisService;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.util
 * @Description: 登录信息
 * @date 2020/8/26 下午3:04
 */
@Slf4j
public class UserUtils {
    private static UserAutoConfig userAutoConfig = null;
    private static RedisService redisService;
    private static MdmCommonUserFeign mdmCommonUserFeign;

    private static void initUserFeign() {
        if (mdmCommonUserFeign == null) {
            mdmCommonUserFeign = SpringApplicationContextUtil.getApplicationContext().getBean(MdmCommonUserFeign.class);
        }
    }

    private static void initRedis() {
        if (redisService == null) {
            redisService = SpringApplicationContextUtil.getApplicationContext().getBean(RedisService.class);
        }
    }

    private static void initConfig() {
        if (userAutoConfig == null) {
            userAutoConfig = SpringApplicationContextUtil.getApplicationContext().getBean(UserAutoConfig.class);
        }
    }

    /**
     * 获取当前登录人信息
     *
     * @return
     */
    public static UserRedis getUser() {
        UserRedis user = ThreadLocalUtil.getUser();
        if (user != null && StringUtils.isNotEmpty(user.getUsername())) {
            return user;
        }
        UserRedis userRedis = null;
        try {
            String token = getToken();
            if (StringUtils.isEmpty(token)) {
                log.info("未获取登录用户token信息");
            } else {
                initRedis();
                Object o = redisService.get(RedisParam.TOKEN + token);
                if (o == null) {
                    setUserSeconds(token, null, RedisParam.TIME10);
                } else {
                    userRedis = (UserRedis) o;
                    ThreadLocalUtil.setUser(userRedis);
                }
            }
        } catch (Exception e) {
            log.error("获取当前用户:", e);
        }
        return userRedis;
    }

    /**
     * 获取当前token
     *
     * @return
     */
    public static String getToken() {
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        Object tokenLock = ThreadLocalUtil.getObj(GlobalParam.TOKEN);
        String token = tokenLock == null ? "" : tokenLock.toString();
        if (StringUtils.isNotEmpty(token) || request == null) {
            return token;
        }
        if (request != null) {
            token = request.getHeader(GlobalParam.TOKEN);
            if (StringUtils.isEmpty(token)) {
                Cookie cookie = CookiesUtil.getCookieByName(request, GlobalParam.TOKEN);
                if (cookie != null) {
                    token = cookie.getValue();
                }
            }
        }
        return token;
    }

    public static String getToken(String key) {
        initRedis();
        return redisService.getAndSet(RedisParam.TOKEN + key);
    }

    /**
     * 将登录token添加到本地线程变量
     *
     * @param token
     */
    public static void setToken(String token) {
        ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
        map.put(GlobalParam.TOKEN, token);
        ThreadLocalUtil.stObj(map);
    }

    /**
     * 默认设置当前登录权限，如果没有登录情况
     */
    public static void doTokenForNull() {
        initConfig();
        if (StringUtils.isEmpty(userAutoConfig.getUsername())) {
            return;
        }
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            String token = UUID.randomUUID().toString().replaceAll("-", "");
            map.put(GlobalParam.TOKEN, token);
//            UserRedis userRedis = new UserRedis();
//            userRedis.setUsername(userAutoConfig.getUsername());
            ThreadLocalUtil.stObj(map);
            createUserToRedis(token, userAutoConfig.getUsername(), LoginFromTypeEnum.TEMPORARY.getValue(), RedisParam.SECONDS_OF_HOUR);
//            userRedis.setId("admin");
//            redisService.setHours(token, userRedis, RedisParam.TIME1);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
    }

    /**
     * 默认设置当前登录权限，如果没有登录情况
     */
    public static void doTokenForUserName(String userName, long seconds) {
        String token = UUID.randomUUID().toString().replaceAll("-", "");
        createUserToRedis(token, userName, LoginFromTypeEnum.TEMPORARY.getValue(), seconds);
    }

    /**
     * 缓存用户信息
     *
     * @param token
     * @param userRedis
     */
    public static void setUser(String token, UserRedis userRedis) {
        Assert.hasText(token, "token不能为空");
        Assert.notNull(userRedis, "userRedis不能为空");
        initRedis();
        String username = userRedis.getUsername();
        String fromtype = userRedis.getFromtype();
        Assert.hasText(username, "用户帐号不能为空");
        Assert.hasText(fromtype, "登录类型不能为空");
        setUserGroupSeconds(token, username, fromtype, RedisParam.SECONDS_OF_DAY);
        setDaysUser(token, userRedis);
    }

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userName
     */
    @Deprecated
    public static void setUser(String token, String userName) {
        initRedis();
        Assert.hasText(token, "token不能为空");
        Assert.hasText(userName, "用户帐号不能为空");
        initUserFeign();
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(userName);
        //预先缓存会话信息，给下面的feign请求使用
        setDaysUser(token, userRedis);
        Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
        MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                .map(Result::getResult)
                .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
        userRedis.setRealname(mdmUserRespVo.getFullName());
        userRedis.setPoscode(mdmUserRespVo.getPositionCode());
        userRedis.setPosname(mdmUserRespVo.getPositionName());
        userRedis.setOrgname(mdmUserRespVo.getOrgName());
        userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
        userRedis.setFromtype(LoginFromTypeEnum.TEMPORARY.getValue());
        setDaysUser(token, userRedis);
    }

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userName
     */
    public static void setUser(String token, String userName, String fromtype) {
        initRedis();
        Assert.hasText(token, "token不能为空");
        Assert.hasText(userName, "用户帐号不能为空");
        initUserFeign();
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(userName);
        //预先缓存会话信息，给下面的feign请求使用
        setDaysUser(token, userRedis);
        Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
        MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                .map(Result::getResult)
                .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
        userRedis.setRealname(mdmUserRespVo.getFullName());
        userRedis.setPoscode(mdmUserRespVo.getPositionCode());
        userRedis.setPosname(mdmUserRespVo.getPositionName());
        userRedis.setOrgname(mdmUserRespVo.getOrgName());
        userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
        userRedis.setFromtype(StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue());
        setDaysUser(token, userRedis);
    }

    /**
     * 强制用户生成登录信息
     *
     * @param userName
     * @param fromtype
     * @param seconds
     */
    public static UserRedis createUserToRedis(String token,String userName, String fromtype, long seconds) {
        initConfig();
        UserRedis userRedis = null;
        Assert.hasText(userName, "userName不能为空");
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            map.put(GlobalParam.TOKEN, token);
            ThreadLocalUtil.stObj(map);

            initRedis();
            initUserFeign();
            userRedis = new UserRedis();
            userRedis.setUsername(userName);
            //预先缓存会话信息，给下面的feign请求使用
            setUserSeconds(token, userRedis, seconds);
            Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
            MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                    .map(Result::getResult)
                    .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
            userRedis.setRealname(mdmUserRespVo.getFullName());
            userRedis.setPoscode(mdmUserRespVo.getPositionCode());
            userRedis.setPosname(mdmUserRespVo.getPositionName());
            userRedis.setOrgname(mdmUserRespVo.getOrgName());
            userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
            userRedis.setFromtype(StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue());
            setUserSeconds(token, userRedis, seconds);
            setUserGroupSeconds(token, userName, fromtype, seconds);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
        return userRedis;
    }

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userRedis
     * @param fromtype
     * @param seconds
     */
    public static void setUserToRedis(String token, UserRedis userRedis, String fromtype, long seconds) {
        Assert.hasText(token, "token不能为空");
        Assert.notNull(userRedis, "userRedis不能为空");
        fromtype = StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue();
        initRedis();
        String username = userRedis.getUsername();
        Assert.hasText(username, "用户帐号不能为空");
        userRedis.setFromtype(fromtype);
        setUserGroupSeconds(token, username, fromtype, seconds);
        setUserSeconds(token, userRedis, seconds);
    }

    /**
     * 添加用户缓存信息到redis
     *
     * @param token
     * @param userRedis
     */
    protected static void setDaysUser(String token, UserRedis userRedis) {
        setUserSeconds(token, userRedis, RedisParam.SECONDS_OF_DAY);
    }

    /**
     * 添加用户缓存信息到redis
     *
     * @param token     token
     * @param userRedis 缓存信息
     * @param seconds   时间（秒）
     */
    private static void setUserSeconds(String token, UserRedis userRedis, long seconds) {
        redisService.setSeconds(RedisParam.TOKEN + token, userRedis, seconds);
    }

    /**
     * 缓存token到用户记录登录来源分组
     *
     * @param token    token
     * @param userName 用户名
     * @param fromtype 登录来源
     * @param seconds  时间（秒）
     */
    private static void setUserGroupSeconds(String token, String userName, String fromtype, long seconds) {
        redisService.hset(RedisParam.USER_GROUP_PREFIX + userName + ":" + fromtype, token, token, seconds);
    }

    /**
     * 从redis移除token
     *
     * @param token
     */
    protected static void deleteToken(String token) {
        if (StringUtils.isNotEmpty(token)) {
            deleteToken(Collections.singleton(token));
        }
    }

    /**
     * 从redis批量移除token
     *
     * @param tokens
     */
    protected static void deleteToken(Set<String> tokens) {
        if (!tokens.isEmpty()) {
            redisService.del(tokens.stream().map(x -> RedisParam.TOKEN + x).toArray(String[]::new));
            //如果需要同步删除token与用户和登录来源的缓存，请注释上一行，放开下面的代码
//            tokens.forEach(x -> {
//                Object o = redisService.get(RedisParam.TOKEN + x);
//                if (o != null) {
//                    redisService.del(RedisParam.TOKEN + x);
//                    UserRedis userRedis = (UserRedis) o;
//                    if (StringUtils.isNotEmpty(userRedis.getUsername()) && StringUtils.isNotEmpty(userRedis.getFromtype())) {
//                        redisService.hdel(RedisParam.USER_GROUP_PREFIX + userRedis.getUsername() + ":" + userRedis.getFromtype(), x);
//                    }
//                }
//            });
        }
    }

    /**
     * 指定用户全平台退出登录
     *
     * @param username 用户登录名
     */
    public static void deleteUser(String username) {
        deleteUser(Collections.singletonList(username));
    }

    /**
     * 指定用户指定平台退出登录
     *
     * @param username     用户登录名
     * @param fromtypeList 来源类型编码
     */
    public static void deleteUser(String username, List<String> fromtypeList) {
        if (StringUtils.isEmpty(username)) {
            return;
        }
        if (CollectionUtil.listEmpty(fromtypeList)) {
            return;
        }
        deleteUser(Collections.singletonList(username), fromtypeList);
    }

    /**
     * 指定用户全平台退出登录
     *
     * @param usernameList
     */
    public static void deleteUser(List<String> usernameList) {
        if (CollectionUtils.isEmpty(usernameList)) {
            return;
        }
        List<String> fromTypeList = new ArrayList<>();
        for (LoginFromTypeEnum loginFromTypeEnum :
                LoginFromTypeEnum.values()) {
            fromTypeList.add(loginFromTypeEnum.getValue());
        }
        deleteUser(usernameList, fromTypeList);
    }

    /**
     * 指定用户指定平台退出登录
     *
     * @param usernameList
     * @param fromtypeList
     */
    public static void deleteUser(List<String> usernameList, List<String> fromtypeList) {
        if (CollectionUtil.listEmpty(usernameList)) {
            return;
        }
        if (CollectionUtil.listEmpty(fromtypeList)) {
            return;
        }
        Set<String> tokenSet = new HashSet<>(16);
        Set<String> userGroupHashKeySet = new HashSet<>(16);
        for (String username :
                usernameList) {
            for (String fromType :
                    fromtypeList) {
                String hashKey = RedisParam.USER_GROUP_PREFIX + username + ":" + fromType;
                Map<?, ?> hmget = redisService.hmget(hashKey);
                if (hmget != null) {
                    Map<String, String> tokenMap = (Map<String, String>) hmget;
                    tokenSet.addAll(tokenMap.keySet());
                    userGroupHashKeySet.add(hashKey);
                }
            }
        }
        if (!tokenSet.isEmpty()) {
            redisService.del(tokenSet.stream().map(x -> RedisParam.TOKEN + x).toArray(String[]::new));
            redisService.del(userGroupHashKeySet.toArray(new String[userGroupHashKeySet.size()]));
        }
    }

    /**
     * 当前登录用户全平台退出
     */
    public static void remove() {
        initRedis();
        String username = Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getUsername)
                .orElse(null);
        if (!StringUtils.isEmpty(username)) {
            deleteUser(username);
        }
    }

    /**
     * 退出当前会话
     */
    public static void logout() {
        String token = getToken();
        deleteToken(token);
    }

    /**
     * 当前登录人退出指定平台
     */
    public static void logout(String fromtype) {
        initRedis();
        String username = Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getUsername)
                .orElse(null);
        deleteUser(username, Collections.singletonList(fromtype));
    }

    /**
     * 处理多组织问题
     * @param orgCode
     * @return
     */
    public static List<String> handleOrgCodes(String orgCode){
        if(StringUtils.isEmpty(orgCode)){
            return null;
        }
        return Arrays.asList(orgCode.split(","));
    }
}
