package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.ApprovalType;
import com.biz.crm.eunm.mdm.MdmApprovalConfig;
import com.biz.crm.eunm.mdm.MdmBpmStatusEnum;
import com.biz.crm.mdm.approval.MdmApprovalConfigFeign;
import com.biz.crm.mdm.role.MdmRoleFeign;
import com.biz.crm.nebular.mdm.approval.resp.MdmApprovalConfigConditionRedisVo;
import com.biz.crm.nebular.mdm.approval.resp.MdmApprovalConfigRedisVo;
import com.biz.crm.nebular.mdm.approval.resp.MdmApprovalConfigSubmitterRedisVo;
import com.biz.crm.nebular.mdm.approval.resp.MdmApprovalConfigVo;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.*;

@Slf4j
@Component
public class ApprovalConfigUtil {

    private static RedisService redisService;

    private static MdmApprovalConfigFeign mdmApprovalConfigFeign;

    private static MdmRoleFeign mdmRoleFeign;

    @Autowired
    public void setRedisService(RedisService redis) {
        ApprovalConfigUtil.redisService = redis;
    }

    @Autowired
    public void setMdmApprovalConfigFeign(MdmApprovalConfigFeign approvalConfigFeign) {
        ApprovalConfigUtil.mdmApprovalConfigFeign = approvalConfigFeign;
    }

    @Autowired
    public void setMdmRoleFeign(MdmRoleFeign mdmRoleFeign) {
        ApprovalConfigUtil.mdmRoleFeign = mdmRoleFeign;
    }

    /**
     * 获取审批配置
     *
     * @param businessCode 业务编码
     * @return
     */
    public static MdmApprovalConfigVo getConfig(String businessCode) {
        return getConfig(businessCode, false);
    }

    /**
     * 获取审批配置
     *
     * @param businessCode 业务编码
     * @param notNull      是否允许结果为空，如果不允许，则没有查询到结果会抛异常
     * @return
     */
    public static MdmApprovalConfigVo getConfig(String businessCode, boolean notNull) {
        Assert.hasText(businessCode, "缺失业务编码");
        if (!StringUtils.isEmpty(businessCode)) {
            MdmApprovalConfigRedisVo redisVo = get(businessCode);
            if (notNull) {
                Assert.notNull(redisVo, "未找到审批配置【" + businessCode + "】");
            }
            return CrmBeanUtil.copy(redisVo, MdmApprovalConfigVo.class);
        }
        return null;
    }

    /**
     * 校验审批配置并且返回需要审批的方式
     *
     * @param businessCode 业务编码
     * @param paramObj     参数对象，字段从对象第一层中反射获取
     * @return
     */
    public static ApprovalType checkAndGet(String businessCode, Object paramObj) {

        MdmApprovalConfigRedisVo redisVo = get(businessCode);
        if (redisVo == null || YesNoEnum.yesNoEnum.ZERO.getValue().equals(redisVo.getControlFlag())) {
            return ApprovalType.NULL;
        }
        String businessName = redisVo.getBusinessName();
        Assert.hasText(redisVo.getApprovalType(), "审批配置【" + businessName + "】不完整，缺失审批类型");
        Assert.hasText(redisVo.getControlScope(), "审批配置【" + businessName + "】不完整，缺失控制范围");
        if (MdmApprovalConfig.ControlScope.ALL_IN.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.getEnum(redisVo.getApprovalType());
        }
        if (MdmApprovalConfig.ControlScope.ALL_OUT.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.NULL;
        }
        Assert.hasText(redisVo.getGroupCheckWay(), "审批配置【" + businessName + "】不完整，缺失条件组判断逻辑");
        MdmApprovalConfig.GroupCheckWay groupCheckWay = MdmApprovalConfig.GroupCheckWay.getEnum(redisVo.getGroupCheckWay());
        Assert.notNull(groupCheckWay, "审批配置【" + businessName + "】不完整，条件判断逻辑配置错误");

        Set<Boolean> conditionCheckSet = new HashSet<>(16);
        Set<Boolean> submitterCheckSet = new HashSet<>(16);
        if (MdmApprovalConfig.GroupCheckWay.ALL_CONDITION != groupCheckWay && MdmApprovalConfig.GroupCheckWay.ONE_CONDITION != groupCheckWay) {
            if (CollectionUtil.listNotEmptyNotSizeZero(redisVo.getSubmitterList())) {
                UserRedis user = UserUtils.getUser();
                if (user == null || StringUtils.isEmpty(user.getUsername())) {
                    throw new BusinessException("请先登录");
                }
                for (MdmApprovalConfigSubmitterRedisVo submitter :
                        redisVo.getSubmitterList()) {
                    if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(submitter.getControlFlag())) {
                        continue;
                    }
                    boolean b = checkSubmitter(user, submitter);
                    submitterCheckSet.add(b);
                    if (b) {
                        if (MdmApprovalConfig.GroupCheckWay.ONE_SUBMITTER == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ONE_CONDITION_AND_ONE_SUBMITTER == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ONE_CONDITION_OR_ONE_SUBMITTER == groupCheckWay) {
                            break;
                        }
                    } else {
                        if (MdmApprovalConfig.GroupCheckWay.ALL_SUBMITTER == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ALL_CONDITION_AND_ALL_SUBMITTER == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ALL_CONDITION_OR_ALL_SUBMITTER == groupCheckWay) {
                            break;
                        }
                    }
                }
            }
        }
        if (MdmApprovalConfig.GroupCheckWay.ALL_SUBMITTER != groupCheckWay && MdmApprovalConfig.GroupCheckWay.ONE_SUBMITTER != groupCheckWay) {
            if (CollectionUtil.listNotEmptyNotSizeZero(redisVo.getConditionList())) {
                for (MdmApprovalConfigConditionRedisVo condition :
                        redisVo.getConditionList()) {
                    if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(condition.getControlFlag())) {
                        continue;
                    }
                    String fieldCode = condition.getFieldCode();
                    if (!ReflectUtil.checkFieldExist(paramObj, fieldCode)) {
                        throw new BusinessException("参数未找到字段" + condition.getFieldName() + "[" + condition.getFieldCode() + "]");
                    }
                    Object filedValueObj = ReflectUtil.getFiledValueByName(paramObj, fieldCode);
                    boolean b = checkCondition(filedValueObj == null ? "" : filedValueObj.toString(), condition);
                    conditionCheckSet.add(b);
                    if (b) {
                        if (MdmApprovalConfig.GroupCheckWay.ONE_CONDITION == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ONE_CONDITION_AND_ONE_SUBMITTER == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ONE_CONDITION_OR_ONE_SUBMITTER == groupCheckWay) {
                            break;
                        }
                    } else {
                        if (MdmApprovalConfig.GroupCheckWay.ALL_CONDITION == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ALL_CONDITION_AND_ALL_SUBMITTER == groupCheckWay || MdmApprovalConfig.GroupCheckWay.ALL_CONDITION_OR_ALL_SUBMITTER == groupCheckWay) {
                            break;
                        }
                    }
                }
            }
        }

        boolean check = true;
        if (MdmApprovalConfig.GroupCheckWay.ALL_CONDITION == groupCheckWay) {
            check = !conditionCheckSet.contains(false);
        } else if (MdmApprovalConfig.GroupCheckWay.ONE_CONDITION == groupCheckWay) {
            check = conditionCheckSet.contains(true);
        } else if (MdmApprovalConfig.GroupCheckWay.ALL_SUBMITTER == groupCheckWay) {
            check = !submitterCheckSet.contains(false);
        } else if (MdmApprovalConfig.GroupCheckWay.ONE_SUBMITTER == groupCheckWay) {
            check = submitterCheckSet.contains(true);
        } else if (MdmApprovalConfig.GroupCheckWay.ALL_CONDITION_AND_ALL_SUBMITTER == groupCheckWay) {
            check = !conditionCheckSet.contains(false) && !submitterCheckSet.contains(false);
        } else if (MdmApprovalConfig.GroupCheckWay.ALL_CONDITION_OR_ALL_SUBMITTER == groupCheckWay) {
            check = !conditionCheckSet.contains(false) || !submitterCheckSet.contains(false);
        } else if (MdmApprovalConfig.GroupCheckWay.ONE_CONDITION_AND_ONE_SUBMITTER == groupCheckWay) {
            check = conditionCheckSet.contains(true) && submitterCheckSet.contains(true);
        } else if (MdmApprovalConfig.GroupCheckWay.ONE_CONDITION_OR_ONE_SUBMITTER == groupCheckWay) {
            check = conditionCheckSet.contains(true) || submitterCheckSet.contains(true);
        } else {

        }

        if (check && MdmApprovalConfig.ControlScope.PART_IN_CONDITION.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.getEnum(redisVo.getApprovalType());
        } else if (!check && MdmApprovalConfig.ControlScope.PART_IN_CONDITION.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.NULL;
        } else if (check && MdmApprovalConfig.ControlScope.PART_OUT_CONDITION.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.NULL;
        } else if (!check && MdmApprovalConfig.ControlScope.PART_OUT_CONDITION.getValue().equals(redisVo.getControlScope())) {
            return ApprovalType.getEnum(redisVo.getApprovalType());
        } else {

        }
        return ApprovalType.NULL;
    }

    /**
     * 校验数据是否可以编辑
     *
     * @param approvalType   审批类型
     * @param approvalStatus 当前审批状态
     * @return
     */
    public static boolean editableCheck(String approvalType, String approvalStatus) {
        if (approvalType != null && !"".equals(approvalType) && approvalStatus != null && !"".equals(approvalStatus)) {
            if (ApprovalType.BPM.getValue().equals(approvalType) || ApprovalType.STATUS.getValue().equals(approvalType)) {
                if (MdmBpmStatusEnum.PASS.getValue().equals(approvalStatus) || MdmBpmStatusEnum.APPROVAL.getValue().equals(approvalStatus)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 清除缓存
     *
     * @param businessCode 业务编码
     */
    public static void deleteCache(String businessCode) {
        if (StringUtils.isNotEmpty(businessCode)) {
            redisService.del(RedisParam.MDM_APPROVAL_CONFIG + businessCode);
            redisService.del(RedisParam.MDM_APPROVAL_CONFIG_LOCK + businessCode);
        }
    }

    /**
     * 清除缓存
     *
     * @param businessCodeList 业务编码集合
     */
    public static void deleteCache(List<String> businessCodeList) {
        if (businessCodeList != null && businessCodeList.size() > 0) {
            redisService.del(businessCodeList.stream().map(x -> RedisParam.MDM_APPROVAL_CONFIG + x).toArray(String[]::new));
            redisService.del(businessCodeList.stream().map(x -> RedisParam.MDM_APPROVAL_CONFIG_LOCK + x).toArray(String[]::new));
        }
    }

    /*-------------------------------------------------以下为私有方法-------------------------------------------------*/

    private static boolean checkCondition(String fieldValue, MdmApprovalConfigConditionRedisVo condition) {
        if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(condition.getControlFlag())) {
            return true;
        }
        String fieldKeywords = condition.getFieldKeywords();
        if (StringUtils.isEmpty(fieldKeywords)) {
            throw new BusinessException("条件" + condition.getFieldName() + "缺失关键值");
        }
        String[] split = fieldKeywords.split(",");
        for (String keywords :
                split) {
            if (MdmApprovalConfig.FieldCompareType.EQ.getValue().equals(condition.getCompareType())) {
                if (fieldValue.equals(keywords)) {
                    return true;
                }
            }
        }
        return false;
    }

    private static boolean checkSubmitter(UserRedis user, MdmApprovalConfigSubmitterRedisVo submitter) {

        Assert.hasText(submitter.getSubmitterType(), "缺失提交人匹配对象");
        Set<String> keyWordSet = new HashSet<>(Arrays.asList(submitter.getFieldKeywords().split(",")));
        if (MdmApprovalConfig.SubmitterType.ORG.getValue().equals(submitter.getSubmitterType())) {
            if (StringUtils.isNotEmpty(user.getOrgcode())) {
                List<String> strings = new ArrayList<>();
                strings.addAll(Arrays.asList(user.getOrgcode().split(",")));
                if (MdmApprovalConfig.SubmitterCompareType.EQ.getValue().equals(submitter.getCompareType())) {
                    strings.retainAll(keyWordSet);
                    if (strings.size() > 0) {
                        return true;
                    }
                } else if (MdmApprovalConfig.SubmitterCompareType.NE.getValue().equals(submitter.getCompareType())) {
                    strings.retainAll(keyWordSet);
                    if (strings.size() == 0) {
                        return true;
                    }
                } else if (MdmApprovalConfig.SubmitterCompareType.IN_PARENT.getValue().equals(submitter.getCompareType()) || MdmApprovalConfig.SubmitterCompareType.NOT_IN_PARENT.getValue().equals(submitter.getCompareType())) {
                    List<String> parentOrgCodeSet = OrgUtil.getParentOrgCodeListIncludeSelfEnable(new ArrayList<>(keyWordSet));
                    strings.retainAll(parentOrgCodeSet);
                    if (MdmApprovalConfig.SubmitterCompareType.IN_PARENT.getValue().equals(submitter.getCompareType()) && strings.size() > 0) {
                        return true;
                    }
                    if (MdmApprovalConfig.SubmitterCompareType.NOT_IN_PARENT.getValue().equals(submitter.getCompareType()) && strings.size() == 0) {
                        return true;
                    }
                } else if (MdmApprovalConfig.SubmitterCompareType.IN_CHILDREN.getValue().equals(submitter.getCompareType()) || MdmApprovalConfig.SubmitterCompareType.NOT_IN_CHILDREN.getValue().equals(submitter.getCompareType())) {
                    List<String> childrenOrgCodeSet = OrgUtil.getChildrenOrgCodeListIncludeSelfEnable(new ArrayList<>(keyWordSet));
                    strings.retainAll(childrenOrgCodeSet);
                    if (MdmApprovalConfig.SubmitterCompareType.IN_CHILDREN.getValue().equals(submitter.getCompareType()) && strings.size() > 0) {
                        return true;
                    }
                    if (MdmApprovalConfig.SubmitterCompareType.NOT_IN_CHILDREN.getValue().equals(submitter.getCompareType()) && strings.size() == 0) {
                        return true;
                    }
                } else {
                    throw new BusinessException("无效的匹配方式");
                }
            }
        } else if (MdmApprovalConfig.SubmitterType.POSITION.getValue().equals(submitter.getSubmitterType())) {
            if (StringUtils.isNotEmpty(user.getPoscode())) {
                String positionCode = user.getPoscode();
                if (MdmApprovalConfig.SubmitterCompareType.EQ.getValue().equals(submitter.getCompareType())) {
                    if (keyWordSet.contains(positionCode)) {
                        return true;
                    }
                } else if (MdmApprovalConfig.SubmitterCompareType.NE.getValue().equals(submitter.getCompareType())) {
                    if (!keyWordSet.contains(positionCode)) {
                        return true;
                    }
                } else if (MdmApprovalConfig.SubmitterCompareType.IN_PARENT.getValue().equals(submitter.getCompareType()) || MdmApprovalConfig.SubmitterCompareType.NOT_IN_PARENT.getValue().equals(submitter.getCompareType())) {
                    Set<String> parentPositionCodeSet = new HashSet<>(PositionUtil.getParentPositionCodeListIncludeSelfEnable(new ArrayList<>(keyWordSet)));
                    if (MdmApprovalConfig.SubmitterCompareType.IN_PARENT.getValue().equals(submitter.getCompareType()) && parentPositionCodeSet.contains(positionCode)) {
                        return true;
                    }
                    if (MdmApprovalConfig.SubmitterCompareType.NOT_IN_PARENT.getValue().equals(submitter.getCompareType()) && !parentPositionCodeSet.contains(positionCode)) {
                        return true;
                    }
                } else if (MdmApprovalConfig.SubmitterCompareType.IN_CHILDREN.getValue().equals(submitter.getCompareType()) || MdmApprovalConfig.SubmitterCompareType.NOT_IN_CHILDREN.getValue().equals(submitter.getCompareType())) {
                    Set<String> childrenPositionCodeSet = new HashSet<>(PositionUtil.getChildrenPositionCodeListIncludeSelfEnable(new ArrayList<>(keyWordSet)));
                    if (MdmApprovalConfig.SubmitterCompareType.IN_CHILDREN.getValue().equals(submitter.getCompareType()) && childrenPositionCodeSet.contains(positionCode)) {
                        return true;
                    }
                    if (MdmApprovalConfig.SubmitterCompareType.NOT_IN_CHILDREN.getValue().equals(submitter.getCompareType()) && !childrenPositionCodeSet.contains(positionCode)) {
                        return true;
                    }
                } else {
                    throw new BusinessException("无效的匹配方式");
                }
            }

        } else if (MdmApprovalConfig.SubmitterType.ROLE.getValue().equals(submitter.getSubmitterType())) {

            Set<String> roleCodeSet = new HashSet<>(16);
            if (UserTypeEnum.USER.getCode().equals(user.getUsertype())) {
                Result<List<String>> roleByPositionCode = mdmRoleFeign.getRoleByPositionCode(user.getPoscode());
                if (!roleByPositionCode.isSuccess()) {
                    throw new BusinessException(roleByPositionCode.getMessage());
                }
                roleCodeSet.addAll(roleByPositionCode.getResult());
            } else {
                Result<List<String>> roleByUserName = mdmRoleFeign.getRoleByUserName(user.getUsername());
                if (!roleByUserName.isSuccess()) {
                    throw new BusinessException(roleByUserName.getMessage());
                }
                roleCodeSet.addAll(roleByUserName.getResult());
            }

            if (MdmApprovalConfig.SubmitterCompareType.EQ.getValue().equals(submitter.getCompareType())) {
                roleCodeSet.retainAll(keyWordSet);
                if (roleCodeSet.size() > 0) {
                    return true;
                }
            } else if (MdmApprovalConfig.SubmitterCompareType.NE.getValue().equals(submitter.getCompareType())) {
                roleCodeSet.retainAll(keyWordSet);
                if (roleCodeSet.size() == 0) {
                    return true;
                }
            } else {

            }
        } else {

        }
        return false;
    }

    private static MdmApprovalConfigRedisVo get(String businessCode) {
        MdmApprovalConfigRedisVo redisVo = null;
        if (!checkLock(businessCode)) {
            String key = RedisParam.MDM_APPROVAL_CONFIG + businessCode;
            Object o = redisService.get(key);
            if (o == null) {
                Result<MdmApprovalConfigRedisVo> result = mdmApprovalConfigFeign.queryForRedis(businessCode);
                Assert.isTrue(result.isSuccess(), "获取审批配置失败");
                if (result.getResult() == null) {
                    lock(businessCode);
                } else {
                    redisVo = result.getResult();
                    redisService.setSeconds(key, redisVo, RedisParam.SECONDS_OF_MONTH);
                }
            } else {
                redisVo = (MdmApprovalConfigRedisVo) o;
            }
        }
        return redisVo;
    }

    private static void lock(String businessCode) {
        redisService.set(RedisParam.MDM_APPROVAL_CONFIG_LOCK + businessCode, RedisParam.SECONDS_OF_MONTH);
    }

    private static boolean checkLock(String businessCode) {
        Object o = redisService.get(RedisParam.MDM_APPROVAL_CONFIG_LOCK + businessCode);
        if (o != null) {
            return true;
        }
        return false;
    }

}
