package com.biz.crm.util;

import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.httpclient.ConnectTimeoutException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.NoHttpResponseException;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

import javax.annotation.PostConstruct;
import javax.net.ssl.SSLException;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

/**
 * 请求
 *
 * @Author wei·yang
 * @Date Created in 2020-04-23 15:55
 */
@Slf4j
public class HttpClientUtils {
    /**
     * 时间设置
     */
    private static final RequestConfig REQUEST_CONFIG_TIME_OUT = RequestConfig.custom()
            .setSocketTimeout(20000)
            .setConnectTimeout(50000)
            .setConnectionRequestTimeout(50000)
            .build();

    private static PoolingHttpClientConnectionManager cm = null;

    @PostConstruct
    public void init() {
        cm = new PoolingHttpClientConnectionManager();
        cm.setMaxTotal(1000);
        cm.setDefaultMaxPerRoute(1000);
    }

    /**
     * @param isPooled 是否使用连接池
     * @return
     */
    public static CloseableHttpClient getClient(boolean isPooled) {
        HttpRequestRetryHandler retryHandler = new HttpRequestRetryHandler() {
            @Override
            public boolean retryRequest(IOException e, int retryTimes, HttpContext httpContext) {
                if (retryTimes > 2) {
                    return false;
                }
                if (e instanceof UnknownHostException || e instanceof ConnectTimeoutException
                        || !(e instanceof SSLException) || e instanceof NoHttpResponseException) {
                    return true;
                }

                HttpClientContext clientContext = HttpClientContext.adapt(httpContext);
                HttpRequest request = clientContext.getRequest();
                boolean idempotent = !(request instanceof HttpEntityEnclosingRequest);
                if (idempotent) {
                    // 如果请求被认为是幂等的，那么就重试。即重复执行不影响程序其他效果的
                    return true;
                }
                return false;
            }
        };
        if (isPooled) {
            return HttpClients.custom().setConnectionManager(cm).setRetryHandler(retryHandler).build();
        }
        return HttpClients.createDefault();

    }

    public static final Result doPostWithRequest(String path, HttpServletRequest request) {
        Result<String> result = new Result<>();
        Enumeration params = request.getParameterNames();
        List<NameValuePair> nameValuePairs = Lists.newArrayList();
        while (params.hasMoreElements()) {
            String paramName = (String) params.nextElement();
            nameValuePairs.add(new BasicNameValuePair(paramName, request.getParameter(paramName)));
        }
        HttpPost httpPost = new HttpPost(path);

        httpPost.setConfig(REQUEST_CONFIG_TIME_OUT);
        try {
            httpPost.setEntity(new UrlEncodedFormEntity(nameValuePairs));
            return execReq(httpPost);
        } catch (UnsupportedEncodingException e) {
            log.error("do post error: ", e);
            result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
            result.setMessage(e.getMessage());
        }
        return result;
    }

    /**
     * get请求
     * @param path
     * @param hander
     * @return
     */
    public static final Result<String> doGet(String path, Map<String, String> hander) {
        Result<String> result = new Result<>();
        log.info("doGet from " + path);
        HttpGet httpGet = new HttpGet(path);
        httpGet.setConfig(REQUEST_CONFIG_TIME_OUT);
        try {
            //设置header
            httpGet.setHeader("Content-Type", "application/json; charset=utf-8");
            hander.forEach((k,v) -> {
                httpGet.setHeader(k,v);
            });
            result = execReqGet(httpGet);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
            result.setMessage(e.getMessage());
        }
        return result;
    }

    private static Result<String> execReqGet(HttpGet httpGet) {
        Result<String> result = new Result<>();
        try {
            CloseableHttpResponse response = getClient(true).execute(httpGet);
            if (response != null) {
                try {
                    if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                        result.setCode(ComConstant.SC_OK_200);
                        result.setResult(EntityUtils.toString(response.getEntity()));
                        return result;
                    } else {
                        result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
                        result.setMessage("http get request error: " + httpGet.getURI());
                        return result;
                    }
                } finally {
                    EntityUtils.consume(response.getEntity());
                }
            } else {
                result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
                result.setMessage("http get request error: " + httpGet.getURI());
                return result;
            }
        } catch (IOException e) {
            log.error("doGet error: ", e);
            result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
            result.setMessage(e.getMessage());
            return result;
        }
    }

    /**
     * post请求
     * @param path
     * @param params
     * @return
     */
    public static final Result<String> doPost(String path, Map<String, Object> params) {
        Result<String> result = new Result<>();
        log.debug("doPost from " + path, params);
        HttpPost httpPost = new HttpPost(path);
        httpPost.setConfig(REQUEST_CONFIG_TIME_OUT);
        try {
            httpPost.setEntity(new UrlEncodedFormEntity(createParams(params)));
            result = execReq(httpPost);
        } catch (UnsupportedEncodingException e) {
            log.error(e.getMessage(), e);
            result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
            result.setMessage(e.getMessage());
        }
        return result;
    }

    /**
     * webservice请求
     * @param path
     * @param xml
     * @return
     */
    public static final Result<String> doPostWebService(String path, String xml) {
        log.info("请求SAP交互URL:{},XML:{}",path, xml);
        HttpPost httpPost = new HttpPost(path);

        httpPost.setConfig(REQUEST_CONFIG_TIME_OUT);
        StringEntity postingString = new StringEntity(xml, StandardCharsets.UTF_8.name());
        httpPost.setEntity(postingString);
        httpPost.setHeader("Content-Type", "text/xml; charset=utf-8");
        Result<String> bodyAsString = execReq(httpPost);
        log.info("请求SAP交互结果Post Result:{}",bodyAsString);
        return bodyAsString;
    }

    /**
     * webservice请求带验证
     * @param path
     * @param userName
     * @param passWord
     * @param xml
     * @return
     */
    public static final Result<String> doPostWebService(String path, String userName, String passWord, String xml) {
        log.info("请求SAP交互URL:{} ｜ XML:{}",path, xml);
        HttpPost httpPost = new HttpPost(path);
        // 手动构建验证信息
        String auth = userName + ":" + passWord;
        byte[] encodedAuth = Base64.encodeBase64(
                auth.getBytes(StandardCharsets.UTF_8));
        String authHeader = "Basic " + new String(encodedAuth);
        // 将验证信息放入到 Header
        httpPost.setHeader(HttpHeaders.AUTHORIZATION, authHeader);
        httpPost.setConfig(REQUEST_CONFIG_TIME_OUT);
        StringEntity postingString = new StringEntity(xml, StandardCharsets.UTF_8.name());
        httpPost.setEntity(postingString);
        httpPost.setHeader("Content-Type", "text/xml; charset=utf-8");
        Result<String> bodyAsString = execReq(httpPost);
        log.info("请求SAP交互结果Post Result:{}",bodyAsString);
        return bodyAsString;
    }

    private static List<NameValuePair> createParams(Map<String, Object> params) {
        List<NameValuePair> nameValuePairs = Lists.newArrayList();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            nameValuePairs.add(new BasicNameValuePair(entry.getKey(), String.valueOf(entry.getValue())));
        }
        return nameValuePairs;
    }

    private static Result<String> execReq(HttpPost httpPost) {
        Result<String> result = new Result<>();
        try {
            CloseableHttpResponse response = getClient(true).execute(httpPost);
            if (response != null) {
                try {
                    if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                        result.setCode(ComConstant.SC_OK_200);
                        result.setResult(EntityUtils.toString(response.getEntity()));
                        return result;
                    } else {
                        result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
                        result.setMessage("http post request error: " + httpPost.getURI());
                        return result;
                    }
                } finally {
                    EntityUtils.consume(response.getEntity());
                }
            } else {
                result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
                result.setMessage("http post request error: " + httpPost.getURI());
                return result;
            }

        } catch (IOException e) {
            log.error("doPost error: ", e);
            result.setCode(ComConstant.SC_INTERNAL_SERVER_ERROR_500);
            result.setMessage(e.getMessage());
            return result;
        }
    }
}
