package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.service.RedisService;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.StringJoiner;


/**
 * 定时任务功能扩展工具
 *  @author: luoqi
 *  @Date: 2021-4-22 13:05
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
@Component
public class JobHelper {
    private static final long JOB_RUNNING_TIME = 3600 * 24 * 2;
    /** 任务执行期间，业务数据分页大小 */
    private static final int JOB_BIZ_DATA_PAGE_SIZE = 500;
    public static final String redisHash = new StringJoiner(RedisParam.DELIMITER).add(RedisParam.SFA_JOB).toString();
    public int getJobBizDataPageSize(){
        return JOB_BIZ_DATA_PAGE_SIZE;
    }

//    private ConcurrentHashMap<String, String> executingMap = new ConcurrentHashMap<>();
    @Resource
    private RedisService redisService;



    /**
     * 任务执行(保存上下文)
     */
    public void executeJobSaveContext(JobContext context){
        this.saveContext(context);
        this.doJobExecute(context);
    }

    protected void saveContext(JobContext context){
        String key = context.getJob().name();
        Job job = context.getJob();
        synchronized (this){
            if(!redisService.putIfAbsent(redisHash, key, context, JOB_RUNNING_TIME)){
                String msg = "任务[" + job.name() + "]正在执行中，请稍后重试！";
                log.warn(msg);
                throw new BusinessException(msg);
            }
        }
    }
    /**
     * 异步执行(保存上下文)
     * @param context
     */
    @Async
    public void executeJobAsyncSaveContext(JobContext context){
        this.executeJobSaveContext(context);
    }
    /**
     * 异步执行
     * @param context
     */
    @Async
    public void executeJobAsync(JobContext context){
        this.doJobExecute(context);
    }
    private void doJobExecute(JobContext context){
        String key = context.getJob().name();
        Job job = context.getJob();
        try {

            UserUtils.doTokenForUserName("admin", JOB_RUNNING_TIME);
            JobExecutor jobExecutor = SpringApplicationContextUtil.getApplicationContext().getBean(context.getExecutorBeanClass());
            jobExecutor.executeJob(context.getJobParam());
        }catch (BeansException e){
            log.error("任务[" + context.getJob().name() + "]启动失败", e);
            throw new BusinessException("任务[" + job.name() + "]启动失败，请重试！", e);
        }catch (Exception e){
            log.error("任务[" + context.getJob().name() + "]启动失败", e);
            throw new BusinessException("任务[" + job.name() + "]启动失败，请重试！", e);
        }finally {
            if(redisService.hmHasKey(redisHash, key)){
                redisService.hmDelete(redisHash, key);
            }
        }
    }






    /**
     * 任务上下文环境
     *  @author: luoqi
     *  @Date: 2021-4-22 12:04
     *  @version: V1.0
     *  @Description:
     */
    @Data
    public static class JobContext{
        //任务执行逻辑
        private Class<? extends JobExecutor> executorBeanClass;

        //任务枚举
        private Job job;

        private Object jobParam;

    }

    /**
     * 任务执行逻辑
     *  @author: luoqi
     *  @Date: 2021-4-22 11:51
     *  @version: V1.0
     *  @Description:
     */
    public interface JobExecutor{
        void executeJob(Object jobParam);
    }


    /**
     * 任务常量
     *  @author: luoqi
     *  @Date: 2021-4-22 11:58
     *  @version: V1.0
     *  @Description:
     */
    public enum Job{
        /**
         * 拜访计划结算任务
         */
        PLAN_INFO_SETTLEMENT_JOB,
        RESOLVE_SFA_VISIT_STEP_ACTIVITY_EXECUTION_JOB,

        /**
         * 拜访日进度报表任务
         */
        VISIT_DAY_RATE_JOB,

        /**
         * 月度积分结算任务
         */
        INTEGRAL_MONTH_JOB
    }
}
