package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.parameter.feign.ParameterFeign;
import com.biz.crm.parameter.vo.ParameterVo;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 全局参数工具类
 *
 * @author Tao.Chen
 * @date 2020-11-28 20:04:56
 */
@Slf4j
@Component
public class ParamUtil {

    private static ParameterFeign parameterFeign;
    private static RedisService redisService;

    @Autowired
    public void setParameterFeign(ParameterFeign feign) {
        ParamUtil.parameterFeign = feign;
    }

    @Autowired
    public void setRedisService(RedisService redis) {
        ParamUtil.redisService = redis;
    }

    /**
     * 根据参数编码获取当前值
     *
     * @param parameterCode
     * @return
     */
    public static String getParameterValue(String parameterCode) {
        if (parameterCode == null || "".equals(parameterCode)) {
            throw new BusinessException("请传入全局参数编码");
        }
        ParameterVo parameterVo = getParameterVo(parameterCode);
        if (parameterVo == null) {
            throw new BusinessException("未配置全局参数：" + parameterCode);
        }
        return parameterVo.getParameterValue();
    }

    /**
     * 根据参数编码获取当前值（不会抛异常，但可能返回null）
     *
     * @param parameterCode
     * @return
     */
    public static String getParameterValueNoException(String parameterCode) {
        try {
            if (parameterCode != null && !"".equals(parameterCode)) {
                ParameterVo parameterVo = getParameterVo(parameterCode);
                if (parameterVo != null) {
                    return parameterVo.getParameterValue();
                }
            }
        } catch (Exception e) {
            log.error("获取全局参数异常：{}", parameterCode);
        }
        return null;
    }

    /**
     * 根据参数编码集合获取当前值Map
     *
     * @param parameterCodeList
     * @return
     */
    public static Map<String, String> getParameterValueMap(List<String> parameterCodeList) {
        Map<String, String> map = new HashMap<>(16);
        if (parameterCodeList != null && parameterCodeList.size() > 0) {
            Map<String, ParameterVo> parameterVoMap = basicMap(parameterCodeList);
            if (!parameterVoMap.isEmpty()) {
                for (Map.Entry<String, ParameterVo> entry :
                        parameterVoMap.entrySet()) {
                    map.put(entry.getKey(), entry.getValue().getParameterValue());
                }
            }
        }
        return map;
    }

    /**
     * 根据参数编码获取对象
     *
     * @param parameterCode
     * @return
     */
    @Deprecated
    public static ParameterVo getParameterVo(String parameterCode) {
        if (parameterCode != null && !"".equals(parameterCode)) {
            Map<String, ParameterVo> parameterVoMap = basicMap(Collections.singletonList(parameterCode));
            if (!parameterVoMap.isEmpty()) {
                return parameterVoMap.get(parameterCode);
            }
        }
        return null;
    }

    /**
     * 根据参数编码集合获取对象map
     *
     * @param parameterCodeList
     * @return
     */
    @Deprecated
    public static Map<String, ParameterVo> getParameterVoMap(List<String> parameterCodeList) {
        if (parameterCodeList != null && parameterCodeList.size() > 0) {
            return basicMap(parameterCodeList);
        }
        return new HashMap<>();
    }

    /**
     * 私有、统一逻辑、批量获取vo
     *
     * @param parameterCodeList
     * @return
     */
    private static Map<String, ParameterVo> basicMap(List<String> parameterCodeList) {
        Map<String, ParameterVo> map = new HashMap<>(16);
        try {
            if (parameterCodeList != null && parameterCodeList.size() > 0) {
                Set<String> unlockCodeSet = new HashSet<>(16);
                for (String parameterCode :
                        parameterCodeList) {
                    ParameterVo param = ThreadLocalUtil.getGlobalParam(parameterCode);
                    if (param != null) {
                        map.put(parameterCode, param);
                    } else if (!checkLock(parameterCode)) {
                        unlockCodeSet.add(parameterCode);
                    } else {

                    }
                }

                Set<String> notCacheCodeSet = new HashSet<>(16);
                if (!unlockCodeSet.isEmpty()) {
                    for (String parameterCode :
                            unlockCodeSet) {
                        Object o = redisService.get(RedisParam.PARAMETER + parameterCode);
                        if (o == null) {
                            notCacheCodeSet.add(parameterCode);
                        } else {
                            ParameterVo parameterVo = (ParameterVo) o;
                            map.put(parameterCode, parameterVo);
                            ThreadLocalUtil.setGlobalParam(parameterCode, parameterVo);
                        }
                    }
                }

                if (!notCacheCodeSet.isEmpty()) {
                    Result<List<ParameterVo>> result = parameterFeign.getParameterRedisList(new ArrayList<>(notCacheCodeSet));
                    if (result.isSuccess()) {
                        List<ParameterVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, ParameterVo> collect = list.stream().collect(Collectors.toMap(ParameterVo::getParameterCode, v -> v));
                            for (String parameterCode :
                                    unlockCodeSet) {
                                if (!collect.containsKey(parameterCode)) {
                                    lock(parameterCode);
                                } else {
                                    ParameterVo parameterVo = collect.get(parameterCode);
                                    addRedisCache(parameterVo);
                                    map.put(parameterCode, parameterVo);
                                    ThreadLocalUtil.setGlobalParam(parameterCode, parameterVo);
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            log.error("全局参数工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 删除缓存
     *
     * @param parameterCode
     */
    public static void deleteRedisCache(String parameterCode) {
        if (parameterCode != null && !"".equals(parameterCode)) {
            redisService.del(RedisParam.PARAMETER + parameterCode);
            redisService.del(RedisParam.PARAMETER_LOCK + parameterCode);
        }
    }

    /**
     * 添加缓存
     *
     * @param parameterVo
     */
    private static void addRedisCache(ParameterVo parameterVo) {
        if (parameterVo != null) {
            redisService.set(RedisParam.PARAMETER + parameterVo.getParameterCode(), parameterVo);
        }
    }

    /**
     * 加锁
     *
     * @param parameterCode
     */
    private static void lock(String parameterCode) {
        if (parameterCode != null && !"".equals(parameterCode)) {
            redisService.setSeconds(RedisParam.PARAMETER_LOCK + parameterCode, parameterCode, RedisParam.TIME5);
        }
    }

    /**
     * 检查是否被锁
     *
     * @param parameterCode
     * @return
     */
    private static boolean checkLock(String parameterCode) {
        if (parameterCode != null && !"".equals(parameterCode)) {
            return redisService.get(RedisParam.PARAMETER_LOCK + parameterCode) != null;
        }
        return false;
    }
}
