package com.biz.crm.util.fee;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.eunm.fee.FeePoolOperationTypeGroupEnum;
import com.biz.crm.nebular.fee.pool.resp.FeePoolPageConfigVo;
import com.biz.crm.nebular.fee.pool.resp.FeeSelectOptionVo;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.ParamUtil;
import com.biz.crm.util.StringUtils;
import org.springframework.util.Assert;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 费用池配置查询校验工具类
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
public class FeePoolConfigUtil {

    static {
        FEE_POOL_AMOUNT_DESC = getFeePoolAmountDesc();
    }

    public static final String FEE_POOL_AMOUNT_DESC;


    /**
     * 根据配置编码获取配置信息
     *
     * @param configCode 配置编码
     * @return
     */
    public static FeePoolPageConfigVo getConfig(String configCode) {
        FeePoolPageConfigVo configVo = new FeePoolPageConfigVo();
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_PAGE_LIST_CONFIG, configCode);
        if (extendMap == null || extendMap.isEmpty()) {
            throw new BusinessException("未找到配置【" + configCode + "】");
        }

        String poolGroup = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_POOL_GROUP);
        if (StringUtils.isNotEmpty(poolGroup)) {
            final Map<String, String> feePoolGroupMap = DictUtil.dictMap(DictConstant.FEE_POOL_GROUP);
            if (!feePoolGroupMap.isEmpty()) {
                configVo.setPoolGroupList(Arrays.stream(poolGroup.split(",")).filter(feePoolGroupMap::containsKey).map(x -> {
                    FeeSelectOptionVo vo = new FeeSelectOptionVo();
                    vo.setValue(x);
                    vo.setText(feePoolGroupMap.get(x));
                    return vo;
                }).collect(Collectors.toList()));
            }
        }

        String poolType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_POOL_TYPE);
        if (StringUtils.isNotEmpty(poolType)) {
            final Map<String, DictDataVo> feePoolTypeMap = DictUtil.list(DictConstant.FEE_POOL_TYPE).stream().collect(Collectors.toMap(DictDataVo::getDictCode, v -> v));
            final Map<String, String> payTypeMap = DictUtil.dictMap(DictConstant.FEE_POOL_PAY_TYPE);
            if (!feePoolTypeMap.isEmpty()) {
                configVo.setPoolTypeList(Arrays.stream(poolType.split(",")).filter(feePoolTypeMap::containsKey).map(x -> {
                    DictDataVo dictDataVo = feePoolTypeMap.get(x);
                    FeeSelectOptionVo vo = new FeeSelectOptionVo();
                    vo.setValue(x);
                    vo.setText(dictDataVo.getDictValue());
                    String payType = dictDataVo.getExtendMap().get(DictConstant.FEE_POOL_TYPE_EXTEND_PAY_TYPE);
                    if (StringUtils.isNotEmpty(payType)) {
                        FeeSelectOptionVo payTypeSelectVo = new FeeSelectOptionVo();
                        payTypeSelectVo.setValue(payType);
                        payTypeSelectVo.setText(payTypeMap.get(payType));
                        vo.setPayTypeList(Collections.singletonList(payTypeSelectVo));
                    }
                    return vo;
                }).collect(Collectors.toList()));
            }
        }

        String useType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_USE_TYPE);
        if (StringUtils.isNotEmpty(useType)) {
            final Map<String, String> useTypeMap = DictUtil.dictMap(DictConstant.FEE_POOL_USE_TYPE);
            if (!useTypeMap.isEmpty()) {
                configVo.setUseTypeList(Arrays.stream(useType.split(",")).filter(useTypeMap::containsKey).map(x -> {
                    FeeSelectOptionVo vo = new FeeSelectOptionVo();
                    vo.setValue(x);
                    vo.setText(useTypeMap.get(x));
                    return vo;
                }).collect(Collectors.toList()));
            }
        }

        List<DictDataVo> operationTypeList = DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE);
        final Map<String, String> operationTypeMap = operationTypeList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
        String queryOperationType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_QUERY_OPERATION_TYPE);
        if (StringUtils.isNotEmpty(queryOperationType)) {
            if (!operationTypeMap.isEmpty()) {
                configVo.setQueryOperationTypeList(Arrays.stream(queryOperationType.split(",")).filter(operationTypeMap::containsKey).map(x -> {
                    FeeSelectOptionVo vo = new FeeSelectOptionVo();
                    vo.setValue(x);
                    vo.setText(operationTypeMap.get(x));
                    return vo;
                }).collect(Collectors.toList()));
            }
        }

        String accountOperationType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_ACCOUNT_OPERATION_TYPE);
        if (StringUtils.isNotEmpty(accountOperationType)) {
            if (!operationTypeMap.isEmpty()) {
                configVo.setAccountOperationTypeList(Arrays.stream(accountOperationType.split(",")).filter(operationTypeMap::containsKey).map(x -> {
                    FeeSelectOptionVo vo = new FeeSelectOptionVo();
                    vo.setValue(x);
                    vo.setText(operationTypeMap.get(x));
                    return vo;
                }).collect(Collectors.toList()));
            }
        }

        String adjustOperationType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_ADJUST_OPERATION_TYPE);
        if (StringUtils.isNotEmpty(adjustOperationType)) {
            if (!operationTypeMap.isEmpty()) {
                configVo.setAdjustOperationTypeList(Arrays.stream(adjustOperationType.split(",")).filter(operationTypeMap::containsKey).map(x -> {
                    FeeSelectOptionVo vo = new FeeSelectOptionVo();
                    vo.setValue(x);
                    vo.setText(operationTypeMap.get(x));
                    return vo;
                }).collect(Collectors.toList()));
            }
        }

        configVo.setFreezeOperationTypeList(operationTypeList.stream()
                .filter(item -> FeePoolOperationTypeGroupEnum.FREEZE.getValue().equals(item.getExtendMap().get(DictConstant.FEE_POOL_OPERATION_TYPE_EXTEND_OPERATION_GROUP)))
                .map(x -> {
                    FeeSelectOptionVo optionVo = new FeeSelectOptionVo();
                    optionVo.setValue(x.getDictCode());
                    optionVo.setText(x.getDictValue());
                    return optionVo;
                }).collect(Collectors.toList()));

        configVo.setUnfreezeOperationTypeList(operationTypeList.stream()
                .filter(item -> FeePoolOperationTypeGroupEnum.UNFREEZE.getValue().equals(item.getExtendMap().get(DictConstant.FEE_POOL_OPERATION_TYPE_EXTEND_OPERATION_GROUP)))
                .map(x -> {
                    FeeSelectOptionVo optionVo = new FeeSelectOptionVo();
                    optionVo.setValue(x.getDictCode());
                    optionVo.setText(x.getDictValue());
                    return optionVo;
                }).collect(Collectors.toList()));

        configVo.setLineDetailAdjustConfigCode(extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_LINE_DETAIL_ADJUST_CONFIG_CODE));

        return configVo;
    }

    /**
     * 根据配置编码获取查询的费用池类型编码集合
     *
     * @param configCode 配置编码
     * @return
     */
    public static List<String> getConfigQueryPoolTypeList(String configCode) {
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_PAGE_LIST_CONFIG, configCode);
        if (extendMap == null || extendMap.isEmpty()) {
            throw new BusinessException("未找到配置【" + configCode + "】的查询费用池类型");
        }
        String poolType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_POOL_TYPE);
        if (StringUtils.isNotEmpty(poolType)) {
            return Arrays.asList(poolType.split(","));
        }
        return Collections.emptyList();
    }

    /**
     * 根据配置编码获取查询的操作类型编码集合
     *
     * @param configCode 配置编码
     * @return
     */
    public static List<String> getConfigQueryOperationTypeList(String configCode) {
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_PAGE_LIST_CONFIG, configCode);
        if (extendMap == null || extendMap.isEmpty()) {
            throw new BusinessException("未找到配置【" + configCode + "】的查询费用池类型");
        }
        String operationType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_QUERY_OPERATION_TYPE);
        if (StringUtils.isNotEmpty(operationType)) {
            return Arrays.asList(operationType.split(","));
        }
        return Collections.emptyList();
    }

    /**
     * 根据配置编码获取费用使用类型集合
     *
     * @param configCode 配置编码
     * @return
     */
    public static List<String> getConfigUseTypeList(String configCode) {
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_PAGE_LIST_CONFIG, configCode);
        if (extendMap == null || extendMap.isEmpty()) {
            throw new BusinessException("未找到配置【" + configCode + "】的使用费用类型");
        }
        String operationType = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_USE_TYPE);
        if (StringUtils.isNotEmpty(operationType)) {
            return Arrays.asList(operationType.split(","));
        }
        return Collections.emptyList();
    }

    /**
     * 根据配置编码获取查询的费用池分组编码
     *
     * @param configCode 配置编码
     * @return
     */
    public static String getConfigQueryPoolGroup(String configCode) {
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_PAGE_LIST_CONFIG, configCode);
        if (extendMap == null || extendMap.isEmpty()) {
            throw new BusinessException("未找到配置【" + configCode + "】的查询费用池分组");
        }
        String poolGroup = extendMap.get(DictConstant.FEE_POOL_PAGE_LIST_CONFIG_EXTEND_POOL_GROUP);
        if (StringUtils.isNotEmpty(poolGroup)) {
            return poolGroup;
        }
        return null;
    }

    /**
     * 获取操作类型编码集合，根据分组枚举
     *
     * @param groupEnum 操作类型分组枚举
     * @return
     */
    public static List<String> getOperationTypeList(FeePoolOperationTypeGroupEnum groupEnum) {
        return DictUtil.list(DictConstant.FEE_POOL_OPERATION_TYPE)
                .stream()
                .filter(item -> groupEnum == null || groupEnum.getValue().equals(item.getExtendMap().get(DictConstant.FEE_POOL_OPERATION_TYPE_EXTEND_OPERATION_GROUP)))
                .map(DictDataVo::getDictCode).collect(Collectors.toList());
    }

    /**
     * 操作类型与操作类型分组匹配
     *
     * @param operationType 操作类型
     * @param groupEnum     操作类型分组枚举
     * @return
     */
    public static boolean checkOperationTypeGroup(String operationType, FeePoolOperationTypeGroupEnum groupEnum) {
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.hasText(operationTypeValue, "无效的操作类型");
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.isTrue(extendMap.containsKey(DictConstant.FEE_POOL_OPERATION_TYPE_EXTEND_OPERATION_GROUP), "操作类型" + operationTypeValue + "未配置分组");
        return groupEnum.getValue().equals(extendMap.get(DictConstant.FEE_POOL_OPERATION_TYPE_EXTEND_OPERATION_GROUP));
    }

    /**
     * 根据费用池类型获取支付方式
     *
     * @param poolType 费用池类型
     * @return
     */
    public static String getPayType(String poolType) {
        String poolTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_TYPE, poolType);
        Assert.hasText(poolTypeValue, "无效的费用池类型");
        Map<String, String> discountExtendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_TYPE, poolType);
        String payType = discountExtendMap.get(DictConstant.FEE_POOL_TYPE_EXTEND_PAY_TYPE);
        Assert.hasText(payType, poolTypeValue + "未配置支付方式");
        return payType;
    }

    /**
     * 根据操作类型获取操作类型分组
     *
     * @param operationType 操作类型
     * @return
     */
    public static FeePoolOperationTypeGroupEnum getOperationTypeGroup(String operationType) {
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.hasText(operationTypeValue, "无效的操作类型");
        Map<String, String> extendMap = DictUtil.getExtendMap(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        String group = extendMap.get(DictConstant.FEE_POOL_OPERATION_TYPE_EXTEND_OPERATION_GROUP);
        Assert.hasText(group, "操作类型" + operationTypeValue + "未配置分组");
        FeePoolOperationTypeGroupEnum anEnum = FeePoolOperationTypeGroupEnum.getEnum(group);
        Assert.notNull(anEnum, "操作类型" + operationTypeValue + "配置的分组无效");
        return anEnum;
    }

    /**
     * 获取Amount字段描述
     */
    public static String getFeePoolAmountDesc() {
        String desc = ParamUtil.getParameterValueNoException(ParameterParam.FEE_POOL_AMOUNT_DESC);
        if (StringUtils.isEmpty(desc)) {
            return "金额";
        }
        return desc;
    }

}
