package com.biz.crm.util;

import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mdm.org.MdmOrgRedisFeign;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.mdm.position.MdmPositionLevelFeign;
import com.biz.crm.mdm.position.MdmPositionRedisFeign;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionParentChildrenRedisVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRedisVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionReloadRedisVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserReqVo;
import com.biz.crm.service.RedisService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 职位查询工具类
 */
@Slf4j
@Component
public class PositionUtil {

    private static RedisService redisService;

    private static MdmPositionRedisFeign mdmPositionRedisFeign;

    private static MdmPositionFeign mdmPositionFeign;

  /**
   * 根据用户名查询所有职位
   * @param userName
   * @return
   */
  public static List<MdmPositionRespVo> getAllPositionByUsername(String userName) {
    if(StringUtils.isEmpty(userName)) {
      return Lists.newArrayList();
    }
    MdmUserReqVo param = new MdmUserReqVo();
    param.setUserName(userName);
    List<MdmPositionRespVo> positions = mdmPositionFeign.getUserAllPositionList(param).getResult();
    if(CollectionUtil.listEmpty(positions)) {
      return Lists.newArrayList();
    }
    return positions;
  }

  @Autowired
    public void setRedisService(RedisService redis) {
        PositionUtil.redisService = redis;
    }

    @Autowired
    public void setMdmPositionRedisFeign(MdmPositionRedisFeign positionRedisFeign) {
        PositionUtil.mdmPositionRedisFeign = positionRedisFeign;
    }

    @Autowired
    public void setMdmPositionFeign(MdmPositionFeign mdmpositionfeign) {
      PositionUtil.mdmPositionFeign = mdmpositionfeign;
    }

    /**
     * 根据职位编码获取职位详情
     *
     * @param positionCode 职位编码
     * @return
     */
    public static MdmPositionRespVo getPositionByCode(String positionCode) {
        if (positionCode != null && !"".equals(positionCode)) {
            Map<String, MdmPositionRedisVo> positionRedisMap = getPositionSelfRedisMapFromLocalFirst(Collections.singletonList(positionCode));
            if (positionRedisMap.containsKey(positionCode)) {
                MdmPositionRedisVo mdmPositionRedisVo = positionRedisMap.get(positionCode);
                return CrmBeanUtil.copy(mdmPositionRedisVo, MdmPositionRespVo.class);
            }
        }
        return null;
    }

    /**
     * 根据职位编码集合获取职位详情
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getPositionByCodeList(List<String> positionCodeList) {
        List<MdmPositionRespVo> list = new ArrayList<>();
        if (positionCodeList != null && positionCodeList.size() > 0) {
            HashSet<String> strings = Sets.newHashSet(positionCodeList);
            Map<String, MdmPositionRedisVo> positionRedisMap = getPositionSelfRedisMapFromLocalFirst(Lists.newArrayList(strings));
            if (!positionRedisMap.isEmpty()) {
                list.addAll(positionRedisMap.values().stream().map(item -> CrmBeanUtil.copy(item, MdmPositionRespVo.class)).collect(Collectors.toList()));
            }
        }
        return list;
    }


    /**
     * 根据职位编码获取职位及职位全部上级
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListIncludeSelf(String positionCode) {
        if (positionCode != null && !"".equals(positionCode)) {
            Map<String, MdmPositionParentChildrenRedisVo> map = getPositionParentRedisMapFromLocalFirst(Collections.singletonList(positionCode));
            if (!map.isEmpty()) {
                MdmPositionParentChildrenRedisVo redisVo = map.get(positionCode);
                if (redisVo != null) {
                    List<MdmPositionRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, MdmPositionRespVo.class);
                    }
                }
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码集合获取职位及职位全部上级
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListIncludeSelf(List<String> positionCodeList) {
        if (positionCodeList != null && positionCodeList.size() > 0) {
            Map<String, MdmPositionParentChildrenRedisVo> map = getPositionParentRedisMapFromLocalFirst(positionCodeList);
            Set<MdmPositionRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String positionCode :
                        positionCodeList) {
                    if (!map.containsKey(positionCode)) {
                        continue;
                    }
                    MdmPositionParentChildrenRedisVo redisVo = map.get(positionCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmPositionRespVo.class);
                }
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码获取职位（不含自己）全部上级
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListExcludeSelf(String positionCode) {
        if (positionCode != null && !"".equals(positionCode)) {
            List<MdmPositionRespVo> parentPositionListIncludeSelf = getParentPositionListIncludeSelf(positionCode);
            if (parentPositionListIncludeSelf != null && parentPositionListIncludeSelf.size() > 0) {
                return parentPositionListIncludeSelf.stream().filter(item -> !positionCode.equals(item.getPositionCode())).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的职位
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListExcludeSelf(List<String> positionCodeList) {
        if (positionCodeList != null && positionCodeList.size() > 0) {
            Map<String, MdmPositionParentChildrenRedisVo> map = getPositionParentRedisMapFromLocalFirst(positionCodeList);
            Set<MdmPositionRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String positionCode :
                        positionCodeList) {
                    if (!map.containsKey(positionCode)) {
                        continue;
                    }
                    MdmPositionParentChildrenRedisVo redisVo = map.get(positionCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !positionCode.equals(item.getPositionCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmPositionRespVo.class);
                }
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码集合获取职位（不含参数中的任何一个）全部上级
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListExcludeAnySelf(List<String> positionCodeList) {
        if (positionCodeList != null && positionCodeList.size() > 0) {
            List<MdmPositionRespVo> parentPositionListExcludeSelf = getParentPositionListExcludeSelf(positionCodeList);
            if (parentPositionListExcludeSelf != null && parentPositionListExcludeSelf.size() > 0) {
                final Set<String> positionCodeSet = new HashSet<>(positionCodeList);
                return parentPositionListExcludeSelf.stream().filter(item -> !positionCodeSet.contains(item.getPositionCode())).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }


    /**
     * 根据职位编码获取职位及职位全部下级
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListIncludeSelf(String positionCode) {
        if (positionCode != null && !"".equals(positionCode)) {
            Map<String, MdmPositionParentChildrenRedisVo> map = getPositionChildrenRedisMapFromLocalFirst(Collections.singletonList(positionCode));
            if (!map.isEmpty()) {
                MdmPositionParentChildrenRedisVo redisVo = map.get(positionCode);
                if (redisVo != null) {
                    List<MdmPositionRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, MdmPositionRespVo.class);
                    }
                }
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码集合获取职位及职位全部下级
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListIncludeSelf(List<String> positionCodeList) {
        if (positionCodeList != null && positionCodeList.size() > 0) {
            Map<String, MdmPositionParentChildrenRedisVo> map = getPositionChildrenRedisMapFromLocalFirst(positionCodeList);
            Set<MdmPositionRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String positionCode :
                        positionCodeList) {
                    if (!map.containsKey(positionCode)) {
                        continue;
                    }
                    MdmPositionParentChildrenRedisVo redisVo = map.get(positionCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmPositionRespVo.class);
                }
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码获取职位（不含自己）全部下级
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListExcludeSelf(String positionCode) {
        if (positionCode != null && !"".equals(positionCode)) {
            List<MdmPositionRespVo> parentPositionListIncludeSelf = getChildrenPositionListIncludeSelf(positionCode);
            if (parentPositionListIncludeSelf != null && parentPositionListIncludeSelf.size() > 0) {
                return parentPositionListIncludeSelf.stream().filter(item -> !positionCode.equals(item.getPositionCode())).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的职位
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListExcludeSelf(List<String> positionCodeList) {
        if (positionCodeList != null && positionCodeList.size() > 0) {
            Map<String, MdmPositionParentChildrenRedisVo> map = getPositionChildrenRedisMapFromLocalFirst(positionCodeList);
            Set<MdmPositionRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String positionCode :
                        positionCodeList) {
                    if (!map.containsKey(positionCode)) {
                        continue;
                    }
                    MdmPositionParentChildrenRedisVo redisVo = map.get(positionCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !positionCode.equals(item.getPositionCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), MdmPositionRespVo.class);
                }
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据职位编码集合获取职位（不含参数中任何一个）全部下级
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListExcludeAnySelf(List<String> positionCodeList) {
        if (positionCodeList != null && positionCodeList.size() > 0) {
            List<MdmPositionRespVo> parentPositionListExcludeSelf = getChildrenPositionListExcludeSelf(positionCodeList);
            if (parentPositionListExcludeSelf != null && parentPositionListExcludeSelf.size() > 0) {
                final Set<String> positionCodeSet = new HashSet<>(positionCodeList);
                return parentPositionListExcludeSelf.stream().filter(item -> !positionCodeSet.contains(item.getPositionCode())).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }


    /**
     * 根据职位编码获取职位及职位全部上级（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListIncludeSelfEnable(String positionCode) {
        return getParentPositionListIncludeSelf(positionCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位及职位全部上级（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListIncludeSelfEnable(List<String> positionCodeList) {
        return getParentPositionListIncludeSelf(positionCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码获取职位（不含自己）全部上级（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListExcludeSelfEnable(String positionCode) {
        return getParentPositionListExcludeSelf(positionCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的职位（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListExcludeSelfEnable(List<String> positionCodeList) {
        return getParentPositionListExcludeSelf(positionCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含参数中的任何一个）全部上级（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getParentPositionListExcludeAnySelfEnable(List<String> positionCodeList) {
        return getParentPositionListExcludeAnySelf(positionCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据职位编码获取职位及职位全部下级（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListIncludeSelfEnable(String positionCode) {
        return getChildrenPositionListIncludeSelf(positionCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位及职位全部下级（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListIncludeSelfEnable(List<String> positionCodeList) {
        return getChildrenPositionListIncludeSelf(positionCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码获取职位（不含自己）全部下级（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListExcludeSelfEnable(String positionCode) {
        return getChildrenPositionListExcludeSelf(positionCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的职位（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListExcludeSelfEnable(List<String> positionCodeList) {
        return getChildrenPositionListExcludeSelf(positionCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含参数中任何一个）全部下级（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<MdmPositionRespVo> getChildrenPositionListExcludeAnySelfEnable(List<String> positionCodeList) {
        return getChildrenPositionListExcludeAnySelf(positionCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据职位编码获取职位及职位全部上级职位编码
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getParentPositionCodeListIncludeSelf(String positionCode) {
        return getParentPositionListIncludeSelf(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位及职位全部上级职位编码
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getParentPositionCodeListIncludeSelf(List<String> positionCodeList) {
        return getParentPositionListIncludeSelf(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码获取职位（不含自己）全部上级职位编码
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getParentPositionCodeListExcludeSelf(String positionCode) {
        return getParentPositionListExcludeSelf(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部上级职位编码，如果本身有上下级关系，则会返回处于上级的职位
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getParentPositionCodeListExcludeSelf(List<String> positionCodeList) {
        return getParentPositionListExcludeSelf(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含参数中的任何一个）全部上级职位编码
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getParentPositionCodeListExcludeAnySelf(List<String> positionCodeList) {
        return getParentPositionListExcludeAnySelf(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }


    /**
     * 根据职位编码获取职位及职位全部下级职位编码
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getChildrenPositionCodeListIncludeSelf(String positionCode) {
        return getChildrenPositionListIncludeSelf(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位及职位全部下级职位编码
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getChildrenPositionCodeListIncludeSelf(List<String> positionCodeList) {
        return getChildrenPositionListIncludeSelf(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码获取职位（不含自己）全部下级职位编码
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getChildrenPositionCodeListExcludeSelf(String positionCode) {
        return getChildrenPositionListExcludeSelf(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部下级职位编码，如果参数本身有上下级关系，则会返回处于下级的职位
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getChildrenPositionCodeListExcludeSelf(List<String> positionCodeList) {
        return getChildrenPositionListExcludeSelf(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含参数中任何一个）全部下级职位编码
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getChildrenPositionCodeListExcludeAnySelf(List<String> positionCodeList) {
        return getChildrenPositionListExcludeAnySelf(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }


    /**
     * 根据职位编码获取职位及职位全部上级职位编码（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getParentPositionCodeListIncludeSelfEnable(String positionCode) {
        return getParentPositionListIncludeSelfEnable(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位及职位全部上级职位编码（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getParentPositionCodeListIncludeSelfEnable(List<String> positionCodeList) {
        return getParentPositionListIncludeSelfEnable(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码获取职位（不含自己）全部上级职位编码（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getParentPositionCodeListExcludeSelfEnable(String positionCode) {
        return getParentPositionListExcludeSelfEnable(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部上级职位编码，如果本身有上下级关系，则会返回处于上级的职位（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getParentPositionCodeListExcludeSelfEnable(List<String> positionCodeList) {
        return getParentPositionListExcludeSelfEnable(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含参数中的任何一个）全部上级职位编码（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getParentPositionCodeListExcludeAnySelfEnable(List<String> positionCodeList) {
        return getParentPositionListExcludeAnySelfEnable(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }


    /**
     * 根据职位编码获取职位及职位全部下级职位编码（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getChildrenPositionCodeListIncludeSelfEnable(String positionCode) {
        return getChildrenPositionListIncludeSelfEnable(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位及职位全部下级职位编码（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getChildrenPositionCodeListIncludeSelfEnable(List<String> positionCodeList) {
        return getChildrenPositionListIncludeSelfEnable(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码获取职位（不含自己）全部下级职位编码（只查启用）
     *
     * @param positionCode 职位编码
     * @return
     */
    public static List<String> getChildrenPositionCodeListExcludeSelfEnable(String positionCode) {
        return getChildrenPositionListExcludeSelfEnable(positionCode).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含自己）全部下级职位编码，如果参数本身有上下级关系，则会返回处于下级的职位（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getChildrenPositionCodeListExcludeSelfEnable(List<String> positionCodeList) {
        return getChildrenPositionListExcludeSelfEnable(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }

    /**
     * 根据职位编码集合获取职位（不含参数中任何一个）全部下级职位编码（只查启用）
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    public static List<String> getChildrenPositionCodeListExcludeAnySelfEnable(List<String> positionCodeList) {
        return getChildrenPositionListExcludeAnySelfEnable(positionCodeList).stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
    }


    /**
     * 清除全部职位Redis缓存和本地缓存
     */
    public static void deleteAllCache() {
        redisService.del(RedisParam.MDM_POSITION_LOCK_HASH);
        redisService.del(RedisParam.MDM_POSITION_SELF_HASH);
        redisService.del(RedisParam.MDM_POSITION_PARENT_HASH);
        redisService.del(RedisParam.MDM_POSITION_CHILDREN_HASH);
        ThreadLocalUtil.delPosition();
    }

    /**
     * 重新加载redis缓存
     */
    public static void reloadAllCache() {
        deleteAllCache();
        Result<List<MdmPositionReloadRedisVo>> result = mdmPositionRedisFeign.findRedisReloadList();
        if (result.isSuccess()) {
            List<MdmPositionReloadRedisVo> list = result.getResult();
            if (list != null && list.size() > 0) {
                List<List<MdmPositionReloadRedisVo>> partition = Lists.partition(list, 500);
                for (List<MdmPositionReloadRedisVo> item :
                        partition) {
                    Map<String, MdmPositionRedisVo> selfMap = item.stream().collect(Collectors.toMap(MdmPositionReloadRedisVo::getPositionCode, MdmPositionReloadRedisVo::getSelf));
                    Map<String, List<MdmPositionRedisVo>> parentMap = item.stream().collect(Collectors.toMap(MdmPositionReloadRedisVo::getPositionCode, MdmPositionReloadRedisVo::getParentList));
                    Map<String, List<MdmPositionRedisVo>> childrenMap = item.stream().collect(Collectors.toMap(MdmPositionReloadRedisVo::getPositionCode, MdmPositionReloadRedisVo::getChildrenList));
                    redisService.hmset(RedisParam.MDM_POSITION_SELF_HASH, new HashMap<Object, Object>(selfMap));
                    redisService.hmset(RedisParam.MDM_POSITION_PARENT_HASH, new HashMap<Object, Object>(parentMap));
                    redisService.hmset(RedisParam.MDM_POSITION_CHILDREN_HASH, new HashMap<Object, Object>(childrenMap));
                }
            }
        }
    }

    /*--------------------------------------------------以下是私有方法--------------------------------------------------*/

    /**
     * 获取职位缓存数据，优先从本地缓存取，再从redis取
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    private static Map<String, MdmPositionRedisVo> getPositionSelfRedisMapFromLocalFirst(List<String> positionCodeList) {
        Map<String, MdmPositionRedisVo> map = new HashMap<>(16);
        try {
            if (positionCodeList != null && positionCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String positionCode :
                        positionCodeList) {
                    if (ThreadLocalUtil.positionSelfContains(positionCode)) {
                        MdmPositionRedisVo fromPositionSelf = ThreadLocalUtil.getFromPositionSelf(positionCode);
                        if (fromPositionSelf != null) {
                            map.put(positionCode, fromPositionSelf);
                        }
                    } else {
                        getFromRedisCodeSet.add(positionCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmPositionRedisVo> mapFromRedisFirst = getPositionSelfRedisMapFromRedisFirst(new ArrayList<>(positionCodeList));

                    for (String positionCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(positionCode)) {
                            map.put(positionCode, mapFromRedisFirst.get(positionCode));
                        }
                        ThreadLocalUtil.addToPositionSelf(positionCode, mapFromRedisFirst.get(positionCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("职位查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取职位缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    private static Map<String, MdmPositionRedisVo> getPositionSelfRedisMapFromRedisFirst(List<String> positionCodeList) {
        Map<String, MdmPositionRedisVo> map = new HashMap<>(16);
        try {
            if (positionCodeList != null && positionCodeList.size() > 0) {
                List<MdmPositionRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_POSITION_LOCK_HASH, new HashSet<Object>(positionCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = positionCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> positionListFromRedis = redisService.hmget(RedisParam.MDM_POSITION_SELF_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (positionListFromRedis != null && positionListFromRedis.size() > 0) {
                        List<MdmPositionRedisVo> collect = positionListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmPositionRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmPositionRedisVo::getPositionCode).collect(Collectors.toSet()));
                        }
                    }
                }

                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmPositionRedisVo>> result = mdmPositionRedisFeign.findRedisListByPositionCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmPositionRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmPositionRedisVo> collect = list.stream().collect(Collectors.toMap(MdmPositionRedisVo::getPositionCode, v -> v));

                            Map<String, MdmPositionRedisVo> cacheMap = new HashMap<>(16);
                            for (String positionCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(positionCode)) {
                                    lockCodeMap.put(positionCode, positionCode);
                                } else {
                                    redisVoList.add(collect.get(positionCode));
                                    cacheMap.put(positionCode, collect.get(positionCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_POSITION_SELF_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_POSITION_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmPositionRedisVo::getPositionCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("职位查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取职位上级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    private static Map<String, MdmPositionParentChildrenRedisVo> getPositionParentRedisMapFromLocalFirst(List<String> positionCodeList) {
        Map<String, MdmPositionParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (positionCodeList != null && positionCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String positionCode :
                        positionCodeList) {
                    if (ThreadLocalUtil.positionParentContains(positionCode)) {
                        MdmPositionParentChildrenRedisVo fromPositionParent = ThreadLocalUtil.getFromPositionParent(positionCode);
                        if (fromPositionParent != null) {
                            map.put(positionCode, fromPositionParent);
                        }
                    } else {
                        getFromRedisCodeSet.add(positionCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmPositionParentChildrenRedisVo> mapFromRedisFirst = getPositionParentRedisMapFromRedisFirst(new ArrayList<>(positionCodeList));

                    for (String positionCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(positionCode)) {
                            map.put(positionCode, mapFromRedisFirst.get(positionCode));
                        }
                        ThreadLocalUtil.addToPositionParent(positionCode, mapFromRedisFirst.get(positionCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("职位查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取职位上级缓存数据，优先从redis缓存存取，再从数据库取
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    private static Map<String, MdmPositionParentChildrenRedisVo> getPositionParentRedisMapFromRedisFirst(List<String> positionCodeList) {
        Map<String, MdmPositionParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (positionCodeList != null && positionCodeList.size() > 0) {
                List<MdmPositionParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_POSITION_LOCK_HASH, new HashSet<Object>(positionCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = positionCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> positionListFromRedis = redisService.hmget(RedisParam.MDM_POSITION_PARENT_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (positionListFromRedis != null && positionListFromRedis.size() > 0) {
                        List<MdmPositionParentChildrenRedisVo> collect = positionListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmPositionParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmPositionParentChildrenRedisVo::getPositionCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmPositionParentChildrenRedisVo>> result = mdmPositionRedisFeign.findRedisPositionParentListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmPositionParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmPositionParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmPositionParentChildrenRedisVo::getPositionCode, v -> v));

                            Map<String, MdmPositionParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String positionCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(positionCode)) {
                                    lockCodeMap.put(positionCode, positionCode);
                                } else {
                                    redisVoList.add(collect.get(positionCode));
                                    cacheMap.put(positionCode, collect.get(positionCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_POSITION_PARENT_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_POSITION_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmPositionParentChildrenRedisVo::getPositionCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("职位查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取职位下级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    private static Map<String, MdmPositionParentChildrenRedisVo> getPositionChildrenRedisMapFromLocalFirst(List<String> positionCodeList) {
        Map<String, MdmPositionParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (positionCodeList != null && positionCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String positionCode :
                        positionCodeList) {
                    if (ThreadLocalUtil.positionChildrenContains(positionCode)) {
                        MdmPositionParentChildrenRedisVo fromPositionChildren = ThreadLocalUtil.getFromPositionChildren(positionCode);
                        if (fromPositionChildren != null) {
                            map.put(positionCode, fromPositionChildren);
                        }
                    } else {
                        getFromRedisCodeSet.add(positionCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmPositionParentChildrenRedisVo> mapFromRedisFirst = getPositionChildrenRedisMapFromRedisFirst(new ArrayList<>(positionCodeList));

                    for (String positionCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(positionCode)) {
                            map.put(positionCode, mapFromRedisFirst.get(positionCode));
                        }
                        ThreadLocalUtil.addToPositionChildren(positionCode, mapFromRedisFirst.get(positionCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("职位查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取职位下级缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param positionCodeList 职位编码集合
     * @return
     */
    private static Map<String, MdmPositionParentChildrenRedisVo> getPositionChildrenRedisMapFromRedisFirst(List<String> positionCodeList) {
        Map<String, MdmPositionParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (positionCodeList != null && positionCodeList.size() > 0) {
                List<MdmPositionParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_POSITION_LOCK_HASH, new HashSet<Object>(positionCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = positionCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> positionListFromRedis = redisService.hmget(RedisParam.MDM_POSITION_CHILDREN_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (positionListFromRedis != null && positionListFromRedis.size() > 0) {
                        List<MdmPositionParentChildrenRedisVo> collect = positionListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmPositionParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmPositionParentChildrenRedisVo::getPositionCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmPositionParentChildrenRedisVo>> result = mdmPositionRedisFeign.findRedisPositionChildrenListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmPositionParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmPositionParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmPositionParentChildrenRedisVo::getPositionCode, v -> v));

                            Map<String, MdmPositionParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String positionCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(positionCode)) {
                                    lockCodeMap.put(positionCode, positionCode);
                                } else {
                                    redisVoList.add(collect.get(positionCode));
                                    cacheMap.put(positionCode, collect.get(positionCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_POSITION_CHILDREN_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_POSITION_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmPositionParentChildrenRedisVo::getPositionCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("职位查询工具类异常：{}", e);
        }
        return map;
    }

}
