package com.biz.crm.eunm.dms;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import lombok.Getter;
import lombok.SneakyThrows;

/**
 * 促销政策范围类型枚举（自定义动态枚举）
 * @Author: chenrong
 * @Date: 2021/4/13 9:56
 */
@Getter
public class ScopeTypeDynamicEnum {

  /**
   * 促销范围类型字典编码
   **/
  private static final String PROMOTION_POLICY_SCOPE_TYPE = "promotion_policy_scope_type_new";
  private static final String PREF = "scope";
  private static final String Y = "Y";
  private static final String N = "N";

  //客户组织-枚举
  public static final ScopeTypeDynamicEnum CUS_ORG = initEnum("cusOrg", "客户组织");
  //客户-枚举
  public static final ScopeTypeDynamicEnum CUS = initEnum("cus", "客户");
  //终端组织-枚举
  public static final ScopeTypeDynamicEnum TERMINAL_ORG = initEnum("terminalOrg", "客户组织");
  //终端-枚举
  public static final ScopeTypeDynamicEnum TERMINAL = initEnum("terminal", "终端");
  //渠道-枚举
  public static final ScopeTypeDynamicEnum CUS_CHANNEL = initEnum("channel", "渠道");

  //编码，与相关字典中的编码必须保持一致
  private String code;
  //描述
  private String desc;
  //包含，属性名
  private String containName;
  //非包含，属性名
  private String unContainName;

  ScopeTypeDynamicEnum(String code, String desc, String containName, String unContainName) {
    this.code = code;
    this.desc = desc;
    this.containName = containName;
    this.unContainName = unContainName;
  }

  /**
   * 初始化静态枚举
   * @param code
   * @param desc
   * @return
   */
  private static ScopeTypeDynamicEnum initEnum(String code, String desc) {
    String containName = buildObjectName(code, Y);
    String unContainName = buildObjectName(code, N);
    ScopeTypeDynamicEnum scopeTypeEnum = new ScopeTypeDynamicEnum(code, desc, containName, unContainName);
    return scopeTypeEnum;
  }

  /**
   * 构建参数对象名称
   * @param code
   * @param contains
   * @return
   */
  public static String buildObjectName(String code, String contains) {
    return fieldLineToHump(PREF.concat("_").concat(code).concat("_").concat(contains));
  }

  /**
   * 根据字典编码获取枚举
   * @return
   */
  public static ScopeTypeDynamicEnum getScopeTypeEnumsByCode(String dictCode) {
    String value = DictUtil.dictValue(PROMOTION_POLICY_SCOPE_TYPE, dictCode);
    if (StringUtils.isEmpty(value)) {
      return null;
    }
    String containName = buildObjectName(dictCode, Y);
    String unContainName = buildObjectName(dictCode, N);
    ScopeTypeDynamicEnum scopeTypeEnum = new ScopeTypeDynamicEnum(dictCode, value, containName, unContainName);
    return scopeTypeEnum;
  }

  /**
   * 根据参数对象名称获取枚举
   * @param objectName
   * @return
   */
  public static ScopeTypeDynamicEnum getScopeTypeEnumsByObjectName(String objectName) {
    List<ScopeTypeDynamicEnum> scopeTypeDynamicEnums = getScopeTypeEnums();
    scopeTypeDynamicEnums = scopeTypeDynamicEnums.stream().filter(scope ->
            Objects.equals(objectName, scope.getContainName()) || Objects.equals(objectName, scope.getUnContainName())).collect(Collectors.toList());
    if(CollectionUtil.listEmpty(scopeTypeDynamicEnums)) {
      return null;
    }
    return scopeTypeDynamicEnums.stream().findFirst().get();
  }

  /**
   * 根据字典编码获取枚举
   * @return
   */
  public static List<ScopeTypeDynamicEnum> getScopeTypeEnums() {
    Map<String, String> dict = DictUtil.dictMap(PROMOTION_POLICY_SCOPE_TYPE);
    if (dict == null) {
      return Lists.newArrayList();
    }
    List<ScopeTypeDynamicEnum> scopeTypeEnums = Lists.newArrayList();
    dict.forEach((k, v) -> {
      String code = k;
      String desc = v;
      String containName = buildObjectName(code, Y);
      String unContainName = buildObjectName(code, N);
      ScopeTypeDynamicEnum scopeTypeEnum = new ScopeTypeDynamicEnum(code, desc, containName, unContainName);
      scopeTypeEnums.add(scopeTypeEnum);
    });
    return scopeTypeEnums;
  }

  /**
   * 获取所有静态枚举
   * @return
   */
  @SneakyThrows
  public static List<ScopeTypeDynamicEnum> getAllStaticEnums(){
    Class cls = ScopeTypeDynamicEnum.class;
    Field[] declaredFields = cls.getDeclaredFields();
    List<ScopeTypeDynamicEnum> enums = Lists.newArrayList();
    for(Field f : declaredFields) {
      String modifier = Modifier.toString(f.getModifiers());
      if(Objects.equals(modifier, "public static final")) {
        f.setAccessible(true);
        enums.add((ScopeTypeDynamicEnum) f.get(null));
      }
    }
    return enums;
  }

  /**
   * 数据库字段名转化成java字段名
   * @param sourceName
   * @return
   */
  public static String fieldLineToHump(String sourceName) {
    Pattern linePattern = Pattern.compile("_(\\w)");
    Matcher matcher = linePattern.matcher(sourceName);
    StringBuffer sb = new StringBuffer();
    while (matcher.find()) {
      matcher.appendReplacement(sb, matcher.group(1).toUpperCase());
    }
    matcher.appendTail(sb);
    return sb.toString();
  }

}
