package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.time.LocalTime;

/**
 *  日期类断言工具
 * @ClassName AssertUtils
 * @Author 任钢
 * @Date 2019/2/21 11:28
 * @Description 断言工具类，判断是否满足断言,如果不满足,则抛出CRM异常
 */
public class AssertDateTimeUtils {

    /**
     * 校验时间格式
     * HH:SS:MM
     * @param time
     * @param msg
     */
    public static void checkTime(String time, String msg) {
        try {
            LocalTime.parse(time);
        } catch (Exception e) {
            throw new BusinessException(msg);
        }
    }

    /**
     * 开始时间不能在结束时间之后
     * @param beginTime
     * @param endTime
     */
    public static void notAfterTime(String beginTime, String endTime, String msg) {
        if(LocalTime.parse(beginTime).isAfter(LocalTime.parse(endTime))) {
            throw new BusinessException(msg);
        }
    }

    /**
     * 开始日期不能在结束日期之后
     * @param beginTime
     * @param endTime
     */
    public static void notAfterDate(String beginTime, String endTime, String msg) {
        if(LocalDate.parse(beginTime).isAfter(LocalDate.parse(endTime))) {
            throw new BusinessException(msg);
        }
    }

    /**
     * 一次验证多个时间格式
     * @param msg
     * @param times
     */
    public static void checkTimeMore(String msg, String ...times) {
        for(String time : times) {
            checkTime(time, msg);
        }
    }

    /**
     * 校验日期格式
     * yyyy-MM-dd
     * @param date 日期字符串
     * @param msg 错误格式提示
     */
    public static void checkDate(String date, String msg) {
        try {
            LocalDate.parse(date);
        } catch (Exception e) {
            throw new BusinessException(msg);
        }
    }

    /**
     * 校验日期格式，并且校验空
     * @param date 日期字符串
     * @param nullMsg 空数据提示
     * @param msg 错误格式提示
     */
    public static void checkDateAndNull(String date, String nullMsg, String msg) {
        if(StringUtils.isEmpty(date)) {
            throw new BusinessException(nullMsg);
        }
        checkDate(date, msg);
    }

    /**
     * 校验在两个时间之间
     * @param nowTime
     * @param beginTime
     * @param endTime
     * @param msg
     */
    public static void checkBetweenTime(String nowTime, String beginTime, String endTime, String msg) {
        LocalTime timeNow = LocalTime.parse(nowTime);
        LocalTime timeBegin = LocalTime.parse(beginTime);
        LocalTime timeEnd = LocalTime.parse(endTime);
        if(!(timeBegin.compareTo(timeNow) <= 0 && timeEnd.compareTo(timeNow) >= 0)) {
            throw new BusinessException(msg);
        }
    }


}
