package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.mdm.feign.MdmCodeRuleGeneratorFeign;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.service.RedisService;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * @Author zhangyuzhu
 * @Description code工具类
 * @Created data 2018/8/24 12:40
 */
public class CodeUtil {
    private static RedisService redisService;
    private final long workerIdBits = 5L;
    private final long datacenterIdBits = 5L;
    private final long maxWorkerId = ~(-1L << workerIdBits);
    private final long maxDatacenterId = ~(-1L << datacenterIdBits);
    private final long sequenceBits = 10L;
    private final long timestampLeftShift = sequenceBits;
    /** 生成序列的掩码，这里为4095 (0b111111111111=0xfff=4095) */
    private final long sequenceMask = ~(-1L << sequenceBits);
    /** 工作机器ID(0~31) */
    private long workerId;
    /** 数据中心ID(0~31) */
    private long datacenterId;
    /** 毫秒内序列(0~4095) */
    private long sequence = 0L;
    /** 上次生成ID的时间截 */
    private long lastTimestamp = -1L;
    private static CodeUtil idWorker;
    private final long TWEPOCH = 1596037883039L;
    private static MdmCodeRuleGeneratorFeign mdmCodeRuleGeneratorFeign;
    public static final String RULE_CODE="RULE_CODE_";
    public static final String yyyymmddHHmm="yyyymmddhhmm_";

    /**
     * 快速生成编码
     * @return
     */
    public static String getCode() {
     return getCodeDefault();
    }
    private CodeUtil(long workerId, long datacenterId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }
        if (datacenterId > maxDatacenterId || datacenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0", maxDatacenterId));
        }
        this.workerId = workerId;
        this.datacenterId = datacenterId;
    }
    protected long tilNextMillis(long lastTimestamp) {
        long timestamp = System.currentTimeMillis();
        while (timestamp <= lastTimestamp) {
            timestamp = System.currentTimeMillis();
        }
        return timestamp;
    }
    protected synchronized long nextId() {
        long timestamp = System.currentTimeMillis();
        if (timestamp < lastTimestamp) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp - timestamp));
        }
        //如果是同一时间生成的，则进行毫秒内序列
        if (lastTimestamp == timestamp) {
            sequence = (sequence + 1) & sequenceMask;
            //毫秒内序列溢出
            if (sequence == 0) {
                //阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(lastTimestamp);
            }
        }
        //时间戳改变，毫秒内序列重置
        else {
            sequence = 0L;
        }
        //上次生成ID的时间截
        lastTimestamp = timestamp;
        //移位并通过或运算拼到一起组成64位的ID
        return ((timestamp - TWEPOCH) << timestampLeftShift) | sequence;
    }


    static final char[] DIGITS =
            { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
                    'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j',
                    'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
                    'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 'D',
                    'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
                    'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z' };

    private static String to62RadixString(long seq) {
        StringBuilder sb = new StringBuilder();
        do {
            int remainder = (int) (seq % 62);
            sb.append(DIGITS[remainder]);
            seq = seq / 62;
        } while (seq != 0);
        return sb.reverse().toString();
    }
    /**
     *  生成三位数的code（前面用0填充）
     *  用于生成树结构的规则code
     * @param parentCode
     * @param rank
     * @return
     */
    public static String fullThree(String parentCode,int rank){
        String rankStr = "";
        if(rank < 10){
            rankStr = "00"+ rank;
        }else if(rank < 100){
            rankStr = "0"+ rank;
        }else{
            rankStr = ""+ rank;
        }
        if(parentCode == null || parentCode.equals("")){
            return rankStr;
        }
        return new StringBuffer(parentCode).append(rankStr).toString();
    }


    /**
     * 生成编码(临时用)
     * @return
     */
    public static String createCode(){
        return getCode();
    }

    /**
     * 传入一个功能名称,通过功能名称查询编码规则表获取编码规则,然后生成对应的编码
     * @param functionName
     * @return
     */
    public static String createOneCode(String functionName){
        return generateCodeList(functionName,1).get(0);
    }

    /**
     * 传入一个功能名称,和需要的编码个数通过功能名称查询编码规则表获取编码规则,然后生成对应的编码集合
     * @param functionName 功能代码
     * @param num 编码个数
     * @return
     */
    public static List<String> createCodeList(String functionName, int num){
        return generateCodeList(functionName,num);
    }

    public static void setMdmCodeRuleGeneratorFeign(MdmCodeRuleGeneratorFeign feign) {
        mdmCodeRuleGeneratorFeign = feign;
    }

    /**
     * 生成多个默认编码
     *
     * @param number 生成数量
     * @return
     */
    public static List<String> generateCodeList(int number) {
        List<String> list = new ArrayList<>();
        if (number > 0) {
            Result<List<String>> result = mdmCodeRuleGeneratorFeign.generateCodeList(CodeRuleEnum.DEFAULT_CODE.getCode(), number);
            if (!result.isSuccess()) {
                throw new BusinessException("生成默认编码失败：" + result.getMessage());
            }
            list.addAll(result.getResult());
        }
        return list;
    }

    /**
     * 生成多个编码（按照业务key）
     *
     * @param ruleCode 规则编码
     * @param number   生成数量
     * @return
     */
    public static List<String> generateCodeList(String ruleCode, int number) {
        List<String> list = new ArrayList<>();
        if (ruleCode != null && !"".equals(ruleCode) && number > 0) {
            Result<List<String>> result = mdmCodeRuleGeneratorFeign.generateCodeList(ruleCode, number);
            if (!result.isSuccess()) {
                throw new BusinessException(ruleCode+"生成编码失败：" + result.getMessage());
            }
            return result.getResult();
        }
        return list;
    }

    /**
     * 生成一个默认编码
     *
     * @return
     */
    public static String generateCode() {
        List<String> list = generateCodeList(1);
        if (list != null && list.size() > 0) {
            return list.get(0);
        }
        return "";
    }

    /**
     * 生成一个编码（按照业务key）
     *
     * @param ruleCode 编码规则key
     * @return
     */
    public static String generateCode(String ruleCode) {
        List<String> list = generateCodeList(ruleCode, 1);
        if (list != null && list.size() > 0) {
            return list.get(0);
        }
        return "";
    }

    public static String getCodeDefault(){
        if(redisService==null){
            redisService = SpringApplicationContextUtil.getApplicationContext().getBean(RedisService.class);
        }
        //获取当前时分秒
        String nodate = LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmss_);
        long numCode= redisService.incr(RULE_CODE,1);
        if (numCode==9999999){
            redisService.set(RULE_CODE,1);
            numCode=1;
        }
        return  nodate+String.format("%07d", numCode);
    }

    public static String getCodeYyyymmddHHmm(){
        if(redisService==null){
            redisService = SpringApplicationContextUtil.getApplicationContext().getBean(RedisService.class);
        }
        //获取当前时分秒
        String nodate = LocalDateTime.now().format(CrmDateUtils.yyMMddHHmm);
        long numCode= redisService.incr(yyyymmddHHmm,1);
        if (numCode==9999){
            redisService.set(yyyymmddHHmm,1);
            numCode=1;
        }
        return  nodate+String.format("%04d", numCode);
    }

}
