package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.param.RedisBaseParam;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.config.TokenRenewalUtil;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.mdm.MdmCommonUserFeign;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.service.RedisService;
import com.google.common.collect.Lists;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.util
 * @Description: 登录信息
 * @date 2020/8/26 下午3:04
 */
@Slf4j
@Component
public class UserUtils {
    private static UserAutoConfig userAutoConfig;
    private static RedisService redisService;
    private static MdmCommonUserFeign mdmCommonUserFeign;
    private static TokenRenewalUtil tokenRenewalUtil;
    private static RedisTemplate<String, Object> redisTemplate;

    @Autowired
    public void setRedisService(RedisService redisService) {
        UserUtils.redisService = redisService;
    }
    @Autowired
    public  void setTokenRenewalUtil(TokenRenewalUtil tokenRenewalUtil) {
        UserUtils.tokenRenewalUtil = tokenRenewalUtil;
    }

    @Autowired
    public void setUserAutoConfig(UserAutoConfig userAutoConfig) {
        UserUtils.userAutoConfig = userAutoConfig;
    }

    @Autowired
    public void setRedisTemplate(RedisTemplate<String, Object> redisTemplate) {
        UserUtils.redisTemplate = redisTemplate;
    }

    @Autowired
    public void setMdmCommonUserFeign(MdmCommonUserFeign mdmCommonUserFeign) {
        UserUtils.mdmCommonUserFeign = mdmCommonUserFeign;
    }

    /**
     * 获取当前登录人信息
     *
     * @return
     */
    public static UserRedis getUser() {
        UserRedis user = ThreadLocalUtil.getUser();
        if (user != null && StringUtils.isNotEmpty(user.getUsername())) {
            return user;
        }
        UserRedis userRedis = null;
        try {
            String token = getToken();
            if (StringUtils.isEmpty(token)) {
                log.info("未获取登录用户token信息");
            } else {
                Object o = redisService.get(RedisBaseParam.TOKEN + token);
                if (o == null) {
                    setUserSeconds(token, null, RedisParam.TIME10);
                } else {
                    userRedis = (UserRedis) o;
                    ThreadLocalUtil.setUser(userRedis);
                }
            }
        } catch (Exception e) {
            log.error("获取当前用户:", e);
        }
        return userRedis;
    }

    /**
     * 获取当前token
     *
     * @return
     */
    public static String getToken() {
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        Object tokenLock = ThreadLocalUtil.getObj(GlobalParam.TOKEN);
        String token = tokenLock == null ? "" : tokenLock.toString();
        if (StringUtils.isNotEmpty(token) || request == null) {
            return token;
        }
        if (request != null) {
            token = request.getHeader(GlobalParam.TOKEN);
            if (StringUtils.isEmpty(token)) {
                Cookie cookie = CookiesUtil.getCookieByName(request, GlobalParam.TOKEN);
                if (cookie != null) {
                    token = cookie.getValue();
                }
            }
        }
        return token;
    }

    public static String getToken(String key) {
        return redisService.getAndSet(RedisBaseParam.TOKEN + key);
    }

    /**
     * 将登录token添加到本地线程变量
     *
     * @param token
     */
    public static void setToken(String token) {
        ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
        map.put(GlobalParam.TOKEN, token);
        ThreadLocalUtil.stObj(map);
    }

    /**
     * 默认设置当前登录权限，如果没有登录情况
     */
    public static void doTokenForNull() {
        if (StringUtils.isEmpty(userAutoConfig.getUsername())) {
            return;
        }
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            String token = UUID.randomUUID().toString().replaceAll("-", "");
            map.put(GlobalParam.TOKEN, token);
//            UserRedis userRedis = new UserRedis();
//            userRedis.setUsername(userAutoConfig.getUsername());
            ThreadLocalUtil.stObj(map);
            createUserToRedis(token, userAutoConfig.getUsername(), LoginFromTypeEnum.TEMPORARY.getValue(), RedisParam.SECONDS_OF_HOUR);
//            userRedis.setId("admin");
//            redisService.setHours(token, userRedis, RedisParam.TIME1);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
    }

    /**
     * 默认设置当前登录权限，如果没有登录情况
     */
    public static void doTokenForUserName(String userName, long seconds) {
        String token = UUID.randomUUID().toString().replaceAll("-", "");
        createUserToRedis(token, userName, LoginFromTypeEnum.TEMPORARY.getValue(), seconds);
    }

    /**
     * 缓存用户信息
     *
     * @param token
     * @param userRedis
     */
    public static void setUser(String token, UserRedis userRedis) {Assert.hasText(token, "token不能为空");
        Assert.notNull(userRedis, "userRedis不能为空");
        String username = userRedis.getUsername();
        String fromtype = userRedis.getFromtype();
        Assert.hasText(username, "用户帐号不能为空");
        Assert.hasText(fromtype, "登录类型不能为空");
        // 如果此账号已经登录过了 获取登录时的token,再根据此token 删除掉redis里面的信息
      /*  List<String> excludeUser = Lists.newArrayList();
        excludeUser.add("admin");
        if(!excludeUser.contains(username)){
            deleteHasLoginToken(username,fromtype);
        }*/
        setUserGroupSeconds(token, username, fromtype, RedisParam.SECONDS_OF_DAY);
        setDaysUser(token, userRedis);
    }

/*    private static void deleteHasLoginToken(String username, String fromtype) {
        Set<Object> keys = redisTemplate.opsForHash().keys(RedisParam.USER_GROUP_PREFIX + username + ":" + fromtype);

        // 每一次登录时都会去删除此用户的上一次登录的token,所以keys集合中只会有一个值
        String oldToken ="";
        if(!CollectionUtils.isEmpty(keys)){
            Iterator<Object> iterator = keys.iterator();
            while (iterator.hasNext()) {
                 oldToken = (String) iterator.next();
            }
        }
        deleteToken(oldToken);
    }*/

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userName
     */
    @Deprecated
    public static void setUser(String token, String userName) {
        Assert.hasText(token, "token不能为空");
        Assert.hasText(userName, "用户帐号不能为空");
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(userName);
        //预先缓存会话信息，给下面的feign请求使用
        setDaysUser(token, userRedis);
        Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
        MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                .map(Result::getResult)
                .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
        userRedis.setRealname(mdmUserRespVo.getFullName());
        userRedis.setPoscode(mdmUserRespVo.getPositionCode());
        userRedis.setPosname(mdmUserRespVo.getPositionName());
        userRedis.setOrgname(mdmUserRespVo.getOrgName());
        userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
        userRedis.setFromtype(LoginFromTypeEnum.TEMPORARY.getValue());
        setDaysUser(token, userRedis);
    }

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userName
     */
    public static void setUser(String token, String userName, String fromtype) {
        Assert.hasText(token, "token不能为空");
        Assert.hasText(userName, "用户帐号不能为空");
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(userName);
        //预先缓存会话信息，给下面的feign请求使用
        setDaysUser(token, userRedis);
        Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
        MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                .map(Result::getResult)
                .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
        userRedis.setRealname(mdmUserRespVo.getFullName());
        userRedis.setPoscode(mdmUserRespVo.getPositionCode());
        userRedis.setPosname(mdmUserRespVo.getPositionName());
        userRedis.setOrgname(mdmUserRespVo.getOrgName());
        userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
        userRedis.setFromtype(StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue());
        setDaysUser(token, userRedis);
    }

    /**
     * 强制用户生成登录信息
     *
     * @param userName
     * @param fromtype
     * @param seconds
     */
    public static UserRedis createUserToRedis(String token,String userName, String fromtype, long seconds) {
        UserRedis userRedis = null;
        Assert.hasText(userName, "userName不能为空");
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            map.put(GlobalParam.TOKEN, token);
            ThreadLocalUtil.stObj(map);

            userRedis = new UserRedis();
            userRedis.setUsername(userName);
            //预先缓存会话信息，给下面的feign请求使用
            setUserSeconds(token, userRedis, seconds);
            Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
            MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                    .map(Result::getResult)
                    .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
            userRedis.setRealname(mdmUserRespVo.getFullName());
            MdmPositionRespVo mdmPositionRespVo = mdmUserRespVo.getPositionList().stream()
                    .filter(o -> YesNoEnum.yesNoEnum.ONE.getValue().equals(o.getPrimaryFlag()))
                    .findFirst().orElse(null);
            userRedis.setPoscode(mdmPositionRespVo.getPositionCode());
            userRedis.setPosname(mdmPositionRespVo.getPositionName());
            userRedis.setOrgname(mdmPositionRespVo.getOrgName());
            userRedis.setOrgcode(mdmPositionRespVo.getOrgCode());
            userRedis.setUsertype(mdmUserRespVo.getUserType());
            userRedis.setFromtype(StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue());
            setUserSeconds(token, userRedis, seconds);
            setUserGroupSeconds(token, userName, fromtype, seconds);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
        return userRedis;
    }

    /**
     * 强制用户生成登录信息
     *
     * @param userName
     * @param fromtype
     * @param seconds
     */
    public static UserRedis createUserToRedis(String token,String userName, String posCode, String fromtype, long seconds) {
        UserRedis userRedis = null;
        Assert.hasText(userName, "userName不能为空");
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            map.put(GlobalParam.TOKEN, token);
            ThreadLocalUtil.stObj(map);

            userRedis = new UserRedis();
            userRedis.setUsername(userName);
            //预先缓存会话信息，给下面的feign请求使用
            setUserSeconds(token, userRedis, seconds);
            Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
            MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                    .map(Result::getResult)
                    .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
            userRedis.setRealname(mdmUserRespVo.getFullName());
            MdmPositionRespVo mdmPositionRespVo = mdmUserRespVo.getPositionList().stream()
                    .filter(o -> posCode.equals(o.getPositionCode()))
                    .findFirst().orElse(null);
            userRedis.setPoscode(mdmPositionRespVo.getPositionCode());
            userRedis.setPosname(mdmPositionRespVo.getPositionName());
            userRedis.setOrgname(mdmPositionRespVo.getOrgName());
            userRedis.setOrgcode(mdmPositionRespVo.getOrgCode());
            userRedis.setUsertype(mdmUserRespVo.getUserType());
            userRedis.setFromtype(StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue());
            setUserSeconds(token, userRedis, seconds);
            setUserGroupSeconds(token, userName, fromtype, seconds);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
        return userRedis;
    }

    /**
     * 强制用户生成登录信息
     *
     * @param token
     * @param userRedis
     * @param fromtype
     * @param seconds
     */
    public static void setUserToRedis(String token, UserRedis userRedis, String fromtype, long seconds) {
        Assert.hasText(token, "token不能为空");
        Assert.notNull(userRedis, "userRedis不能为空");
        fromtype = StringUtils.isNotEmpty(fromtype) ? fromtype : LoginFromTypeEnum.UNKNOWN.getValue();
        String username = userRedis.getUsername();
        Assert.hasText(username, "用户帐号不能为空");
        userRedis.setFromtype(fromtype);
        setUserGroupSeconds(token, username, fromtype, seconds);
        setUserSeconds(token, userRedis, seconds);
    }

    /**
     * 添加用户缓存信息到redis
     *
     * @param token
     * @param userRedis
     */
    protected static void setDaysUser(String token, UserRedis userRedis) {
        setUserSeconds(token, userRedis, tokenRenewalUtil.getTokenTime());
    }

    /**
     * 添加用户缓存信息到redis
     *
     * @param token     token
     * @param userRedis 缓存信息
     * @param seconds   时间（秒）
     */
    private static void setUserSeconds(String token, UserRedis userRedis, long seconds) {
        redisService.setSeconds(RedisBaseParam.TOKEN + token, userRedis, seconds);
        //设置续期时间
        tokenRenewalUtil.login(token,seconds);
    }

    /**
     * 缓存token到用户记录登录来源分组
     *
     * @param token    token
     * @param userName 用户名
     * @param fromtype 登录来源
     * @param seconds  时间（秒）
     */
    private static void setUserGroupSeconds(String token, String userName, String fromtype, long seconds) {
        long currentTimeMillis = System.currentTimeMillis();
        redisService.hset(RedisParam.USER_GROUP_PREFIX + userName + ":" + fromtype, token, currentTimeMillis + (1000L * seconds), seconds);
    }

    /**
     * 从redis移除token
     *
     * @param token
     */
    protected static void deleteToken(String token) {
        if (StringUtils.isNotEmpty(token)) {
//            deleteToken(Collections.singleton(token));
            Object o = redisService.get(RedisBaseParam.TOKEN + token);
            if (o != null) {
                redisService.del(RedisBaseParam.TOKEN + token);
                tokenRenewalUtil.loginOut(token);
                UserRedis userRedis = (UserRedis) o;
                if (StringUtils.isNotEmpty(userRedis.getUsername()) && StringUtils.isNotEmpty(userRedis.getFromtype())) {
                    Map<?, ?> hmget = redisService.hmget(RedisParam.USER_GROUP_PREFIX + userRedis.getUsername() + ":" + userRedis.getFromtype());
                    Set<String> groupTokenSet = new HashSet<>(16);
                    if (hmget != null) {
                        long currentTimeMillis = System.currentTimeMillis();
                        Map<String, Long> map = (Map<String, Long>) hmget;
                        groupTokenSet.addAll(map.entrySet().stream().filter(entry -> entry.getValue() < currentTimeMillis).map(Map.Entry::getKey).collect(Collectors.toSet()));
                    }
                    groupTokenSet.add(token);
                    redisService.hdel(RedisParam.USER_GROUP_PREFIX + userRedis.getUsername() + ":" + userRedis.getFromtype(), groupTokenSet.toArray());
                }
            }
        }
    }

    /**
     * 从redis批量移除token
     *
     * @param tokens
     */
    @Deprecated
    protected static void deleteToken(Set<String> tokens) {
        if (!tokens.isEmpty()) {
//            redisService.del(tokens.stream().map(x -> RedisParam.TOKEN + x).toArray(String[]::new));
            //如果需要同步删除token与用户和登录来源的缓存，请注释上一行，放开下面的代码
            tokens.forEach(token -> {
                Object o = redisService.get(RedisBaseParam.TOKEN + token);
                if (o != null) {
                    redisService.del(RedisBaseParam.TOKEN + token);
                    UserRedis userRedis = (UserRedis) o;
                    if (StringUtils.isNotEmpty(userRedis.getUsername()) && StringUtils.isNotEmpty(userRedis.getFromtype())) {
                        Map<?, ?> hmget = redisService.hmget(RedisParam.USER_GROUP_PREFIX + userRedis.getUsername() + ":" + userRedis.getFromtype());
                        Set<String> groupTokenSet = new HashSet<>(16);
                        if (hmget != null) {
                            long currentTimeMillis = System.currentTimeMillis();
                            Map<String, Long> map = (Map<String, Long>) hmget;
                            groupTokenSet.addAll(map.entrySet().stream().filter(entry -> entry.getValue() < currentTimeMillis).map(Map.Entry::getKey).collect(Collectors.toSet()));
                        }
                        groupTokenSet.add(token);
                        redisService.hdel(RedisParam.USER_GROUP_PREFIX + userRedis.getUsername() + ":" + userRedis.getFromtype(), groupTokenSet.toArray());
                    }
                }
            });
        }
    }

    /**
     * 指定用户全平台退出登录
     *
     * @param username 用户登录名
     */
    public static void deleteUser(String username) {
        deleteUser(Collections.singletonList(username));
    }

    /**
     * 指定用户指定平台退出登录
     *
     * @param username     用户登录名
     * @param fromtypeList 来源类型编码
     */
    public static void deleteUser(String username, List<String> fromtypeList) {
        if (StringUtils.isEmpty(username)) {
            return;
        }
        if (CollectionUtil.listEmpty(fromtypeList)) {
            return;
        }
        deleteUser(Collections.singletonList(username), fromtypeList);
    }

    /**
     * 指定用户全平台退出登录
     *
     * @param usernameList
     */
    public static void deleteUser(List<String> usernameList) {
        if (CollectionUtils.isEmpty(usernameList)) {
            return;
        }
        List<String> fromTypeList = new ArrayList<>();
        for (LoginFromTypeEnum loginFromTypeEnum :
                LoginFromTypeEnum.values()) {
            fromTypeList.add(loginFromTypeEnum.getValue());
        }
        deleteUser(usernameList, fromTypeList);
    }

    /**
     * 指定用户指定平台退出登录
     *
     * @param usernameList
     * @param fromtypeList
     */
    public static void deleteUser(List<String> usernameList, List<String> fromtypeList) {
        if (CollectionUtil.listEmpty(usernameList)) {
            return;
        }
        if (CollectionUtil.listEmpty(fromtypeList)) {
            return;
        }
        Set<String> tokenSet = new HashSet<>(16);
        Set<String> userGroupHashKeySet = new HashSet<>(16);
        for (String username :
                usernameList) {
            for (String fromType :
                    fromtypeList) {
                String hashKey = RedisParam.USER_GROUP_PREFIX + username + ":" + fromType;
                Map<?, ?> hmget = redisService.hmget(hashKey);
                if (hmget != null) {
                    Set<String> objects = (Set<String>) hmget.keySet();
                    tokenSet.addAll(objects);
                    userGroupHashKeySet.add(hashKey);
                }
            }
        }
        if (!tokenSet.isEmpty()) {
            redisService.del(tokenSet.stream().map(x -> RedisBaseParam.TOKEN + x).toArray(String[]::new));
            for(String token : tokenSet){
                tokenRenewalUtil.loginOut(token);
            }
            redisService.del(userGroupHashKeySet.toArray(new String[userGroupHashKeySet.size()]));
        }
    }

    /**
     * 当前登录用户全平台退出
     */
    public static void remove() {
        String username = Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getUsername)
                .orElse(null);
        if (!StringUtils.isEmpty(username)) {
            deleteUser(username);
        }
    }

    /**
     * 退出当前会话
     */
    public static void logout() {
        String token = getToken();
        deleteToken(token);
    }

    /**
     * 当前登录人退出指定平台
     */
    public static void logout(String fromtype) {
        String username = Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getUsername)
                .orElse(null);
        deleteUser(username, Collections.singletonList(fromtype));
    }

    /**
     * 处理多组织问题
     * @param orgCode
     * @return
     */
    public static List<String> handleOrgCodes(String orgCode){
        if(StringUtils.isEmpty(orgCode)){
            return null;
        }
        return Arrays.asList(orgCode.split(","));
    }
}
