package com.biz.crm.minbuynumofproduct.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.minbuynumofproduct.entity.MinBuyNumOfProductEntity;
import com.biz.crm.minbuynumofproduct.repositories.MinBuyNumOfProductRepositories;
import com.biz.crm.minbuynumofproduct.service.MinBuyNumOfProductService;
import com.biz.crm.minbuynumofproduct.utils.MinBuyNumOfProductUtil;
import com.biz.crm.nebular.dms.minbuynumofproduct.MinBuyNumOfProductVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.EsUtil;
import com.biz.crm.util.MinBuyNumOfProductConstant;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchTemplate;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * @Author: huojia
 * @DateTime: 2021/1/25 16:30
 **/
@ConditionalOnMissingBean(name = "MinBuyNumOfProductServiceExpandImpl")
@Service(value = "minBuyNumOfProductService")
public class MinBuyNumOfProductServiceImpl implements MinBuyNumOfProductService {

    @Resource
    private ElasticsearchTemplate elasticsearchTemplate;

    @Resource
    private RedissonUtil redissonUtil;

    @Resource
    private MinBuyNumOfProductRepositories minOrderRepositories;

    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 新增起订量
     * 1、验证集合是否为空，为空则直接返回
     * 2、验证参数，产品编码不能为空，集合内部要去重，onlyKey（产品+组织/客户/终端）不能重复
     * 3、模型转换，同时设置onlyKey
     * 4、存在则下一步，不存在则创建索引，然后直接新增和刷新，结束
     * 5、根据onlyKey删除对应起订量
     * 6、onlyKey（产品+组织/客户/终端）只能有一个起订量，根据onlyKey删除数据库中已存在起订量，然后批量插入
     *
     * @param minBuyNumOfProductVos
     * @author: huojia
     * @date: 2021/1/25 17:12
     * @return: void
     */
    @Transactional
    @Override
    public void add(ArrayList<MinBuyNumOfProductVo> minBuyNumOfProductVos) {
        // 1
        if (CollectionUtils.isEmpty(minBuyNumOfProductVos)) {
            return;
        }
        // 2
        MinBuyNumOfProductUtil.validParamBatch(minBuyNumOfProductVos);
        // 3
        List<MinBuyNumOfProductEntity> minBuyNumOfProductEntities = MinBuyNumOfProductUtil.assembleEntities(minBuyNumOfProductVos);
        // 4
        if (!EsUtil.indexExsit(elasticsearchTemplate, MinBuyNumOfProductEntity.class, redissonUtil)) {
            minOrderRepositories.saveAll(minBuyNumOfProductEntities);
            //elasticsearchTemplate.refresh(MinBuyNumOfProductConstant.ES_MIN_BUY_NUM_OF_PRODUCT_INDEX_NAME);
            return;
        }
        // 5
        List<String> onlyKeys = minBuyNumOfProductEntities.stream()
                .map(MinBuyNumOfProductEntity::getOnlyKey).collect(Collectors.toList());
        minOrderRepositories.deleteAllByOnlyKeyIn(onlyKeys);
        // 6
        minOrderRepositories.saveAll(minBuyNumOfProductEntities);
        //elasticsearchTemplate.refresh(MinBuyNumOfProductConstant.ES_MIN_BUY_NUM_OF_PRODUCT_INDEX_NAME);
        List<MinBuyNumOfProductVo> logVos = CrmBeanUtil.copyList(minBuyNumOfProductEntities, MinBuyNumOfProductVo.class);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (MinBuyNumOfProductVo vo : logVos) {
            crmLogSendUtil.sendForAdd(menuCodeObj.toString(),vo.getId(),vo.getId(),vo);
        }
    }

    /**
     * 修改
     * 1、为空则不作操作
     * 2、验证参数
     * 3、验证要修改的起订量是否还存在
     * 4、修改
     *
     * @param minBuyNumOfProductVo
     * @author: huojia
     * @date: 2021/1/26 14:55
     * @return: void
     */
    @Transactional
    @Override
    public void update(MinBuyNumOfProductVo minBuyNumOfProductVo) {
        // 1
        if (minBuyNumOfProductVo == null) {
            return;
        }
        if (!EsUtil.indexExsit(elasticsearchTemplate, MinBuyNumOfProductEntity.class, redissonUtil)) {
            return;
        }
        // 2
        MinBuyNumOfProductUtil.validParam(minBuyNumOfProductVo);
        // 3
        MinBuyNumOfProductEntity minBuyNumOfProductEntity = minOrderRepositories.findAllById(minBuyNumOfProductVo.getId());
        ValidateUtils.validate(minBuyNumOfProductEntity, "要修改的起订量不存在或者已被删除！");
        MinBuyNumOfProductEntity oldObject = minOrderRepositories.findAllById(minBuyNumOfProductVo.getId());
        // 4
        minOrderRepositories.save(CrmBeanUtil.copy(minBuyNumOfProductVo, MinBuyNumOfProductEntity.class));
        //elasticsearchTemplate.refresh(MinBuyNumOfProductConstant.ES_MIN_BUY_NUM_OF_PRODUCT_INDEX_NAME);
        MinBuyNumOfProductEntity newObject = minOrderRepositories.findAllById(minBuyNumOfProductVo.getId());
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newObject.getId(),newObject.getId(),oldObject,newObject);
    }

    /**
     * 分页搜索起订量
     * 1、es中不存在索引则直接返回空list
     * 2、构建查询条件
     * 3、查询
     *
     * @param minBuyNumOfProductVo
     * @author: huojia
     * @date: 2021/1/25 20:15
     * @return: java.lang.String
     */
    @Override
    public PageResult<MinBuyNumOfProductVo> list(MinBuyNumOfProductVo minBuyNumOfProductVo) {
        // 1
        PageResult<MinBuyNumOfProductVo> pageResult = PageResult.<MinBuyNumOfProductVo>builder()
                .data(new ArrayList<>())
                .count(Long.parseLong("0"))
                .build();
        if (!EsUtil.indexExsit(elasticsearchTemplate, MinBuyNumOfProductEntity.class, redissonUtil)) {
            return pageResult;
        }
        // 2
        BoolQueryBuilder boolQueryBuilder = MinBuyNumOfProductUtil.buildQuery(minBuyNumOfProductVo);
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MinBuyNumOfProductConstant.ES_MIN_BUY_NUM_OF_PRODUCT_INDEX_NAME)
                .withQuery(boolQueryBuilder)
                .withSort(SortBuilders.fieldSort("operationTimeNum").unmappedType("keyword").order(SortOrder.DESC))
                .withPageable(PageRequest.of(minBuyNumOfProductVo.getPageNum().intValue() - 1, minBuyNumOfProductVo.getPageSize()))
                .build();
        // 3
        Page<MinBuyNumOfProductEntity> scroll = elasticsearchTemplate.queryForPage(searchQuery, MinBuyNumOfProductEntity.class);
        List<MinBuyNumOfProductEntity> minBuyNumOfProductEntities = scroll.getContent();
        if (CollectionUtils.isEmpty(minBuyNumOfProductEntities)) {
            return pageResult;
        }
        return PageResult.<MinBuyNumOfProductVo>builder()
                .data(CrmBeanUtil.copyList(minBuyNumOfProductEntities, MinBuyNumOfProductVo.class))
                .count(scroll.getTotalElements())
                .build();
    }

    /**
     * 根据id批量修改启禁状态
     * 1、集合为空，直接返回
     * 2、根据id获取要修改数据
     * 3、修改状态
     * 4、重新插入
     *
     * @param ids
     * @param code
     * @author: huojia
     * @date: 2021/1/26 14:06
     * @return: void
     */
    @Transactional
    @Override
    public void updateEffectiveFlag(ArrayList<String> ids, Integer code) {
        // 1
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException("请至少选择一个要操作的起订量！");
        }
        if (!EsUtil.indexExsit(elasticsearchTemplate, MinBuyNumOfProductEntity.class, redissonUtil)) {
            return;
        }
        // 2
        List<MinBuyNumOfProductEntity> minBuyNumOfProductEntities = minOrderRepositories.findAllByIdIn(ids);
        if (CollectionUtils.isEmpty(minBuyNumOfProductEntities) || minBuyNumOfProductEntities.size() < ids.size()) {
            throw new BusinessException("部分起订量配置不存在或已被删除，请刷新后重试！");
        }
        // 3
        minBuyNumOfProductEntities = minBuyNumOfProductEntities.stream()
                .map(minBuyNumOfProductEntity -> minBuyNumOfProductEntity.setEffectiveFlag(code))
                .collect(Collectors.toList());
        // 4
        minOrderRepositories.saveAll(minBuyNumOfProductEntities);
        //elasticsearchTemplate.refresh(MinBuyNumOfProductConstant.ES_MIN_BUY_NUM_OF_PRODUCT_INDEX_NAME);
    }

    /**
     * 根据id批量删除
     *
     * @param ids
     * @author: huojia
     * @date: 2021/1/26 15:17
     * @return: void
     */
    @Transactional
    @Override
    public void delByIds(ArrayList<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException("请至少选择一个要删除的起订量！");
        }
        if (!EsUtil.indexExsit(elasticsearchTemplate, MinBuyNumOfProductEntity.class, redissonUtil)) {
            return;
        }
        List<MinBuyNumOfProductEntity> entities = minOrderRepositories.findAllByIdIn(ids);
        minOrderRepositories.deleteAllByIdIn(ids);
        //elasticsearchTemplate.refresh(MinBuyNumOfProductConstant.ES_MIN_BUY_NUM_OF_PRODUCT_INDEX_NAME);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        List<MinBuyNumOfProductVo> logVos = CrmBeanUtil.copyList(entities, MinBuyNumOfProductVo.class);
        for (MinBuyNumOfProductVo vo : logVos) {
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),vo.getId(),vo.getId(),vo);
        }
    }

    /**
     * 通过客户及其组织查询产品的最小起订量
     * 1、校验入参如果客户和组织编码同时为空，或者产品列表为空，返回空map
     * 2、根据客户编码查询并构建满足条件map
     * 3、根据组织编码查询并构建满足条件的map
     * 4、整合客户条件结果和组织条件结果（客户优先级高于组织）
     * @param cusCode 客户编码
     * @param orgCode 客户组织编码
     * @param productCodes 产品
     * @return
     */
    @Override
    public Map<String, MinBuyNumOfProductVo> findByCusCodeAndProductCodesToCus(String cusCode, String orgCode, List<String> productCodes) {
        //1、
        if(CollectionUtil.listEmpty(productCodes) || StringUtils.isEmpty(cusCode) && StringUtils.isEmpty(orgCode)) {
            return Maps.newHashMap();
        }
        Map<String, MinBuyNumOfProductVo> result = Maps.newHashMap();
        Map<String, MinBuyNumOfProductVo> cusMap = Maps.newHashMap();
        Map<String, MinBuyNumOfProductVo> orgMap = Maps.newHashMap();
        //2、
        if(StringUtils.isNotEmpty(cusCode)) {
            List<MinBuyNumOfProductEntity> cusEntities = this.minOrderRepositories.findByCusCodeAndEffectiveFlag(cusCode, YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
            cusMap.putAll(this.buildProductMap(CrmBeanUtil.copyList(cusEntities, MinBuyNumOfProductVo.class), productCodes));
            result.putAll(cusMap);
        }
        //3、
        if (StringUtils.isNotEmpty(orgCode)) {
            List<MinBuyNumOfProductEntity> orgEntities = this.minOrderRepositories.findByOrgCodeAndEffectiveFlag(orgCode, YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
            orgMap.putAll(this.buildProductMap(CrmBeanUtil.copyList(orgEntities, MinBuyNumOfProductVo.class), productCodes));
        }
        //4、
        List<String> nonKeys = productCodes.stream().filter(k -> !cusMap.keySet().contains(k)).collect(Collectors.toList());
        if(CollectionUtil.listNotEmpty(nonKeys)) {
            nonKeys.forEach(k -> result.put(k, orgMap.get(k)));
        }
        return result;
    }


    /**
     * 根据条件产品维度构建map
     * 1、校验入参
     * 2、构建全量商品map
     * 3、根据条件商品编码列表筛选结果
     * @param vos
     * @param productCodes
     * @return
     */
    private Map<String, MinBuyNumOfProductVo> buildProductMap(List<MinBuyNumOfProductVo> vos, List<String> productCodes) {
        //1、
        if(CollectionUtil.listEmpty(vos) || CollectionUtil.listEmpty(productCodes)) {
            return Maps.newHashMap();
        }
        //2、
        Map<String, MinBuyNumOfProductVo> voMap = vos.stream().collect(Collectors.toMap(MinBuyNumOfProductVo::getProductCode, a -> a, (a, b) -> a));
        //3、
        List<String> keys = voMap.keySet().stream().filter(k -> !productCodes.contains(k)).collect(Collectors.toList());
        keys.forEach(k -> voMap.remove(k));
        return voMap;
    }
}