package com.biz.crm.notice.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.dms.NoticeEunm;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.nebular.dms.notice.ListReq;
import com.biz.crm.nebular.dms.notice.ListRes;
import com.biz.crm.nebular.dms.notice.NoticeAreaVo;
import com.biz.crm.nebular.dms.notice.NoticeFileVo;
import com.biz.crm.nebular.dms.notice.NoticeVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.notice.entity.NoticeEntity;
import com.biz.crm.notice.mapper.NoticeMapper;
import com.biz.crm.notice.service.NoticeAreaService;
import com.biz.crm.notice.service.NoticeFileService;
import com.biz.crm.notice.service.NoticeService;
import com.biz.crm.notice.utils.NoticeAreaUtil;
import com.biz.crm.notice.utils.NoticeUtil;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/9/11 16:43
 **/
@ConditionalOnMissingBean(name = "noticeServiceExpandImpl")
@Service(value = "noticeService")
public class NoticeServiceImpl<M extends BaseMapper<T>,T>
        extends ServiceImpl<NoticeMapper, NoticeEntity>
        implements NoticeService {

    @Resource
    private NoticeMapper noticeMapper;

    @Resource
    private NoticeFileService noticeFileService;

    @Resource
    private NoticeAreaService noticeAreaService;

    @Resource
    private MdmOrgFeign mdmOrgFeign;

    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 1、验证主表参数
     * 2、组装保存
     * 3、保存文件
     * 4、保存区域
     * @param noticeVo
     */
    @Transactional
    @Override
    public void add(NoticeVo noticeVo) {
        //1
        ValidateUtils.validate(noticeVo.getTitle(),"标题不能为空!");
        ValidateUtils.validate(noticeVo.getType(),"请指定类型!");
        ValidateUtils.validate(noticeVo.getStartTime(),"请指定有效时间段!");
        ValidateUtils.validate(noticeVo.getStartTime(),"请指定有效时间段!");

        //2
        NoticeEntity entity = new NoticeEntity();
        BeanUtils.copyProperties(noticeVo,entity);
        UserRedis userRedis = UserUtils.getUser();
        entity.setPublishOrgCode(userRedis.getOrgcode());
        entity.setPublishOrgName(userRedis.getOrgname());
        entity.setVisitNum(0);
        noticeMapper.insert(entity);

        //3
        noticeFileService.replace(entity,noticeVo.getFiles());

        //4
        noticeAreaService.replace(entity,NoticeAreaUtil.addAllAreas(noticeVo));
        //5、发送vo用于记录日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), entity.getId(),noticeVo.getTitle(),noticeVo);
    }


    /**
     * 1、验证主表参数
     * 2、组装保存
     * 3、保存文件
     * 4、保存区域
     * @param noticeVo
     */
    @Transactional
    @Override
    public void edit(NoticeVo noticeVo) {
        //1
        ValidateUtils.validate(noticeVo.getId(),"请指定要编辑的资料!");
        ValidateUtils.validate(noticeVo.getTitle(),"标题不能为空!");
        ValidateUtils.validate(noticeVo.getType(),"请指定类型!");
        ValidateUtils.validate(noticeVo.getStartTime(),"请指定有效时间段!");
        ValidateUtils.validate(noticeVo.getStartTime(),"请指定有效时间段!");
        NoticeEntity entity = noticeMapper.selectById(noticeVo.getId());
        ValidateUtils.validate(entity,"您要编辑的资料不存在或者已经被删除!");

        //2
        NoticeVo oldObject = this.findById(noticeVo.getId());
        BeanUtils.copyProperties(noticeVo,entity);
        UserRedis userRedis = UserUtils.getUser();
        entity.setPublishOrgCode(userRedis.getOrgcode());
        entity.setPublishOrgName(userRedis.getOrgname());
        noticeMapper.updateById(entity);

        //3
        noticeFileService.replace(entity,noticeVo.getFiles());

        //4
        noticeAreaService.replace(entity,NoticeAreaUtil.addAllAreas(noticeVo));

        //获取修改后的对象
        NoticeVo newObject = this.findById(noticeVo.getId());
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                entity.getId(),entity.getTitle(),oldObject,newObject);
    }

    /**
     * 1、先查询主信息并且组装
     * 2、查询文件信息并且组装
     * 3、查询范围信息并且组装
     * @param id
     * @return
     */
    @Override
    @CrmDictMethod
    public NoticeVo findById(String id) {
        NoticeVo vo = null;
        //1
        ValidateUtils.validate(id,"请指定需要查询的资料！");
        NoticeEntity entity = noticeMapper.selectById(id);
        if(null == entity){
            return vo;
        }
        vo = new NoticeVo();
        BeanUtils.copyProperties(entity,vo);

        //2
        vo.setFiles(noticeFileService.findByNoticeId(id));

        //3
        Map<String, List<NoticeAreaVo>> areaMap = NoticeAreaUtil.distArea(noticeAreaService.findByNoticeId(id));
        vo.setOrgs(areaMap.get("org"));
        vo.setNonOrgs(areaMap.get("nonOrg"));
        vo.setCustomers(areaMap.get("customer"));
        vo.setNonCustomers(areaMap.get("nonCustomer"));

        vo.setNoticeState(NoticeUtil.transState(vo.getStartTime(),vo.getEndTime()));
        return vo;
    }


    @CrmDictMethod
    @Override
    public PageResult<NoticeVo> list(NoticeVo noticeVo) {
         QueryWrapper<NoticeVo> wrapper = Wrappers.<NoticeVo>query()
                .like(!StringUtils.isEmpty(noticeVo.getTitle()), "title", noticeVo.getTitle())
                .eq(null != noticeVo.getType(), "type", noticeVo.getType())
                .eq(!StringUtils.isEmpty(noticeVo.getPublishOrgCode()), "publish_org_code", noticeVo.getPublishOrgCode())
                 .like(!StringUtils.isEmpty(noticeVo.getPublishOrgName()), "publish_org_name", noticeVo.getPublishOrgName());
         if(null != noticeVo.getNoticeState()){
             String now = DateUtil.date2Str(DateUtil.getDate(), DateUtil.datetimeFormat);
             if(noticeVo.getNoticeState().intValue() == NoticeEunm.NoticeStateEunm.TOBEGIN.getCode().intValue()){
                 wrapper.gt("start_time", now);
             }else if(noticeVo.getNoticeState().intValue() == NoticeEunm.NoticeStateEunm.ONGOING.getCode().intValue()){
                 wrapper.lt("start_time", now);
                 wrapper.gt("end_time", now);
             }else {
                 wrapper.lt("end_time", now);
             }
         }
        wrapper.orderByDesc("create_date", "create_date_second");
        Page<NoticeVo> page = PageUtil.buildPage(noticeVo.getPageNum(), noticeVo.getPageSize());
        List<NoticeVo> list = noticeMapper.list(page, wrapper);
        if(!CollectionUtils.isEmpty(list)){
            for(NoticeVo vo : list){
                vo.setNoticeState(NoticeUtil.transState(vo.getStartTime(),vo.getEndTime()));
            }
        }
        return PageResult.<NoticeVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @CrmDictMethod
    @Override
    public PageResult<ListRes> listByUser(ListReq listReq,List<String> orgCodes) {
        Page<NoticeVo> page = PageUtil.buildPage(listReq.getPageNum(), listReq.getPageSize());
        //查找所有上级组织
        List<String> orgList = this.findParentOrgs(orgCodes);
        if(!CollectionUtil.listEmpty(orgCodes) && !CollectionUtil.listEmpty(orgList)) {
            orgList.addAll(orgCodes);
        }
        // 添加msgState筛选字段，并且只查询已开始的公告
        List<ListRes> list = noticeMapper.listByUser(page,listReq, DateUtil.getDate(DateUtil.DEFAULT_DATE_ALL_PATTERN)
        , YesNoEnum.YesNoCodeNumberEnum.NO.getCode(),YesNoEnum.YesNoCodeNumberEnum.YES.getCode(),
                YesNoEnum.YesNoCodeNumberEnum.NO.getCode(),orgList);
        if(!CollectionUtils.isEmpty(list)){
            Map<String, List<NoticeFileVo>> fileMap = noticeFileService.findByNoticeIds(list.stream().map(NoticeVo::getId).collect(Collectors.toList()));
            for(ListRes listRes : list){
                listRes.setNoticeState(NoticeUtil.transState(listRes.getStartTime(), listRes.getEndTime()));
                listRes.setMsgState(StringUtils.isEmpty(listRes.getNoticeLogId())?0:1);
                listRes.setFiles(fileMap.get(listRes.getId()));
            }
        }
        return PageResult.<ListRes>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询所有父级组织
     * @param orgCodes
     * @return
     */
    private List<String> findParentOrgs(List<String> orgCodes) {
        if(CollectionUtil.listEmpty(orgCodes)) {
            return Lists.newArrayList();
        }
        List<String> orgList = Lists.newArrayList();
        MdmOrgReqVo mdmOrgReqVo = new MdmOrgReqVo();
        orgCodes.forEach(code -> {
            mdmOrgReqVo.setOrgCode(code);
            List<MdmOrgRespVo> parentList = ApiResultUtil.objResult(mdmOrgFeign.findAllParentOrgList(mdmOrgReqVo), true);
            if(!CollectionUtil.listEmpty(parentList)) {
                parentList.forEach(parent -> orgList.add(parent.getOrgCode()));
            }
        });
        return orgList;
    }

    /**
     * 1、删除公告主体信息
     * 2、删除文件和区域
     * @param ids
     */
    @Transactional
    @Override
    public void delByIds(ArrayList<String> ids) {
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        List<NoticeVo> noticeVos = this.findByIds(ids);
        //校验公告是否有效
        noticeVos.forEach(vo -> this.validateTime(vo));
        //1
        noticeMapper.deleteBatchIds(ids);

        //2
        noticeFileService.delByNoticeIds(ids);
        noticeAreaService.delByNoticeIds(ids);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (NoticeVo vo : noticeVos) {
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),
                    vo.getId(),vo.getTitle(),vo);
        }
    }

    /**
     * 校验：进行中和已结束的公告不可删除
     * @param vo
     */
    private void validateTime(NoticeVo vo) {
        ValidateUtils.validate(vo, "公告不能为空");
        if(StringUtils.isEmpty(vo.getStartTime()) || StringUtils.isEmpty(vo.getEndTime())) {
            return;
        }
        try {
            Date startTime = DateUtils.parseDate(vo.getStartTime(), DateUtil.DEFAULT_DAY_PATTERN, DateUtil.DEFAULT_DATE_ALL_PATTERN);
            ValidateUtils.isTrue(TimeUtil.bigThan(startTime, new Date()), "进行中和已结束的公告【%s】不可删除", vo.getTitle());
        } catch (ParseException e) {
            throw new BusinessException(e.getMessage(), e);
        }
    }

    /**
     * 1、删除公告主体信息
     * 2、删除文件和范围
     * @param noticeVo
     */
    @Transactional
    @Override
    public void delByParam(NoticeVo noticeVo) {
        if(null == noticeVo){
            throw new BusinessException("请指定查询条件!");
        }
        QueryWrapper<NoticeEntity> wrapper = Wrappers.<NoticeEntity>query()
                .like(!StringUtils.isEmpty(noticeVo.getTitle()), "title", noticeVo.getTitle())
                .eq(null != noticeVo.getType(), "type", noticeVo.getType())
                .eq(!StringUtils.isEmpty(noticeVo.getPublishOrgCode()), "publish_org_code", noticeVo.getPublishOrgCode());
        List<NoticeEntity> entities = noticeMapper.selectList(wrapper);
        if(CollectionUtils.isEmpty(entities)){
            return ;
        }
        List<String> ids = entities.stream()
                .map(NoticeEntity::getId)
                .collect(Collectors.toList());
        //1
        noticeMapper.delete(wrapper);

        //2
        noticeFileService.delByNoticeIds(ids);
        noticeAreaService.delByNoticeIds(ids);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for(NoticeEntity entity : entities){
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),
                    entity.getId(),entity.getTitle(),entity);
        }
    }

    /**
     * 1、验证
     * 2、修改保存
     * @param noticeId
     */
    @Override
    @Klock(keys = {DmsConstant.NOTICE_LOG_LOCK ,"#noticeId"},waitTime = 10,leaseTime = 5)
    @Transactional
    public void updateVisitNum(String noticeId) {
        //1
        NoticeEntity entity = noticeMapper.selectById(noticeId);
        ValidateUtils.validate(entity,"该公告已被删除!");

        //2
        entity.setVisitNum(entity.getVisitNum() + 1);
        noticeMapper.updateById(entity);
    }

    /**
     * 根据id列表查询
     * @param ids
     * @return
     */
    @Override
    public List<NoticeVo> findByIds(List<String> ids) {
        if(CollectionUtil.listEmpty(ids)) {
            return Lists.newArrayList();
        }
        List<NoticeEntity> entities = this.noticeMapper.selectList(Wrappers.<NoticeEntity>query()
        .in("id", ids));
        return CrmBeanUtil.copyList(entities, NoticeVo.class);
    }

}
