package com.biz.crm.promotion.service.component.validator.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.dms.PromotionPolicyEunm;
import com.biz.crm.nebular.dms.promotion.*;
import com.biz.crm.promotion.entity.PromotionPolicyEntity;
import com.biz.crm.promotion.entity.PromotionPolicyTemplateEntity;
import com.biz.crm.promotion.mapper.PromotionPolicyTemplateMapper;
import com.biz.crm.promotion.service.component.ProductPriceMatcher;
import com.biz.crm.promotion.service.component.resolver.ladder.LadderResolverHelper;
import com.biz.crm.promotion.service.component.validator.PromotionPolicyEditValidator;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmDateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * 政策编辑费率阶梯配置校验
 *  @author: luoqi
 *  @Date: 2021-1-6 16:47
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
@Component("promotionPolicyEditFeeRateLadderValidator")
@Order(2)
public class PromotionPolicyEditFeeRateLadderValidator implements PromotionPolicyEditValidator {
    @Resource
    private ProductPriceMatcher productPriceMatcher;
    @Resource
    private LadderResolverHelper ladderResolverHelper;

    /**
     * 校验
     *
     * @param vo
     */
    @Override
    public void validate(PromotionPolicyEditReqVo vo) throws BusinessException {
        //费率
        BigDecimal feeRate = vo.getFeeRate();
        if(null == feeRate){
            return;
        }
        //条件规则
        PromotionPolicyRuleEditVo ruleEditVo = this.getConditionRuleConfigVo(vo);
        if(null == ruleEditVo){
            return;
        }


        //解析前端提交上来的本品和赠品，并询价
        FeeRateLadderValidatorContext context = this.priceMatch(vo);

        if(null == context){
            return;
        }
        //商品价格
        Map<String, BigDecimal> priceMap = context.getPriceMap();
        //本品编码
        Set<String> productCodesCurrentY = context.getProductCodesCurrentY();
        //赠品编码
        Set<String> productCodesCurrentN = context.getProductCodesCurrentN();

        List<PromotionRuleEditVo.ControlRow> controlRows = ruleEditVo.getControlRows();
        for (PromotionRuleEditVo.ControlRow controlRow : controlRows) {
            BigDecimal valueAsProductCurrentY = this.ladderResolverHelper.valueAsProductCurrentY(ruleEditVo, controlRow);
            BigDecimal valueAsProductCurrentN = this.ladderResolverHelper.valueAsProductCurrentN(ruleEditVo, controlRow);
            BigDecimal amountAsCurrentY = priceMap.get(productCodesCurrentY.iterator().next()).multiply(valueAsProductCurrentY);
            BigDecimal amountAsCurrentN = priceMap.get(productCodesCurrentN.iterator().next()).multiply(valueAsProductCurrentN);
            int compareTo = amountAsCurrentN.divide(amountAsCurrentY, 4 , RoundingMode.HALF_UP).compareTo(feeRate.divide(new BigDecimal(100)));

            if(compareTo > 0){
                throw new BusinessException("当前促销规则费率大于政策最大可用费率（不可设置）。");
            }
        }



    }

    /**
     * 解析前端提交上来的本品和赠品，并询价
     * @param vo
     * @return
     */
    private FeeRateLadderValidatorContext priceMatch(PromotionPolicyEditReqVo vo){
        //校验必要的参数
        this.check(vo);
        Set<String> productCodesCurrentY = Sets.newHashSet()//本品编码
                , productCodesCurrentN = Sets.newHashSet();//赠品编码
        //解析收集前端提交的本品和赠品编码
        this.resolveProductCodes(vo, productCodesCurrentY, productCodesCurrentN);
        //没有配置商品 返回空
        if(CollectionUtils.isEmpty(productCodesCurrentY) || CollectionUtils.isEmpty(productCodesCurrentN)){
            return null;
        }
        //全部商品编码
        Set<String> productCodes = Sets.newHashSet();
        productCodes.addAll(productCodesCurrentY);
        productCodes.addAll(productCodesCurrentN);

        //组装询价参数
        ProductPriceMatcher.PriceMatchParam priceMatchParam = new ProductPriceMatcher.PriceMatchParam();
        priceMatchParam.setChanelCode(vo.getChanel());
        priceMatchParam.setSaleCompanyCode(vo.getSaleCompanyCode());
        Map<String, BigDecimal> priceMap = Maps.newHashMap();
        //开始询价
        for (String productCode : productCodes) {
            priceMatchParam.setProductCode(productCode);
            //调用项目下的询价服务
            BigDecimal price = this.productPriceMatcher.priceMatch(priceMatchParam);
            if(null == price){
                throw new BusinessException("项目询价服务失败，请重试！");
            }
            priceMap.put(productCode, price);
        }
        FeeRateLadderValidatorContext context = new FeeRateLadderValidatorContext();
        context.setPriceMap(priceMap);
        context.setProductCodesCurrentN(productCodesCurrentN);
        context.setProductCodesCurrentY(productCodesCurrentY);
        return context;
    }

    /**
     * 解析收集前端提交的本品和赠品编码
     * @param vo
     * @param productCodesCurrentY
     * @param productCodesCurrentN
     */
    private void resolveProductCodes(PromotionPolicyEditReqVo vo, Set<String> productCodesCurrentY, Set<String> productCodesCurrentN){
        //前端配置的促销商品
        PromotionPolicyProduct promotionPolicyProduct = vo.getPromotionPolicyProduct();
        //没有配置商品 返回空
        if(null == promotionPolicyProduct){
            return;
        }

        //前端配置的本品
        List<PromotionPolicyProductEditVo> productsCurrentY = promotionPolicyProduct.getProductsCurrentY();
        //前端配置的赠品
        List<PromotionPolicyProductEditVo> productsCurrentN = promotionPolicyProduct.getProductsCurrentN();
        //没有配置本品 返回空
        if(CollectionUtils.isEmpty(productsCurrentY)){
            return;
        }


        for (PromotionPolicyProductEditVo promotionPolicyProductEditVo : productsCurrentY) {
            //本品
            productCodesCurrentY.add(promotionPolicyProductEditVo.getProductCode());
            //本品赠品一一对应时，赠品就在本品对象上
            if(StringUtils.isNotBlank(promotionPolicyProductEditVo.getProductCodeGift())){
                productCodesCurrentN.add(promotionPolicyProductEditVo.getProductCodeGift());
            }
        }
        if(productsCurrentY.size() > 1){
            throw new BusinessException("最多只能配置一个本品！");
        }
        if(!CollectionUtils.isEmpty(productsCurrentN)){
            if(productsCurrentN.size() > 1){
                throw new BusinessException("最多只能配置一个赠品！");
            }
            for (PromotionPolicyProductEditVo promotionPolicyProductEditVo : productsCurrentN) {
                productCodesCurrentN.add(promotionPolicyProductEditVo.getProductCode());
            }

        }

    }

    private void check(PromotionPolicyEditReqVo vo){
        if(StringUtils.isBlank(vo.getSaleCompanyCode())){
            throw new BusinessException("销售公司不能为空！");
        }
        if(StringUtils.isBlank(vo.getChanel())){
            throw new BusinessException("渠道不能为空！");
        }
    }

    /**
     * 条件规则
     * @param vo
     * @return
     */
    private PromotionPolicyRuleEditVo getConditionRuleConfigVo(PromotionPolicyEditReqVo vo){
        //前端提交的规则数据
        PromotionPolicyRule promotionPolicyRule = vo.getPromotionPolicyRule();
        if(null == promotionPolicyRule){
            return null;
        }
        //条件规则
        List<PromotionPolicyRuleEditVo> rulesAsCondition = promotionPolicyRule.getRulesAsCondition();
        if(CollectionUtils.isEmpty(rulesAsCondition)){
            return null;
        }
        PromotionPolicyRuleEditVo ruleEditVo = rulesAsCondition.get(0);
        if(null == ruleEditVo){
            return null;
        }
        return ruleEditVo;
    }

    @Data
    public static class FeeRateLadderValidatorContext{
        //商品价格
        private Map<String, BigDecimal> priceMap;
        //本品编码
        private Set<String> productCodesCurrentY;
        //赠品编码
        private Set<String> productCodesCurrentN;
    }
}
