package com.biz.crm.promotion.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.dms.PromotionPolicyEunm;
import com.biz.crm.mdm.customer.MdmCustomerROrgFeign;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.mdm.product.MdmProductFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.dms.promotion.*;
import com.biz.crm.nebular.dms.promotion.param.QueryProductPolicyParamVo;
import com.biz.crm.nebular.dms.promotion.policy.resp.CalculateRuleResponse;
import com.biz.crm.nebular.dms.promotion.policy.resp.PromotionPolicyInfoDataVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerROrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.promotion.entity.*;
import com.biz.crm.promotion.entity.resp.PromotionPolicyPageResp;
import com.biz.crm.promotion.mapper.PromotionPolicyMapper;
import com.biz.crm.promotion.service.*;
import com.biz.crm.promotion.service.component.RuleExecuteParam;
import com.biz.crm.promotion.service.component.RuleExecutor;
import com.biz.crm.promotion.service.component.filter.AbstractFilter;
import com.biz.crm.promotion.service.component.filter.AbstractProductFilter;
import com.biz.crm.promotion.service.component.validator.PromotionPolicyEditValidator;
import com.biz.crm.promotion.util.PromotionUtil;
import com.biz.crm.util.*;
import com.biz.crm.util.dms.PromotionPolicyUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @author: luoqi
 * @Date: 2020-10-15 16:50
 * @version: V1.0
 * @Description:
 */
@Transactional
@ConditionalOnMissingBean(name = "promotionPolicyServiceExpandImpl")
@Service("promotionPolicyService")
public class PromotionPolicyServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<PromotionPolicyMapper, PromotionPolicyEntity> implements PromotionPolicyService {

    @Resource
    private PromotionPolicyMapper promotionPolicyMapper;

    @Resource
    private PromotionPolicyRuleService promotionPolicyRuleService;
    @Resource
    private PromotionRuleService promotionRuleService;
    @Resource
    private PromotionPolicyScopeService promotionPolicyScopeService;
    @Resource
    private PromotionPolicyProductService promotionPolicyProductService;
    @Resource
    private RocketMQProducer rocketMQProducer;
    @Resource
    private MdmProductFeign mdmProductFeign;
    @Resource
    private MdmCustomerROrgFeign mdmCustomerROrgFeign;
    @Resource
    private MdmOrgFeign mdmOrgFeign;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Resource(name = "defaultRuleExecutor")
    private RuleExecutor defaultRuleExecutor;
    @Resource
    private MdmTerminalFeign mdmTerminalFeign;
    @Resource
    private PromotionPolicyTemplateService promotionPolicyTemplateService;
    /**
     * 项目上可使用 @PostConstruct 注解移除不需要的过滤器
     * @PostConstruct
     *     protected void removeFilter(){
     *         if(CollectionUtils.isEmpty(this.promotionPolicyFilters)){
     *             return;
     *         }
     *         this.promotionPolicyFilters.remove(0);
     *     }
     */
    @Resource
    private List<AbstractFilter> abstractFilters;
    /**
     * 商品查询相关filter
     */
    @Resource
    private List<AbstractProductFilter> abstractProductFilters;
    /**
     * 项目上可使用 @PostConstruct 注解移除不需要的校验逻辑
     * @PostConstruct
     *     protected void removeValidator(){
     *         if(CollectionUtils.isEmpty(this.promotionPolicyEditValidators)){
     *             return;
     *         }
     *         this.promotionPolicyEditValidators.remove(0);
     *     }
     */
    @Resource
    private List<PromotionPolicyEditValidator> promotionPolicyEditValidators;



    @Override
    public void addOrUpdate(PromotionPolicyEditReqVo vo) {
        //如果新增 将先自动生成促销编码
        if(StringUtils.isEmpty(vo.getId())){
            vo.setPromotionPolicyCode(CodeUtil.createOneCode(CrmCodeRuleConstants.DMS_PROMOTION_CODE));
        }
        this.checkParams(vo);
        if (StringUtils.isBlank(vo.getId())) {
            //促销编码自动生成了,不用验证了
//            QueryWrapper<PromotionPolicyEntity> wrapper = new QueryWrapper<>();
//            wrapper.eq("promotion_policy_code", vo.getPromotionPolicyCode());
//            if (this.promotionPolicyMapper.selectCount(wrapper) > 0) {
//                throw new BusinessException("已存在的促销编码[" + vo.getPromotionPolicyCode() + "]。");
//            }
            PromotionPolicyEntity temp = CrmBeanUtil.copy(vo, PromotionPolicyEntity.class);
            this.promotionPolicyMapper.insert(temp);
            crmLogSendUtil.sendForAdd(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), vo.getId(),null,vo);

            vo.setId(temp.getId());
        } else {
            PromotionPolicyEntity oldEntity = this.promotionPolicyMapper.selectById(vo.getId());
            if (null == oldEntity) {
                throw new BusinessException("政策数据不存在,ID=[" + vo.getId() + "]");
            }
            //a) 待执行时可以编辑除编码外的所有数据
            vo.setPromotionPolicyCode(oldEntity.getPromotionPolicyCode());
            PromotionPolicyVo oldobj = CrmBeanUtil.copy(oldEntity,PromotionPolicyVo.class);

            this.doUpdate(oldEntity, vo);
            PromotionPolicyVo newEntity = this.findById(vo.getId());
            PromotionPolicyVo newobj = CrmBeanUtil.copy(newEntity,PromotionPolicyVo.class);

            crmLogSendUtil.sendForUpdate(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), vo.getId(),null,oldobj,newobj);

        }
        this.promotionPolicyRuleService.refreshRule(vo);
        this.promotionPolicyScopeService.refreshScope(vo);
        this.promotionPolicyProductService.refreshProduct(vo);
        //通知可购清单更新
        this.countBeginAndEndAffectedCustoms(DateUtil.getDate(DateUtil.DEFAULT_DATE_FORMAT), Lists.newArrayList(vo.getId()));
        //删除缓存
        PromotionUtil.deleteCaches(Lists.newArrayList(vo.getPromotionPolicyCode()));
    }

    /**
     * 政策编辑校验
     * @param vo
     */
    private void checkParams(PromotionPolicyEditReqVo vo){
        if(CollectionUtils.isEmpty(this.promotionPolicyEditValidators)){
            return;
        }
        for (PromotionPolicyEditValidator promotionPolicyEditValidator : this.promotionPolicyEditValidators) {
            promotionPolicyEditValidator.validate(vo);
        }
    }

    /**
     * 计算给定时间生效和失效政策影响到的客户列表
     *
     * @param date
     * @param promotionIds
     */
    @Override
    public List<String> countBeginAndEndAffectedCustoms(String date, List<String> promotionIds) {
        if(CollectionUtils.isEmpty(promotionIds)){
            return Lists.newArrayList();
        }
        List<String> custCodes = this.promotionPolicyScopeService.countCustCodesByPromotionPolicyId(Sets.newHashSet(promotionIds));
        if (!CollectionUtils.isEmpty(custCodes)) {
            //发送mq
            RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
            rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(custCodes));
            rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.CUS_AVAILBLELIST_REFRSH_PROMOTION);
            rocketMQProducer.convertAndSend(rocketMQMessageBody);
        }
        return custCodes;
    }

    /**
     * 查询给定的客户在给定的商品下能够享受的促销政策
     * <p>
     * 1.查询客户属于哪些组织
     * 2.查询客户属于哪些渠道
     * 3.
     *
     * @param custCode
     * @param productCode
     * @author: luoqi
     * @Date: 2020-10-21 11:20
     * @version: V1.0
     * @Description:
     */
    @Override
    public List<PromotionPolicyInfoDataVo> loadPromotionPolicyInfoByCustAndProduct(String custCode, String productCode) {
        if (StringUtils.isBlank(custCode) || StringUtils.isBlank(productCode)) {
            throw new BusinessException("请指定客户和商品编码");
        }
        QueryProductPolicyParamVo productPolicyParamVo = new QueryProductPolicyParamVo();
        productPolicyParamVo.setCustCode(custCode);
        productPolicyParamVo.setProductCode(productCode);
        return this.loadPromotionPolicyInfos(productPolicyParamVo);
    }

    /**
     * 根据条件查询促销政策
     * @param productPolicyParamVo
     * @return
     */
    private List<PromotionPolicyInfoDataVo> loadPromotionPolicyInfos(QueryProductPolicyParamVo productPolicyParamVo) {

        String custCode = productPolicyParamVo.getCustCode(), productCode = productPolicyParamVo.getProductCode();
        //组织编码
        Set<String> orgCodes = this.buildOrgCodes(custCode, productPolicyParamVo.getOrgCode());
        Set<String> terminalOrgCode = this.buildOrgCodesByTerminal(productPolicyParamVo.getTerminalCode(), productPolicyParamVo.getOrgCode());
        Set<String> promotionPolicyIdAsProduct = promotionPolicyProductService.selectPromotionPolicyIdByProductCode(productCode);
        Set<String> promotionPolicyIdAsScope = promotionPolicyScopeService.selectPromotionPolicyIdByScopeCode(custCode, orgCodes);
        Set<String> promotionPolicyIdAsScopeTerminal = promotionPolicyScopeService.selectPromotionPolicyIdByTerminal(productPolicyParamVo.getTerminalCode(), terminalOrgCode);
        promotionPolicyIdAsScope.addAll(promotionPolicyIdAsScopeTerminal);
        List<String> merged = this.merge(promotionPolicyIdAsProduct, promotionPolicyIdAsScope);
        if (CollectionUtils.isEmpty(merged)) {
            return Lists.newArrayList();
        }
        return this.selectActivities(CrmBeanUtil.copy(productPolicyParamVo, PromotionPolicyEntity.class), merged);
    }

    /**
     * 根据终端编码构建组织
     * @param terminalCode
     * @param orgCode
     * @return
     */
    private Set<String> buildOrgCodesByTerminal(String terminalCode, String orgCode) {
//        if(StringUtils.isNotBlank(orgCode) || StringUtils.isBlank(terminalCode)) {
//            return Sets.newHashSet();
//        }
        //已经指定了组织编码
        if(StringUtils.isNotBlank(orgCode)){
            //查询组织的上级组织
            return this.loadParentOrgCodes(Sets.newHashSet(orgCode));
        }
        //组织编码为空，尝试查询客户对应的组织
        if(StringUtils.isBlank(terminalCode)){
            //未指定客户，返回空列表
            return Sets.newHashSet();
        }
        MdmTerminalVo param = new MdmTerminalVo();
        param.setTerminalCodeList(Lists.newArrayList(terminalCode));
        List<MdmOrgRespVo> result = ApiResultUtil.objResult(mdmTerminalFeign.detailBatch(param), true);
        if(CollectionUtil.listEmpty(result)) {
            return Sets.newHashSet();
        }
        return result.stream().collect(Collectors.groupingBy(MdmOrgRespVo::getOrgCode)).keySet();
    }

    /**
     * 查询生效的政策
     *
     * @param ids
     * @return
     */
    private List<PromotionPolicyInfoDataVo> selectActivities(PromotionPolicyEntity entity, List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        String now = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        LambdaQueryWrapper<PromotionPolicyEntity> wrapper = Wrappers.lambdaQuery(PromotionPolicyEntity.class)
                .in(PromotionPolicyEntity::getId, ids)
                .eq(PromotionPolicyEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode());//查询启用的

        List<String> chanel = Lists.newArrayList(PromotionPolicyEntity.ALL)
                , saleCompanyCode = Lists.newArrayList(PromotionPolicyEntity.ALL);
        if (null != entity) {
            if (StringUtils.isNotBlank(entity.getChanel())) {
                //渠道
                chanel.add(entity.getChanel());
            }
            if (StringUtils.isNotBlank(entity.getSaleCompanyCode())) {
                //销售公司
                saleCompanyCode.add(entity.getSaleCompanyCode());
            }
            if (StringUtils.isNotBlank(entity.getExt1())) {
                wrapper.eq(PromotionPolicyEntity::getExt1, entity.getExt1());
            }
            if (StringUtils.isNotBlank(entity.getExt2())) {
                wrapper.eq(PromotionPolicyEntity::getExt2, entity.getExt2());
            }
            if (StringUtils.isNotBlank(entity.getExt3())) {
                wrapper.eq(PromotionPolicyEntity::getExt3, entity.getExt3());
            }
            if (StringUtils.isNotBlank(entity.getExt4())) {
                wrapper.eq(PromotionPolicyEntity::getExt4, entity.getExt4());
            }
            if (StringUtils.isNotBlank(entity.getExt5())) {
                wrapper.eq(PromotionPolicyEntity::getExt5, entity.getExt5());
            }
        }
        //渠道
        wrapper.in(PromotionPolicyEntity::getChanel, chanel);
        //销售公司
        wrapper.in(PromotionPolicyEntity::getSaleCompanyCode, saleCompanyCode);

        //查询生效期内的
        wrapper.le(PromotionPolicyEntity::getBeginTime, now + " " + CrmDateUtils.TIME_STR_00)
                .ge(PromotionPolicyEntity::getEndTime, now + " " + CrmDateUtils.TIME_STR_235959);
        List<PromotionPolicyInfoDataVo> dataVos = CrmBeanUtil.copyList(promotionPolicyMapper.selectList(wrapper), PromotionPolicyInfoDataVo.class);
        //记录命中明细
        String hitKey = wrapper.getCustomSqlSegment();
        String hitValue = JsonPropertyUtil.toJsonString(wrapper.getParamNameValuePairs());
        this.addHitDetail(dataVos, hitKey, hitValue);
        return dataVos;
    }

    private void addHitDetail(List<PromotionPolicyInfoDataVo> dataVos, String hitKey, String hitValue){
        if(null == dataVos){
            return;
        }
        PromotionPolicyInfoDataVo.HitDetail hitDetail = new PromotionPolicyInfoDataVo.HitDetail(hitKey, hitValue);
        dataVos.forEach(v -> {
            v.addHitDetail(hitDetail);
        });
    }

    /**
     * 组装组织编码
     *
     * @param custCode
     * @return
     */
    private Set<String> buildOrgCodes(String custCode, String orgCode) {
        //已经指定了组织编码
        if(StringUtils.isNotBlank(orgCode)){
            //查询组织的上级组织
            return this.loadParentOrgCodes(Sets.newHashSet(orgCode));
        }
        //组织编码为空，尝试查询客户对应的组织
        if(StringUtils.isBlank(custCode)){
            //未指定客户，返回空列表
            return Sets.newHashSet();
        }
        //根据经销商编码查询其组织编码列表
        Result<List<String>> result = this.mdmCustomerROrgFeign.findOrgCodeListByCustomerCode(new MdmCustomerROrgReqVo().setCustomerCode(custCode));
        List<String> orgCodes = ApiResultUtil.objResult(result);
        if(null == orgCodes){
            return Sets.newHashSet();
        }
        //查询组织的上级组织
        return this.loadParentOrgCodes(Sets.newHashSet(orgCodes));
    }

    /**
     * 查询组织的上级组织
     * @param orgCodes
     * @return
     */
    private Set<String> loadParentOrgCodes(Set<String> orgCodes){
        List<String> parentOrgCodes = ApiResultUtil.objResult(this.mdmOrgFeign.findAllParentOrgCodeExcludeSelf(Lists.newArrayList(orgCodes)));
        orgCodes.addAll(parentOrgCodes);
        return orgCodes;
    }

    /**
     * 合并商品、经销商的政策列表，得到经销商的商品列表最终可享受的政策
     *
     * @param promotionPolicyIds
     * @param promotionPolicyIdAsScope
     */
    private List<String> merge(Set<String> promotionPolicyIds, Set<String> promotionPolicyIdAsScope) {
        return promotionPolicyIds.stream()
                // 商品对应的政策与范围包含对应的政策合并
                .filter(v -> {
                    return promotionPolicyIdAsScope.contains(v);
                }).collect(Collectors.toList());
    }

    /**
     * 查询给定的客户在给定的商品列表下能够享受的促销政策
     *
     * @param custCode
     * @param productCodes
     * @author: luoqi
     * @Date: 2020-10-21 11:20
     * @version: V1.0
     * @Description:
     */
    @Override
    public Map<String, List<PromotionPolicyInfoVo>> loadPromotionPolicyInfoByCustAndProducts(String custCode, List<String> productCodes) {
        if (StringUtils.isBlank(custCode) || CollectionUtils.isEmpty(productCodes)) {
            throw new BusinessException("请指定客户和商品编码");
        }
        //筛选非空
        productCodes = productCodes.stream().filter(v -> { return StringUtils.isNotBlank(v);}).collect(Collectors.toList());
        //组织编码
        Set<String> orgCodes = this.buildOrgCodes(custCode, null);
        //经销商享受的政策
        Set<String> promotionPolicyIdAsScope = promotionPolicyScopeService.selectPromotionPolicyIdByScopeCode(custCode, orgCodes);
        //商品编码去重
        Set<String> distinctProductCodes = Sets.newHashSet(productCodes);
        productCodes = Lists.newArrayList(distinctProductCodes);
        //商品编码按每次 999 的长度进行分片处理，防止一次性处理的商品过多
        List<List<String>> partitionList = Lists.partition(productCodes, 999);
        Map<String, List<PromotionPolicyInfoVo>> results = this.partitionListProccess(partitionList, promotionPolicyIdAsScope);
        //计算不存在的商品或商品政策为空的数据
        for (String productCode : distinctProductCodes) {
            if (results.containsKey(productCode)) {
                continue;
            }
            results.put(productCode, Lists.newArrayList());
        }
        return results;
    }

    /**
     * 给定的客户在给定的商品列表下是否能享受促销政策
     *
     * @param custCode
     * @param productCodes
     * @return
     */
    @Override
    public Map<String, Boolean> custProductsHasPromotionPolicy(String custCode, List<String> productCodes) {
        Map<String, List<PromotionPolicyInfoVo>> temp = this.loadPromotionPolicyInfoByCustAndProducts(custCode, productCodes);
        Map<String, Boolean> data = Maps.newHashMap();
        temp.forEach((s, promotionPolicyInfoVos) -> {
            data.put(s, !CollectionUtils.isEmpty(promotionPolicyInfoVos));
        });
        return data;
    }

    private Map<String, List<PromotionPolicyInfoVo>> partitionListProccess(List<List<String>> partitionList, Set<String> promotionPolicyIdAsScope) {
        Map<String, List<PromotionPolicyInfoVo>> results = Maps.newHashMap();
        for (List<String> list : partitionList) {
            //商品享受的政策
            List<PromotionPolicyProductEntity> promotionPolicyIdAsProduct = promotionPolicyProductService.selectPromotionPolicyIdByProductCodes(list);
            Map<String, Set<String>> productCodeMapPromotionPolicyIds = promotionPolicyIdAsProduct.stream().collect(Collectors.groupingBy(PromotionPolicyProductEntity::getProductCode
                    , Collectors.mapping(PromotionPolicyProductEntity::getPromotionPolicyId, Collectors.toSet())));
            //合并商品、经销商的政策列表，得到经销商的商品政策映射关系
            Map<String, List<String>> custProductPromotionPolicyId = Maps.newHashMap();
            //会使用到的政策id
            Set<String> mergedPromotionPolicyId = Sets.newHashSet();
            for (Map.Entry<String, Set<String>> entry : productCodeMapPromotionPolicyIds.entrySet()) {
                String productCode = entry.getKey();
                Set<String> promotionPolicyIds = entry.getValue();
                //合并商品、经销商的政策列表，得到经销商的商品列表最终可享受的政策
                List<String> temp = this.merge(promotionPolicyIds, promotionPolicyIdAsScope);
                //商品政策映射关系
                custProductPromotionPolicyId.put(productCode, temp);
                //收集会使用到的政策id
                mergedPromotionPolicyId.addAll(temp);
            }
            //根据 custProductPromotionPolicyId 映射关系，组装最终的商品政策数据
            results.putAll(this.loadPromotionPolicyInfoVo(mergedPromotionPolicyId, custProductPromotionPolicyId));
        }
        return results;
    }

    /**
     * 根据 custProductPromotionPolicyId 映射关系，组装最终的商品政策数据
     *
     * @param mergedPromotionPolicyId      会使用到的政策id
     * @param custProductPromotionPolicyId 商品政策映射关系
     * @return
     */
    private Map<String, List<PromotionPolicyInfoVo>> loadPromotionPolicyInfoVo(Set<String> mergedPromotionPolicyId, Map<String, List<String>> custProductPromotionPolicyId) {
        //会使用到的政策数据
        Map<String, PromotionPolicyInfoVo> infoVoMap;
        if (CollectionUtils.isEmpty(mergedPromotionPolicyId)) {
            infoVoMap = Maps.newHashMap();
        } else {
            //使用政策id 查询会使用到的政策数据
            infoVoMap = this.selectActivities(null, Lists.newArrayList(mergedPromotionPolicyId))
                    .stream().collect(Collectors.toMap(PromotionPolicyInfoVo::getId, v -> v, (r, r2) -> r2));
        }
        //根据 custProductPromotionPolicyId 映射关系，组装最终的商品政策数据
        Map<String, List<PromotionPolicyInfoVo>> custProductPromotionPolicyInfoVo = Maps.newHashMap();
        for (Map.Entry<String, List<String>> entry : custProductPromotionPolicyId.entrySet()) {
            String productCode = entry.getKey();
            List<String> promotionPolicyIds = entry.getValue();
            List<PromotionPolicyInfoVo> infoVos = Lists.newArrayList();
            promotionPolicyIds.forEach(v -> {
                if (infoVoMap.containsKey(v)) {
                    infoVos.add(infoVoMap.get(v));
                }
            });
            custProductPromotionPolicyInfoVo.put(productCode, infoVos);
        }
        return custProductPromotionPolicyInfoVo;
    }

    /**
     * 加载政策编辑页面数据
     *
     * @param id
     * @author: luoqi
     * @Date: 2020-10-20 17:07
     * @version: V1.0
     * @Description:
     */
    @Override
    public PromotionPolicyEditVo loadPolicyEditVo(String id, String templateId) {
        PromotionPolicyEditVo templateEditVo;
        if (StringUtils.isNotBlank(id)) {
            PromotionPolicyEntity entity = promotionPolicyMapper.selectById(id);
            AssertUtils.isNotNull(entity, "未加载到政策数据");
            templateEditVo = CrmBeanUtil.copy(entity, PromotionPolicyEditVo.class);
        } else {
            AssertUtils.isNotEmpty(templateId, "请选择促销政策模板");
            templateEditVo = new PromotionPolicyEditVo();
            templateEditVo.setTemplateId(templateId);
        }
        this.promotionPolicyProductService.loadListForEditPromotionPolicy(templateEditVo);
        this.promotionPolicyScopeService.loadListForEditPromotionPolicy(templateEditVo);
        this.promotionPolicyRuleService.loadListForEditPromotionPolicy(templateEditVo);
        Map<String, String> map = DictUtil.getDictValueMapsByCodes(PromotionPolicyEunm.dict_type_as_promotion_policy_status);
        templateEditVo.setPromotionPolicyStatus(PromotionPolicyUtil.promotionPolicyStatus(templateEditVo.getEnableStatus(), templateEditVo.getBeginTime(), templateEditVo.getEndTime()));
        if(null == map){
            return templateEditVo;
        }
        templateEditVo.setPromotionPolicyStatusName(map.get(templateEditVo.getPromotionPolicyStatus()));

        if(PromotionPolicyEntity.ALL.equals(templateEditVo.getChanel())){
            templateEditVo.setChanel(null);
        }
        if(PromotionPolicyEntity.ALL.equals(templateEditVo.getSaleCompanyCode())){
            templateEditVo.setSaleCompanyCode(null);
            templateEditVo.setSaleCompanyName(null);
        }
        return templateEditVo;
    }


    /**
     * a) 待执行时可以编辑除编码外的所有数据
     * b) 执行中时只可编辑结束时间调整至当前时间之后的时间点
     * c) 已结束时不可编辑
     * d) 待执行时可删除
     *
     * @author: luoqi
     * @Date: 2020-10-19 16:30
     * @version: V1.0
     * @Description:
     */
    private void doUpdate(PromotionPolicyEntity oldEntity, PromotionPolicyEditReqVo vo) {
        PromotionPolicyEntity entity = CrmBeanUtil.copy(vo, PromotionPolicyEntity.class);

        String timeActive = PromotionPolicyUtil.timeActive(oldEntity.getBeginTime(), oldEntity.getEndTime());
        if (PromotionPolicyEunm.PromotionPolicyStatusEunm.WAITING.getCode().equals(timeActive)) {
            // 待执行编辑时可能会换模板，所以先冲掉需置空的字段
            this.promotionPolicyMapper.updateNullValById(entity);
        } else if (PromotionPolicyEunm.PromotionPolicyStatusEunm.RUNNING.getCode().equals(timeActive)) {
            // 执行中时（日期已生效）只可编辑结束时间调整至当前时间之后的时间点
            long now = System.currentTimeMillis() / 1000;
            if (now > CrmDateUtils.parseyyyyMMddHHmmss(vo.getEndTime())
                    .toEpochSecond(ZoneOffset.of("+8"))) {
                throw new BusinessException("执行中的政策只可编辑结束时间调整至当前时间之后的时间点");
            }
            entity = new PromotionPolicyEntity();
            entity.setId(vo.getId());
            entity.setEndTime(vo.getEndTime());
        } else {
            throw new BusinessException("该政策已结束，不可修改！");
        }

        this.promotionPolicyMapper.updateById(entity);
    }

    @Override
    public PageResult<PromotionPolicyPageResp> getPageList(PromotionPolicyQueryVo vo) {
        Page<PromotionPolicyQueryVo> page = PageUtil.buildPage(vo.getPageNum(), vo.getPageSize());
        List<PromotionPolicyPageResp> list = this.promotionPolicyMapper.findList(page, vo);
        Map<String, String> map = DictUtil.getDictValueMapsByCodes(PromotionPolicyEunm.dict_type_as_promotion_policy_status);
        list.forEach(v -> {
            if(PromotionPolicyEntity.ALL.equals(v.getChanel())){
                v.setChanel(null);
            }
            if(PromotionPolicyEntity.ALL.equals(v.getSaleCompanyCode())){
                v.setSaleCompanyCode(null);
                v.setSaleCompanyName(null);
            }
            v.setPromotionPolicyStatusNoEnableStatus(PromotionPolicyUtil.promotionPolicyStatusNoEnable(v.getBeginTime(), v.getEndTime()));
            v.setPromotionPolicyStatus(PromotionPolicyUtil.promotionPolicyStatus(v.getEnableStatus(), v.getBeginTime(), v.getEndTime()));
            if(null == map){
                return;
            }
            v.setPromotionPolicyStatusName(map.get(v.getPromotionPolicyStatus()));

        });
        return PageResult.<PromotionPolicyPageResp>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询模板下的政策
     *
     * @param templateId
     * @param templateCode
     * @author: luoqi
     * @Date: 2020-10-16 11:20
     * @version: V1.0
     * @Description:
     */
    @Override
    public List<PromotionPolicyEntity> getListByTemplate(String templateId, String templateCode) {

        QueryWrapper<PromotionPolicyEntity> wrapper = Wrappers.query();
        if (StringUtils.isNotBlank(templateId)) {
            wrapper.in("template_id", templateId);
        } else if (StringUtils.isNotBlank(templateCode)) {
            wrapper.in("template_code", templateCode);
        }
        return promotionPolicyMapper.selectList(wrapper);
    }

    /**
     * 删除
     *
     * @param promotionPolicyIds
     * @author: luoqi
     * @Date: 2020-10-16 15:46
     * @version: V1.0
     * @Description:
     */
    @Override
    public void del(List<String> promotionPolicyIds) {
        this.delCheck(promotionPolicyIds);
        this.promotionPolicyRuleService.delByPromotionPolicyIdOrCode(promotionPolicyIds, null);
        this.promotionPolicyScopeService.delByPromotionPolicyIdOrCode(promotionPolicyIds, null);
        this.promotionPolicyProductService.delByPromotionPolicyIdOrCode(promotionPolicyIds, null);
        for(String id : promotionPolicyIds){
            PromotionPolicyVo promotionPolicyVo = this.findById(id);
            crmLogSendUtil.sendForDel(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(),
                    promotionPolicyVo.getId(),null,promotionPolicyVo);
        }
        this.promotionPolicyMapper.deleteBatchIds(promotionPolicyIds);
        //通知可购清单更新
        this.countBeginAndEndAffectedCustoms(DateUtil.getDate(DateUtil.DEFAULT_DATE_FORMAT), promotionPolicyIds);
        //删除缓存
        PromotionUtil.deleteCaches(this.findByIds(Lists.newArrayList(promotionPolicyIds))
                .stream().map(PromotionPolicyVo::getPromotionPolicyCode).collect(Collectors.toList()));
    }

    private void delCheck(List<String> promotionPolicyIds) {
        if (CollectionUtils.isEmpty(promotionPolicyIds)) {
            return;
        }
        List<PromotionPolicyEntity> entities = promotionPolicyMapper.selectBatchIds(promotionPolicyIds);
        Set<String> temp = Sets.newHashSet();
        for (PromotionPolicyEntity entity : entities) {
            String timeActive = PromotionPolicyUtil.timeActive(entity.getBeginTime(), entity.getEndTime());
            if (!PromotionPolicyEunm.PromotionPolicyStatusEunm.WAITING.getCode().equals(timeActive)) {
                temp.add(entity.getPromotionPolicyCode());
            }
        }
        if (CollectionUtils.isEmpty(temp)) {
            return;
        }
        throw new BusinessException("操作失败，政策[" + temp.stream().collect(Collectors.joining(", ")) + "]为执行中或已结束状态！");
    }

    /**
     * 启用
     *
     * @param promotionPolicyIds
     * @author: luoqi
     * @Date: 2020-10-16 15:46
     * @version: V1.0
     * @Description:
     */
    @Override
    public void enable(List<String> promotionPolicyIds) {
        if (CollectionUtils.isEmpty(promotionPolicyIds)) {
            return;
        }
        this.updateStatus(promotionPolicyIds, CrmEnableStatusEnum.ENABLE);
        //通知可购清单更新
        this.countBeginAndEndAffectedCustoms(DateUtil.getDate(DateUtil.DEFAULT_DATE_FORMAT), promotionPolicyIds);
        //删除缓存
        PromotionUtil.deleteCaches(this.findByIds(Lists.newArrayList(promotionPolicyIds))
                .stream().map(PromotionPolicyVo::getPromotionPolicyCode).collect(Collectors.toList()));
    }

    /**
     * 禁用
     *
     * @param promotionPolicyIds
     * @author: luoqi
     * @Date: 2020-10-16 15:46
     * @version: V1.0
     * @Description:
     */
    @Override
    public void disable(List<String> promotionPolicyIds) {
        if (CollectionUtils.isEmpty(promotionPolicyIds)) {
            return;
        }
        //删除缓存
        PromotionUtil.deleteCaches(this.findByIds(Lists.newArrayList(promotionPolicyIds))
            .stream().map(PromotionPolicyVo::getPromotionPolicyCode).collect(Collectors.toList()));
        this.updateStatus(promotionPolicyIds, CrmEnableStatusEnum.DISABLE);
        //通知可购清单更新
        this.countBeginAndEndAffectedCustoms(DateUtil.getDate(DateUtil.DEFAULT_DATE_FORMAT), promotionPolicyIds);
    }


    private void updateStatus(List<String> promotionPolicyIds, CrmEnableStatusEnum statusEnum) {
        QueryWrapper<PromotionPolicyEntity> wrapper = Wrappers.query();
        wrapper.in("id", promotionPolicyIds);
        PromotionPolicyEntity entity = new PromotionPolicyEntity();
        entity.setEnableStatus(statusEnum.getCode());
        this.promotionPolicyMapper.update(entity, wrapper);
    }



    /**
     * 加载上下文前置数据
     *
     * @param configInfoQueryVo
     */
    private LoadConfigInfoContext loadConfigInfoContext(PromotionPolicyConfigInfoQueryVo configInfoQueryVo) {

        LoadConfigInfoContext loadConfigInfoContext = new LoadConfigInfoContext(configInfoQueryVo);
        String custCode = configInfoQueryVo.getCustCode();
        String orgCode = configInfoQueryVo.getOrgCode();
        String terminalCode = configInfoQueryVo.getTerminalCode();
        //组织编码, 只用传进来的，不通过经销商编码查
        Set<String> orgCodes = this.buildOrgCodes(custCode, orgCode);
        Set<String> terminalOrgCodes = this.buildOrgCodesByTerminal(terminalCode, orgCode);
        //该客户和组织被覆盖到的 政策范围（包含）
        Set<String> promotionPolicyIdAsScope = this.promotionPolicyScopeService.selectPromotionPolicyIdByScopeCode(custCode, orgCodes);
        //终端享受的政策
        Set<String> promotionPolicyIdAsScopeTerminal = promotionPolicyScopeService.selectPromotionPolicyIdByTerminal(terminalCode, terminalOrgCodes);
        promotionPolicyIdAsScope.addAll(promotionPolicyIdAsScopeTerminal);
        // 加载商品政策
        configInfoQueryVo.getOrderProductInfos().forEach(v -> {
            //加载商品政策
            this.loadProductPromotionPolicy(loadConfigInfoContext, promotionPolicyIdAsScope
                    , v.getProductCode());
            //记录命中明细
            List<PromotionPolicyInfoDataVo> promotionPolicyInfoVos = loadConfigInfoContext.getProductMapPromotionPolicy().get(v.getProductCode());
            this.addHitDetail(promotionPolicyInfoVos, "custCode", custCode);
            this.addHitDetail(promotionPolicyInfoVos, "orgCodes", JsonPropertyUtil.toJsonString(orgCodes));
        });
        //查询政策下的赠品映射
        this.promotionPolicyProductService
                .selectGiftByPromotionPolicyId(loadConfigInfoContext, CommonConstant.GLOBAL.YesOrNo.Y.getItemCode());
        //查询政策下的本品映射
        this.promotionPolicyProductService
                .selectGiftByPromotionPolicyId(loadConfigInfoContext, CommonConstant.GLOBAL.YesOrNo.N.getItemCode());
        //查询所有赠品详细信息
        this.loadProductInfo(loadConfigInfoContext);
        return loadConfigInfoContext;
    }


    private void loadProductInfo(LoadConfigInfoContext loadConfigInfoContext) {
        Map<String, MdmProductRespVo> productMap = Maps.newHashMap();
        Set<String> productCodeGiftY = loadConfigInfoContext.getCurrentProductCodesN();
        if (null != productCodeGiftY) {
            productCodeGiftY.forEach(v -> {
                productMap.put(v, ApiResultUtil.objResult(mdmProductFeign.detail(null, v), true));
            });
        }
        loadConfigInfoContext.setCurrentProductInfoNMap(productMap);

    }

    /**
     * 加载商品政策
     *
     * @param loadConfigInfoContext
     * @param promotionPolicyIdAsScope
     * @param productCode
     */
    private void loadProductPromotionPolicy(LoadConfigInfoContext loadConfigInfoContext, Set<String> promotionPolicyIdAsScope, String productCode) {
        //订单购买的商品命中的政策id
        Set<String> promotionPolicyIdAsProduct = this.promotionPolicyProductService.selectPromotionPolicyIdByProductCode(productCode);
        //合并两份政策id
        List<String> merged = this.merge(promotionPolicyIdAsProduct, promotionPolicyIdAsScope);
        if (CollectionUtils.isEmpty(merged)) {
            return;
        }
        //查询合并出来的政策
        List<PromotionPolicyInfoDataVo> promotionPolicyInfoVos = this.selectActivities(
                CrmBeanUtil.copy(loadConfigInfoContext.getConfigInfoQueryVo(), PromotionPolicyEntity.class), merged);
        loadConfigInfoContext.putPromotionPolicyInfoVos(productCode, promotionPolicyInfoVos);
        this.addHitDetail(promotionPolicyInfoVos, "productCode", productCode);
        this.addHitDetail(promotionPolicyInfoVos, "promotionPolicyIdAsProduct", JsonPropertyUtil.toJsonString(promotionPolicyIdAsProduct));
        this.addHitDetail(promotionPolicyInfoVos, "promotionPolicyIdAsScope", JsonPropertyUtil.toJsonString(promotionPolicyIdAsScope));
    }

    /**
     * 过滤政策
     * 1.组合起定量
     * 2.单品起定量
     * 后期使用 com.biz.crm.promotion.service.component.filter.PromotionPolicyFilter 接口赋予项目上的自定义过滤扩展能力
     *
     * @param context
     */
    private void filterPromotionPolicy(LoadConfigInfoContext context) {
        context.getProductMapPromotionPolicy().forEach((productCode, promotionPolicyInfoVos) -> {
            promotionPolicyInfoVos = promotionPolicyInfoVos.stream().filter(v -> {
                if(CollectionUtils.isEmpty(this.abstractFilters)){
                    return true;
                }
                for(AbstractFilter filter : this.abstractFilters){
                    if(filter.filter(v, context)){
                        continue;
                    }
                    return false;
                }
                return true;
            }).collect(Collectors.toList());
            context.putPromotionPolicyInfoVos(productCode, promotionPolicyInfoVos);
        });

    }

    /**
     * 根据商品查询过滤政策
     * 1.组合起定量
     * 2.单品起定量
     * 后期使用 com.biz.crm.promotion.service.component.filter.PromotionPolicyFilter 接口赋予项目上的自定义过滤扩展能力
     *
     * @param context
     * @param promotionPolicyInfoVos
     */
    private List<PromotionPolicyInfoDataVo> filterPromotionProductPolicy(LoadConfigInfoContext context, List<PromotionPolicyInfoDataVo> promotionPolicyInfoVos) {
        List<PromotionPolicyInfoDataVo> promotionPolicyInfoDataVos = promotionPolicyInfoVos.stream().filter(v -> {
            if(CollectionUtils.isEmpty(this.abstractProductFilters)){
                return true;
            }
            for(AbstractProductFilter filter : this.abstractProductFilters){
                if(filter.filter(v, context)){
                    continue;
                }
                return false;
            }
            return true;
        }).collect(Collectors.toList());
        return promotionPolicyInfoDataVos;
    }



    /**
     * 加载订单下所有商品享受的促销政策和政策规则配置信息
     * 基本流程：
     * 1.加载上下文前置数据 （查询订单下每一个商品能够享受的政策，政策的本品、赠品映射数据，赠品详细信息（这里有性能优化空间））
     * 2.对政策前置数据进行 起订量、组合商品条件过滤（这里以后可以向项目组暴露自定义的过滤扩展）
     * 3.加载政策规则配置数据（这里有性能优化空间）
     * 4.组装政策下的赠品详细信息
     * 5.计算该政策被哪些商品享受，并收集在改商品下(汇总政策到订单的每一个商品上)
     *
     * @param configInfoQueryVo
     * @return
     */
    @Override
    public Map<String, List<PromotionPolicyConfigInfo>> loadPromotionPolicyConfigInfoForOrder(PromotionPolicyConfigInfoQueryVo configInfoQueryVo) {
        //检查参数
        this.checkConfigInfoQueryVo(configInfoQueryVo);

        LoadConfigInfoContext context = this.loadConfigInfoContext(configInfoQueryVo);
        //筛选指定的促销政策，暂时只支持一个本品的情况
        List<PromotionPolicyConfigInfoQueryVo.ProductOrderInfo> orderProductInfos = configInfoQueryVo.getOrderProductInfos();
        for(PromotionPolicyConfigInfoQueryVo.ProductOrderInfo info : orderProductInfos) {
            if(StringUtils.isBlank(info.getPromotionPolicyId())) {
                continue;
            }
            List<PromotionPolicyInfoDataVo> dataVos = context.getProductMapPromotionPolicy().get(info.getProductCode());
            dataVos = dataVos.stream().filter(d -> info.getPromotionPolicyId().contains(d.getId())).collect(Collectors.toList());
            context.getProductMapPromotionPolicy().put(info.getProductCode(), dataVos);
        }
        //对政策前置数据进行 起订量、组合商品条件过滤（这里以后可以向项目组暴露自定义的过滤扩展）
        this.filterPromotionPolicy(context);
        Map<String, List<PromotionPolicyConfigInfo>> result = Maps.newHashMap();
        //遍历商品政策列表
        context.getProductMapPromotionPolicy().forEach((productCode, promotionPolicyInfoVos) -> {
            for (PromotionPolicyInfoDataVo infoVo : promotionPolicyInfoVos) {
                //加载政策规则配置数据
                PromotionPolicyConfigInfo promotionPolicyConfigInfo = this.loadPromotionPolicyRuleInfo(infoVo);
                //组装政策下的赠品详细信息
                this.buildProductInfoGiftY(promotionPolicyConfigInfo, context);
                //计算该政策被哪些商品享受，并收集在该商品下(单个商品享受/多个商品组合享受)
                this.countingPromotionPolicyToProduct(context, promotionPolicyConfigInfo, result);

            }
        });
        //计算政策优惠量
        this.calculateRuleExecute(result, configInfoQueryVo);
        //过滤掉促销上限不足的情况
        result.forEach((k, v) -> {
            List<PromotionPolicyConfigInfo> n = v.stream().filter(s -> s.getPromotionPolicyInfoVo().getCalculateRuleResponse().getResultData() == null).collect(Collectors.toList());
            v.removeAll(n);
        });
        return result;
    }

    /**
     * 计算优惠量
     * @param result
     * @param configInfoQueryVo
     */
    private void calculateRuleExecute(Map<String, List<PromotionPolicyConfigInfo>> result, PromotionPolicyConfigInfoQueryVo configInfoQueryVo){
        if(null == result){
            return;
        }
        List<PromotionPolicyConfigInfoQueryVo.ProductOrderInfo> orderProductInfos = configInfoQueryVo.getOrderProductInfos();
        if(CollectionUtils.isEmpty(orderProductInfos)){
            return;
        }
        result.forEach((s, promotionPolicyConfigInfos) -> {
            if(null == promotionPolicyConfigInfos){
                return;
            }
            promotionPolicyConfigInfos.forEach(v -> {
                RuleExecuteParam ruleExecuteParam = new RuleExecuteParam();
                ruleExecuteParam.setPromotionPolicyConfigInfo(v);
                List<RuleExecuteParam.ProductOrderBuy> productOrderBuy = Lists.newArrayList();
                for(PromotionPolicyConfigInfoQueryVo.ProductOrderInfo productOrderInfo : orderProductInfos){
                    if(s.contains(productOrderInfo.getProductCode())){
                        productOrderBuy.add(new RuleExecuteParam.ProductOrderBuy(productOrderInfo));
                    }
                }

                if(CollectionUtils.isEmpty(productOrderBuy)){
                    return;
                }
                ruleExecuteParam.setProductOrderBuy(productOrderBuy);
                ruleExecuteParam.setCusCode(configInfoQueryVo.getCustCode());
                ruleExecuteParam.setCusTypeCode(configInfoQueryVo.getCusTypeCode());
                ruleExecuteParam.setCusOrgCode(configInfoQueryVo.getOrgCode());
                CalculateRuleResponse ruleResponse = this.defaultRuleExecutor.execute(ruleExecuteParam);
                PromotionPolicyInfoDataVo configInfo = v.getPromotionPolicyInfoVo();
                if(null != configInfo){
                    configInfo.setCalculateRuleResponse(ruleResponse);
                }

            });
        });


    }



    /**
     * 加载商品享受的促销政策和政策规则配置信息
     * @param productPolicyParamVo
     * @return
     */
    @Override
    public List<PromotionPolicyConfigInfo> loadPromotionPolicyConfigInfoByProductCode(QueryProductPolicyParamVo productPolicyParamVo){
        if (StringUtils.isBlank(productPolicyParamVo.getProductCode())) {
            throw new BusinessException("请指定商品编码");
        }
        List<PromotionPolicyInfoDataVo> promotionPolicyInfoVos = this.loadPromotionPolicyInfos(productPolicyParamVo);
        if(CollectionUtils.isEmpty(promotionPolicyInfoVos)){
            return Lists.newArrayList();
        }
        LoadConfigInfoContext context = new LoadConfigInfoContext();
        //执行相关过滤器
//        promotionPolicyInfoVos = this.filterPromotionProductPolicy(context, promotionPolicyInfoVos);
        context.setPromotionPolicyIds(promotionPolicyInfoVos.stream().map(PromotionPolicyInfoVo :: getId).collect(Collectors.toSet()));
        //查询政策下的赠品映射
        this.promotionPolicyProductService
                .selectGiftByPromotionPolicyId(context, CommonConstant.GLOBAL.YesOrNo.Y.getItemCode());
        //查询政策下的本品映射
        this.promotionPolicyProductService
                .selectGiftByPromotionPolicyId(context, CommonConstant.GLOBAL.YesOrNo.N.getItemCode());
        //查询所有赠品详细信息
        this.loadProductInfo(context);
        List<PromotionPolicyConfigInfo> configInfos = Lists.newArrayList();
        for (PromotionPolicyInfoDataVo infoVo : promotionPolicyInfoVos) {
            //加载政策规则配置数据
            PromotionPolicyConfigInfo promotionPolicyConfigInfo = this.loadPromotionPolicyRuleInfo(infoVo);
            //组装政策下的赠品详细信息
            this.buildProductInfoGiftY(promotionPolicyConfigInfo, context);
            configInfos.add(promotionPolicyConfigInfo);
        }
        return configInfos;
    }

    /**
     * 根据id查询政策对象
     * @param promotionPolicyId
     * @return
     */
    @Override
    public PromotionPolicyVo findById(String promotionPolicyId) {
        if(StringUtils.isBlank(promotionPolicyId)) {
            return null;
        }
        PromotionPolicyEntity entity = this.promotionPolicyMapper.selectOne(Wrappers.<PromotionPolicyEntity>query().in("id", promotionPolicyId));
        return CrmBeanUtil.copy(entity, PromotionPolicyVo.class);
    }
    /**
     * 根据ids查询政策对象
     * @param promotionPolicyIds
     * @return
     */
    @Override
    public List<PromotionPolicyVo> findByIds(ArrayList<String> promotionPolicyIds) {
        if(CollectionUtil.listEmpty(promotionPolicyIds)) {
            return Lists.newArrayList();
        }
        List<PromotionPolicyEntity> entitys = this.promotionPolicyMapper.selectList(Wrappers.<PromotionPolicyEntity>query().in("id", promotionPolicyIds));
        return CrmBeanUtil.copyList(entitys, PromotionPolicyVo.class);
    }

    /**
     * 加载订单下所有商品享受的促销政策和政策规则配置信息
     * 筛选促销范围，促销范围、起订量、可用余量
     * 特别注意：因为丰谷线上已使用该方法，请不要随意改动，如果需要改动，请重载该方法
     * @param configInfoQueryVo
     */
    @Override
    public Map<String, List<PromotionPolicyConfigInfo>> loadPromotionPolicyConfigInfoForSelect(PromotionPolicyConfigInfoQueryVo configInfoQueryVo) {
        ValidateUtils.validate(configInfoQueryVo, "参数不能为空");
        ValidateUtils.notEmpty(configInfoQueryVo.getOrderProductInfos(), "订单明细不能为空");
        List<PromotionPolicyConfigInfoQueryVo.ProductOrderInfo> orderInfos = configInfoQueryVo.getOrderProductInfos();
        Map<String, List<PromotionPolicyConfigInfo>> result = Maps.newTreeMap();
        Map<String, List<PromotionPolicyConfigInfo>> productMap = Maps.newTreeMap();
        Map<String, List<PromotionPolicyConfigInfo>> orderResult = this.loadPromotionPolicyConfigInfoForOrder(configInfoQueryVo);
        result.putAll(orderResult);
        orderInfos.forEach(info -> {
            QueryProductPolicyParamVo productParam = new QueryProductPolicyParamVo();
            productParam.setCustCode(configInfoQueryVo.getCustCode());
            productParam.setOrgCode(configInfoQueryVo.getOrgCode());
            productParam.setProductCode(info.getProductCode());
            productParam.setTerminalCode(configInfoQueryVo.getTerminalCode());
            List<PromotionPolicyConfigInfo> productResult = this.loadPromotionPolicyConfigInfoByProductCode(productParam);
            productMap.put(info.getProductCode(), productResult);
            List<PromotionPolicyConfigInfo> infoList = orderResult.get(info.getProductCode());
            if(CollectionUtil.listEmpty(infoList)) {
                productResult.forEach(p -> p.setHit(false));
            } else {
                Set<String> orderPromotionCodes = infoList.stream().collect(Collectors.groupingBy(i -> i.getPromotionPolicyInfoVo().getPromotionPolicyCode())).keySet();
                productResult.stream().forEach(p -> {
                    if(!orderPromotionCodes.contains(p.getPromotionPolicyInfoVo().getPromotionPolicyCode())) {
                        p.setHit(false);
                    } else {
                        PromotionPolicyConfigInfo configInfo = infoList.stream().filter(i ->
                                Objects.equals(i.getPromotionPolicyInfoVo().getPromotionPolicyCode(),
                                        p.getPromotionPolicyInfoVo().getPromotionPolicyCode()))
                                .collect(Collectors.toList()).get(0);
                        productResult.remove(p);
                        productResult.add(configInfo);
                    }
                });
            }
        });
        result.putAll(productMap);
        return result;
    }

    /**
     * 根据客户查询所有可用促销（商品编码作为key）
     * @param cusCode
     * @return
     */
    @Override
    public Map<String, List<PromotionPolicyConfigInfo>> findProductMapByCusCode(String cusCode) {
        if(StringUtils.isBlank(cusCode)) {
            return Maps.newHashMap();
        }
        Set<String> promotionIds = promotionPolicyScopeService.selectPromotionPolicyIdByScopeCode(cusCode, Sets.newHashSet());
        if(!CollectionUtil.collectionNotEmpty(promotionIds)) {
            return Maps.newHashMap();
        }
        List<PromotionPolicyVo> promotionPolicyVos = this.findByIds(Lists.newArrayList(promotionIds));
        Map<String, PromotionPolicyVo> promotionPolicyVoMap = promotionPolicyVos.stream().collect(Collectors.toMap(PromotionPolicyVo::getId, a -> a));
        Map<String, List<PromotionPolicyProductEntity>> productMap = promotionPolicyProductService.findProductsByPromotionIds(Lists.newArrayList(promotionIds), CommonConstant.GLOBAL.YesOrNo.Y.getItemCode());
        Map<String, List<PromotionPolicyConfigInfo>> result = Maps.newHashMap();
        productMap.values().forEach(products ->
            products.forEach(p -> {
                List<PromotionPolicyConfigInfo> promotions = result.get(p.getProductCode());
                if(promotions == null) {
                    promotions = Lists.newArrayList();
                }
                PromotionPolicyInfoDataVo promotionPolicyInfoDataVo = CrmBeanUtil.copy(promotionPolicyVoMap.get(p.getPromotionPolicyId()), PromotionPolicyInfoDataVo.class);
                PromotionPolicyConfigInfo info = new PromotionPolicyConfigInfo();
                info.setPromotionPolicyInfoVo(promotionPolicyInfoDataVo);
                promotions.add(info);
                result.put(p.getProductCode(), promotions);
            })
        );
        return result;
    }

    /**
     * 根据编码查询政策（带出模板）
     * @param promotionPolicyCode
     * @return
     */
    @Override
    public PromotionPolicyVo findByCode(String promotionPolicyCode) {
        if(StringUtils.isBlank(promotionPolicyCode)) {
            return null;
        }
        PromotionPolicyEntity entity = this.promotionPolicyMapper.selectOne(Wrappers.<PromotionPolicyEntity>query().eq("promotion_policy_code", promotionPolicyCode));
        PromotionPolicyVo vo = CrmBeanUtil.copy(entity, PromotionPolicyVo.class);
        if(vo == null) {
            return null;
        }
        PromotionPolicyTemplateVo templateVo = this.promotionPolicyTemplateService.findByCode(vo.getTemplateCode());
        vo.setPromotionPolicyTemplateVo(templateVo);
        return vo;
    }

    //    @Override
    public Map<String, PromotionPolicyConfigInfo> searchPromotionToOrder(PromotionPolicyForOrderReqVo promotionPolicyForOrderReqVo) {
        if(promotionPolicyForOrderReqVo == null) {
            Maps.newHashMap();
        }
        List<PromotionPolicyForOrderProductReqVo> promotionList = promotionPolicyForOrderReqVo.getPromotionList();
        if(CollectionUtil.listEmpty(promotionList)) {
            return Maps.newHashMap();
        }
        Set<String> productCodes = promotionList.stream().collect(Collectors.toMap(PromotionPolicyForOrderProductReqVo::getProductCode, a -> a, (a, b) -> a)).keySet();
        Map<String, PromotionPolicyConfigInfo> result = Maps.newHashMap();
        productCodes.forEach(p -> {
            QueryProductPolicyParamVo param = new QueryProductPolicyParamVo();
            param.setCustCode(promotionPolicyForOrderReqVo.getCusCode());
            param.setOrgCode(promotionPolicyForOrderReqVo.getOrgCode());
            String[] ppArray = p.split(",");
            param.setProductCode(ppArray[0]);
            List<PromotionPolicyConfigInfo> promotionPolicyConfigInfos = this.loadPromotionPolicyConfigInfoByProductCode(param);
            if(CollectionUtil.listNotEmpty(promotionPolicyConfigInfos)) {
                promotionPolicyConfigInfos.forEach(pr -> {
                    String promotionCode = pr.getPromotionPolicyInfoVo().getPromotionPolicyCode();
                    result.put(String.join(",", p, promotionCode), pr);
                });
            }
        });
        return result;
    }

    /**
     * 加载政策规则配置数据
     *
     * @param infoVo
     * @return
     */
    private PromotionPolicyConfigInfo loadPromotionPolicyRuleInfo(PromotionPolicyInfoDataVo infoVo) {
        // 加载政策相关规则脚本
        List<PromotionPolicyRuleEntity> promotionPolicyRules = promotionPolicyRuleService.getRuleByPromotionPolicyIdOrCode(infoVo.getId(), null);
        Map<String, List<PromotionPolicyRuleEntity>> rulesMapAsPolicyRules = promotionPolicyRules.stream().collect(Collectors.groupingBy(PromotionPolicyRuleEntity::getRuleType));
        PromotionPolicyRuleEntity conditionAsPolicyRules = rulesMapAsPolicyRules.get(PromotionRuleVo.RULE_TYPE.condition.name()).get(0);
        PromotionPolicyRuleEntity calculateAsPolicyRules = rulesMapAsPolicyRules.get(PromotionRuleVo.RULE_TYPE.calculate.name()).get(0);
        List<PromotionPolicyRuleEntity> limitedAsPolicyRules = rulesMapAsPolicyRules.get(PromotionRuleVo.RULE_TYPE.limited.name());

        // 加载规则脚本
        Set<String> ruleIds = promotionPolicyRules.stream().map(PromotionPolicyRuleEntity::getPromotionRuleId).collect(Collectors.toSet());
        Map<String, List<PromotionRuleEntity>> rulesMap = promotionRuleService.getListByIds(ruleIds).stream().collect(Collectors.groupingBy(PromotionRuleEntity::getRuleType));
        PromotionRuleEntity condition = rulesMap.get(PromotionRuleVo.RULE_TYPE.condition.name()).get(0);
        PromotionRuleEntity calculate = rulesMap.get(PromotionRuleVo.RULE_TYPE.calculate.name()).get(0);

        //计算规则
        PromotionPolicyRuleVo calculateRuleVo = CrmBeanUtil.copy(calculateAsPolicyRules, PromotionPolicyRuleVo.class);
        calculateRuleVo.setFuncBody(calculate.getFuncBody());
        //条件规则
        PromotionPolicyRuleVo conditionRuleVo = CrmBeanUtil.copy(conditionAsPolicyRules, PromotionPolicyRuleVo.class);
        conditionRuleVo.setFuncBody(condition.getFuncBody());
        //条件规则
        List<PromotionPolicyRuleVo> limitedRuleVo = CrmBeanUtil.copyList(limitedAsPolicyRules, PromotionPolicyRuleVo.class);
        List<PromotionRuleEntity> limitedTemp = rulesMap.get(PromotionRuleVo.RULE_TYPE.limited.name());

        Map<String, PromotionRuleEntity> limited = null == limitedTemp ? Maps.newHashMap() : limitedTemp.stream()
                .collect(Collectors.toMap(PromotionRuleEntity::getRuleCode, v -> v, (t, t2) -> t2));
        limitedRuleVo.forEach(v -> {
            PromotionRuleEntity temp = limited.get(v.getRuleCode());
            if (null != temp) {
                v.setFuncBody(temp.getFuncBody());
            }
        });

        PromotionPolicyConfigInfo promotionPolicyConfigInfo = new PromotionPolicyConfigInfo();

        promotionPolicyConfigInfo.setPromotionPolicyInfoVo(infoVo);
        promotionPolicyConfigInfo.setCalculate(calculateRuleVo);
        promotionPolicyConfigInfo.setCondition(conditionRuleVo);
        promotionPolicyConfigInfo.setLimited(limitedRuleVo);
        return promotionPolicyConfigInfo;
    }

    /**
     * 计算该政策被哪些商品享受，并收集在改商品下
     *
     * @param context
     * @param promotionPolicyConfigInfo
     * @param result
     */
    private void countingPromotionPolicyToProduct(LoadConfigInfoContext context, PromotionPolicyConfigInfo promotionPolicyConfigInfo
            , Map<String, List<PromotionPolicyConfigInfo>> result) {
        //该政策的本品
        Map<String, PromotionPolicyProductEntity> currentProduct = context.getCurrentProductY().get(promotionPolicyConfigInfo.getPromotionPolicyInfoVo().getId());
        if (CollectionUtils.isEmpty(currentProduct)) {
            return;
        }
        //该政策可能有多个本品组合享受， 多个本品时 用 , 间隔
        String productCodes = currentProduct.keySet().stream().collect(Collectors.joining(","));
        if (StringUtils.isBlank(productCodes)) {
            return;
        }
        List<PromotionPolicyConfigInfo> configInfos = result.get(productCodes);
        if (null == configInfos) {
            configInfos = Lists.newArrayList();
        }
        //该组商品编码已经有对该政策的映射
        if (context.isDuplicationPromotionPolicy(productCodes, promotionPolicyConfigInfo.getPromotionPolicyInfoVo().getId())) {
            return;
        }
        configInfos.add(promotionPolicyConfigInfo);
        result.put(productCodes, configInfos);
    }

    /**
     * 组装政策下的赠品详细信息
     *
     * @param promotionPolicyConfigInfo
     * @param context
     */
    private void buildProductInfoGiftY(PromotionPolicyConfigInfo promotionPolicyConfigInfo, LoadConfigInfoContext context) {
        Map<String, MdmProductRespVo> currentProductInfoNMap = context.getCurrentProductInfoNMap();
        String promotionPolicyId = promotionPolicyConfigInfo.getPromotionPolicyInfoVo().getId();
        Map<String, PromotionPolicyProductEntity> currentProductN = context.getCurrentProductN().get(promotionPolicyId);
//        if(null == currentProductN){
//            return;
//        }
        List<MdmProductRespVo> productsGift = Lists.newArrayList();
        currentProductN.forEach((productCode, entity) -> {
            MdmProductRespVo productRespVo = currentProductInfoNMap.get(productCode);
            if (null != productRespVo) {
                productsGift.add(productRespVo);
            }
        });
        promotionPolicyConfigInfo.setProductsGift(productsGift);
    }

    /**
     * @param configInfoQueryVo
     */
    private void checkConfigInfoQueryVo(PromotionPolicyConfigInfoQueryVo configInfoQueryVo) {
        if (null == configInfoQueryVo) {
            throw new BusinessException("参数不能为空");
        }
        if (StringUtils.isBlank(configInfoQueryVo.getCustCode())) {
            throw new BusinessException("请指定客户编码");
        }
//        if(StringUtils.isBlank(configInfoQueryVo.getOrgCode())){
//            throw new BusinessException("请指定区域组织编码");
//        }
        if (CollectionUtils.isEmpty(configInfoQueryVo.getOrderProductInfos())) {
            throw new BusinessException("请指定订单商品信息");
        }
        configInfoQueryVo.getOrderProductInfos().forEach(v -> {
            if (StringUtils.isBlank(v.getProductCode())) {
                throw new BusinessException("请指定商品编码");
            }
            if (null == v.getProductBuyNo()) {
                throw new BusinessException("请指定商品[" + v.getProductCode() + "]购买数量");
            }
            if (null == v.getProductBuyAmount()) {
                throw new BusinessException("请指定商品[" + v.getProductCode() + "]购买金额");
            }
        });
    }


}
