package com.biz.crm.promotion.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.dms.promotion.*;
import com.biz.crm.promotion.entity.PromotionPolicyEntity;
import com.biz.crm.promotion.entity.PromotionPolicyTemplateEntity;
import com.biz.crm.promotion.entity.PromotionPolicyTemplateRuleEntity;
import com.biz.crm.promotion.entity.PromotionRuleEntity;
import com.biz.crm.promotion.mapper.PromotionPolicyTemplateMapper;
import com.biz.crm.promotion.mapper.PromotionPolicyTemplateRuleMapper;
import com.biz.crm.promotion.service.PromotionPolicyService;
import com.biz.crm.promotion.service.PromotionPolicyTemplateRuleService;
import com.biz.crm.promotion.service.PromotionPolicyTemplateService;
import com.biz.crm.promotion.service.PromotionRuleService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;


/**
 *  @author: luoqi
 *  @Date: 2020-10-15 16:50
 *  @version: V1.0
 *  @Description:
 */
@Transactional
@ConditionalOnMissingBean(name = "promotionPolicyTemplateServiceExpandImpl")
@Service("promotionPolicyTemplateService")
public class PromotionPolicyTemplateServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<PromotionPolicyTemplateMapper, PromotionPolicyTemplateEntity>
        implements PromotionPolicyTemplateService {

    @Resource
    private PromotionPolicyTemplateMapper promotionPolicyTemplateMapper;
    @Resource
    private PromotionPolicyTemplateRuleService promotionPolicyTemplateRuleService;
    @Resource
    private PromotionPolicyTemplateRuleMapper promotionPolicyTemplateRuleMapper;
    @Resource
    private PromotionRuleService promotionRuleService;
    @Resource
    private PromotionPolicyService promotionPolicyService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;



    @Override
    @Transactional
    public void addOrUpdate(PromotionPolicyTemplateEditVo vo) {
        this.checkParams(vo);
        if(StringUtils.isBlank(vo.getId())){
            QueryWrapper<PromotionPolicyTemplateEntity> wrapper = new QueryWrapper<>();
            wrapper.eq("template_code", vo.getTemplateCode());
            if(this.promotionPolicyTemplateMapper.selectCount(wrapper) > 0){
                throw new BusinessException("已存在的模板编码[" + vo.getTemplateCode() + "]。");
            }
            PromotionPolicyTemplateEntity newTemplate = CrmBeanUtil.copy(vo, PromotionPolicyTemplateEntity.class);
            this.promotionPolicyTemplateMapper.insert(newTemplate);
            crmLogSendUtil.sendForAdd(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), newTemplate.getId(),newTemplate.getId(),vo);

            vo.setId(newTemplate.getId());
        }else{
            if(null == this.promotionPolicyTemplateMapper.selectById(vo.getId())){
                throw new BusinessException("模板数据不存在,ID=[" + vo.getId() + "]");
            }
            PromotionPolicyTemplateEntity oldEntity = this.getById(vo.getId());
            PromotionPolicyTemplateEditVo oldobj = CrmBeanUtil.copy(oldEntity,PromotionPolicyTemplateEditVo.class);

            this.updateCheck(Lists.newArrayList(vo.getTemplateCode()));
            this.promotionPolicyTemplateMapper.updateById(CrmBeanUtil.copy(vo, PromotionPolicyTemplateEntity.class));
            crmLogSendUtil.sendForUpdate(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), oldEntity.getId(),oldEntity.getId(),oldobj,vo);

        }
        this.refreshTemplateRule(vo);
    }

    /**
     * 加载模板编辑页面数据
     *
     * @param templateId
     * @author: luoqi
     * @Date: 2020-10-17 11:47
     * @version: V1.0
     * @Description:
     */
    @Override
    public PromotionPolicyTemplateEditVo loadTemplateEditVo(String templateId, String promotionType) {
        PromotionPolicyTemplateEditVo templateEditVo;
        if(StringUtils.isNotBlank(templateId)){
            PromotionPolicyTemplateEntity entity = promotionPolicyTemplateMapper.selectById(templateId);
            AssertUtils.isNotNull(entity,"未加载到模板数据");
            templateEditVo = CrmBeanUtil.copy(entity, PromotionPolicyTemplateEditVo.class);
        }else {
            AssertUtils.isNotEmpty(promotionType,"请指定模板促销类型");
            templateEditVo = new PromotionPolicyTemplateEditVo();
            templateEditVo.setPromotionType(promotionType);
        }
        promotionRuleService.loadTemplateRuleGroupByRuleType(templateEditVo, true);
        return templateEditVo;
    }


    /**
     * 加载模板信息-政策编辑场景
     *  @author: luoqi
     *  @Date: 2020-10-20 17:21
     *  @version: V1.0
     *  @Description:
     */
    @Override
    public PromotionPolicyTemplateEditVo loadTemplateInfo(String templateId) {
        PromotionPolicyTemplateEntity entity = promotionPolicyTemplateMapper.selectById(templateId);
        AssertUtils.isNotNull(entity,"未加载到模板数据");
        PromotionPolicyTemplateEditVo templateEditVo = CrmBeanUtil.copy(entity, PromotionPolicyTemplateEditVo.class);
        promotionRuleService.loadTemplateRuleGroupByRuleType(templateEditVo, false);
        return templateEditVo;
    }

    /**
     * 刷新模板规则
     *  @author: luoqi
     *  @Date: 2020-10-17 11:36
     *  @version: V1.0
     *  @Description:
     */
    private void refreshTemplateRule(PromotionPolicyTemplateEditVo vo){
        //删除旧的模板规则
        if(StringUtils.isNotBlank(vo.getTemplateCode())){
            this.delTemplateRule(null, Lists.newArrayList(vo.getTemplateCode()));
        }

        //根据用户提交的规则列表，加载规则数据
        Map<String, PromotionRuleEntity> ruleMapping = this.loadPromotionRule(vo);
        if(null == ruleMapping){
            return;
        }
        //根据规则数据，组装模板规则实体
        List<PromotionPolicyTemplateRuleEntity> templateRuleEntities = this.buildTemplateRuleEntitys(ruleMapping, vo);
        //插入模板规则
        promotionPolicyTemplateRuleService.saveList(templateRuleEntities);

    }


    /**
     * 根据规则数据，组装模板规则实体
     *  @author: luoqi
     *  @Date: 2020-10-17 11:39
     *  @version: V1.0
     *  @Description:
     */
    private List<PromotionPolicyTemplateRuleEntity> buildTemplateRuleEntitys(Map<String, PromotionRuleEntity> ruleMapping, PromotionPolicyTemplateEditVo vo){
        List<PromotionRuleSelectionVo> all = vo.takeAllRule();

        List<PromotionPolicyTemplateRuleEntity> templateRuleEntities = Lists.newArrayList();
        for(PromotionRuleSelectionVo ruleSelectionVo : all){
            if(ruleMapping.containsKey(ruleSelectionVo.getId())){
                PromotionRuleEntity promotionRuleEntity = ruleMapping.get(ruleSelectionVo.getId());
                PromotionPolicyTemplateRuleEntity temp = CrmBeanUtil.copy(promotionRuleEntity, PromotionPolicyTemplateRuleEntity.class);
                temp.setPromotionRuleId(promotionRuleEntity.getId());
                temp.setPromotionPolicyTemplateId(vo.getId());
                temp.setPromotionPolicyTemplateCode(vo.getTemplateCode());
                temp.setRuleCode(promotionRuleEntity.getRuleCode());
                temp.setRuleType(promotionRuleEntity.getRuleType());
                temp.setPromotionType(promotionRuleEntity.getPromotionType());
                temp.setId(null);
                templateRuleEntities.add(temp);
            }else{
                throw new BusinessException("规则ID[" + ruleSelectionVo.getId() + "]不存在。");
            }
        }
        return templateRuleEntities;
    }


    /**
     * 根据用户提交的规则列表，加载规则数据
     *  @author: luoqi
     *  @Date: 2020-10-17 11:39
     *  @version: V1.0
     *  @Description:
     */
    private Map<String, PromotionRuleEntity> loadPromotionRule(PromotionPolicyTemplateEditVo vo){
        List<PromotionRuleSelectionVo> all = vo.takeAllRule();
        Set<String> ids = Sets.newHashSet();
        if(null != all){
            ids.addAll(all.stream().map(PromotionRuleSelectionVo :: getId).collect(Collectors.toSet()));
        }
        if(ids.size() == 0){
            return null;
        }
        return promotionRuleService.getListByIds(ids).stream().collect(Collectors.toMap(PromotionRuleEntity :: getId, v -> v, (t, t2) -> t2));
    }

    @Override
    public PageResult<PromotionPolicyTemplateVo> getPageList(PromotionPolicyTemplateQueryVo vo) {
        Page<PromotionPolicyTemplateQueryVo> page = PageUtil.buildPage(vo.getPageNum(), vo.getPageSize());
        List<PromotionPolicyTemplateVo> list = this.promotionPolicyTemplateMapper.findList(page, vo);
        return PageResult.<PromotionPolicyTemplateVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 删除
     *
     * @param templateIds
     * @author: luoqi
     * @Date: 2020-10-16 15:46
     * @version: V1.0
     * @Description:
     */
    @Transactional
    @Override
    public void del(List<String> templateIds) {
        this.updateCheck(templateIds);
        this.delTemplateRule(templateIds, null);
        QueryWrapper<PromotionPolicyTemplateEntity> wrapper = Wrappers.query();
        wrapper.in("id", templateIds);
        List<PromotionPolicyTemplateEntity> promotionPolicyTemplateEntities = this.promotionPolicyTemplateMapper.selectList(wrapper);

        for(PromotionPolicyTemplateEntity entity : promotionPolicyTemplateEntities){
            PromotionPolicyTemplateVo promotionPolicyTemplateVo = CrmBeanUtil.copy(entity,PromotionPolicyTemplateVo.class);
            crmLogSendUtil.sendForDel(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(),
                    promotionPolicyTemplateVo.getId(),null,promotionPolicyTemplateVo);
        }
        this.promotionPolicyTemplateMapper.delete(wrapper);
    }
    private void delTemplateRule(List<String> templateIds, List<String> templateCodes){
        QueryWrapper<PromotionPolicyTemplateRuleEntity> rulleWrapper = Wrappers.query();
        if(!CollectionUtils.isEmpty(templateIds)){
            rulleWrapper.in("promotion_Policy_Template_id", templateIds);
        }else if(!CollectionUtils.isEmpty(templateCodes)){
            rulleWrapper.in("promotion_Policy_Template_code", templateCodes);
        }
        promotionPolicyTemplateRuleMapper.delete(rulleWrapper);
    }

    /**
     * 启用
     *
     * @param templateIds
     * @author: luoqi
     * @Date: 2020-10-16 15:46
     * @version: V1.0
     * @Description:
     */
    @Transactional
    @Override
    public void enable(List<String> templateIds) {
        if(CollectionUtils.isEmpty(templateIds)){
            return;
        }
        this.updateStatus(templateIds, CrmEnableStatusEnum.ENABLE);
    }

    private void updateStatus(List<String> templateIds, CrmEnableStatusEnum statusEnum){
        QueryWrapper<PromotionPolicyTemplateEntity> wrapper = Wrappers.<PromotionPolicyTemplateEntity>query();
        wrapper.in("id", templateIds);
        PromotionPolicyTemplateEntity entity = new PromotionPolicyTemplateEntity();
        entity.setEnableStatus(statusEnum.getCode());
        this.promotionPolicyTemplateMapper.update(entity, wrapper);
    }
    /**
     * 禁用
     *
     * @param templateIds
     * @author: luoqi
     * @Date: 2020-10-16 15:46
     * @version: V1.0
     * @Description:
     */
    @Transactional
    @Override
    public void disable(List<String> templateIds) {
        if(CollectionUtils.isEmpty(templateIds)){
            return;
        }
        this.updateStatus(templateIds, CrmEnableStatusEnum.DISABLE);
    }

    /**
     * 根据编码查询
     * @param templateCode
     * @return
     */
    @Override
    public PromotionPolicyTemplateVo findByCode(String templateCode) {
        if(StringUtils.isBlank(templateCode)) {
            return null;
        }
        PromotionPolicyTemplateEntity entity = this.promotionPolicyTemplateMapper.selectOne(Wrappers.<PromotionPolicyTemplateEntity>query().eq("template_code", templateCode));
        return CrmBeanUtil.copy(entity, PromotionPolicyTemplateVo.class);
    }

    private void updateCheck(List<String> templateIds){
        if(CollectionUtils.isEmpty(templateIds)){
            return;
        }
        for(String templateId : templateIds){
            List<PromotionPolicyEntity> list = promotionPolicyService.getListByTemplate(templateId, null);
            if( ! CollectionUtils.isEmpty(list)){
                throw new BusinessException("操作失败，模板[" + list.get(0).getTemplateCode() + "]已有关联政策");
            }
        }

    }
    /**
     * 校验参数
     *  @author: luoqi
     *  @Date: 2020-10-15 17:42
     *  @version: V1.0
     *  @Description:
     */
    private void checkParams(PromotionPolicyTemplateEditVo vo){
        AssertUtils.isNotNull(vo,"无效参数，请重试");
        AssertUtils.isNotEmpty(vo.getPromotionProduct(),"促销商品不能为空!");
        AssertUtils.isNotEmpty(vo.getPromotionType(),"促销类型不能为空!");
        AssertUtils.isNotEmpty(vo.getTemplateCode(),"模板编码不能为空!");
    }
}
