package com.biz.crm.promotion.service.npromotion.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.dms.PromotionPolicyEunm;
import com.biz.crm.nebular.dms.npromotion.bo.LimitedParamBo;
import com.biz.crm.nebular.dms.npromotion.bo.LimitedResultBo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionProductVo;
import com.biz.crm.nebular.dms.promotion.PromotionRuleEditVo;
import com.biz.crm.promotion.enums.AccountTypeDynamicEnum;
import com.biz.crm.eunm.dms.RuleTypeDynamicEnum;
import com.biz.crm.promotion.service.PromotionRuleService;
import com.biz.crm.promotion.service.npromotion.PromotionComputeService;
import com.biz.crm.promotion.service.npromotion.PromotionService;
import com.biz.crm.promotion.service.npromotion.beans.AbstractCalculateComputer;
import com.biz.crm.promotion.service.npromotion.beans.AbstractConditionComputer;
import com.biz.crm.promotion.service.npromotion.beans.AbstractLimitedComputer;
import com.biz.crm.promotion.service.npromotion.beans.filters.AbstractCommonFilter;
import com.biz.crm.promotion.service.npromotion.beans.filters.AbstractMoqFilter;
import com.biz.crm.promotion.service.npromotion.beans.filters.AbstractProductFilter;
import com.biz.crm.promotion.service.npromotion.beans.filters.AbstractScopeFilter;
import com.biz.crm.nebular.dms.npromotion.bo.CalculateParamBo;
import com.biz.crm.nebular.dms.npromotion.bo.CalculateResultBo;
import com.biz.crm.nebular.dms.npromotion.vo.CalculateHitResultVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionEditVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionQueryReq;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionAvailableResp;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionRuleVo;
import com.biz.crm.promotion.util.PromotionUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

import static com.biz.crm.promotion.service.npromotion.beans.filters.AbstractMoqFilter.MOQ_MATCH_MSG_OK;

/**
 * 促销政策计算（应用）逻辑处理类
 * @Author: chenrong
 * @Date: 2021/4/21 14:40
 */
@Service
@ConditionalOnMissingBean(name = "promotionComputeServiceExtendImpl")
public class PromotionComputeServiceImpl implements PromotionComputeService {

  @Resource
  private PromotionService promotionService;
  @Resource
  private AbstractProductFilter productFilter;
  @Resource
  private AbstractMoqFilter moqFilter;
  @Autowired(required = false)
  private Map<String, AbstractCommonFilter> commonFilterMap;

  /**
   * 根据客户以及商品（本品）查询可用的所有促销政策
   * @param param
   * @return
   */
  @Override
  public List<PromotionAvailableResp> findAvailablePromotionsByAccount(PromotionQueryReq param) {
    if(param == null || StringUtils.isEmpty(param.getAccountCode())) {
      return Lists.newArrayList();
    }
    //1、如果用户类型为空，默认为客户类型
    if(StringUtils.isEmpty(param.getAccountType())) {
      param.setAccountType(AccountTypeDynamicEnum.CUSTOMER.getCode());
    }
    AccountTypeDynamicEnum accountTypeDynamicEnum = AccountTypeDynamicEnum.getAccountTypeEnumsByCode(param.getAccountType());
    if(accountTypeDynamicEnum == null || accountTypeDynamicEnum.getFilerBeanCls() == null) {
      return Lists.newArrayList();
    }
    List<PromotionAvailableResp> result = Lists.newArrayList();
    //2、范围筛选
    AbstractScopeFilter filter = PromotionUtil.getBean(AbstractScopeFilter.class);
    if(filter == null) {
      return Lists.newArrayList();
    }
    List<String> promotionCodes = filter.apply(param);
    if(CollectionUtil.listEmpty(promotionCodes)) {
      return Lists.newArrayList();
    }
    //3、商品筛选
    Map<String, List<String>> productPromotionsMap = productFilter.apply(param);
    //4、遍历商品筛选结果，与范围筛选出来的政策编码取交集，只有同时满足的政策，才会进行下一步处理
    productPromotionsMap.forEach((productCode, promotionCodeList) -> promotionCodeList.retainAll(promotionCodes));
    //5、遍历商品筛选结果，用促销编码列表获取促销对象列表，构建返回对象
    productPromotionsMap.forEach((k, v) -> {
      PromotionAvailableResp promotionHitResp = new PromotionAvailableResp();
      promotionHitResp.setProductCode(k);
      List<PromotionEditVo> promotionEditVos = PromotionUtil.getCacheList(v, promotionService);
      promotionHitResp.setPromotions(filterAvailable(promotionEditVos));
      if(CollectionUtils.isNotEmpty(promotionEditVos)){
        result.add(promotionHitResp);
      }
    });
    //6、公共过滤器过滤
    result.forEach(rel ->  rel.setPromotions(this.commonFilter(rel.getPromotions(), param)));
    return result;
  }



  @Override
  public List<PromotionEditVo> findHitPromotionsByPromotionCodes(PromotionQueryReq param) {
    //1、校验必要参数非空
    if(param == null || CollectionUtil.listEmpty(param.getSaleProductVos()) || StringUtils.isEmpty(param.getAccountCode())
    || StringUtils.isEmpty(param.getAccountType())) {
      return Lists.newArrayList();
    }
    //TODO 该接口，暂时要求必须传入促销政策
//    ValidateUtils.notEmpty(param.getPromotionCodes(), "查询命中促销政策时，促销政策编码必须传入");
    //TODO 暂时只支持同时查一个促销政策
//    ValidateUtils.isTrue(param.getPromotionCodes().size() == 1, "只支持同时查一个促销政策");
    //2、首先筛选出客户、商品下的所有政策
    List<PromotionAvailableResp> availableResult = this.findAvailablePromotionsByAccount(param);
    List<PromotionEditVo> availableList = Lists.newArrayList();
    availableResult.forEach(hitResp -> availableList.addAll(hitResp.getPromotions()));
    //3、如果参数传入了备选政策编码，则需要根据这些编码筛选出可用政策
    List<PromotionEditVo> availableFilterList = availableList;
    if(CollectionUtil.listNotEmpty(param.getPromotionCodes())) {
      availableFilterList = availableList.stream().filter(promotion ->
              param.getPromotionCodes().contains(promotion.getPromotionPolicyCode())).collect(Collectors.toList());
    }
    Map<String, PromotionEditVo> promotionEditVoMap = availableFilterList.stream().collect(
            Collectors.toMap(PromotionEditVo::getPromotionPolicyCode, a -> a, (a, b) -> a)
    );

    //4、阶梯计算规则计算并过滤
    List<PromotionEditVo> result = promotionEditVoMap.values().stream().collect(Collectors.toList());
    for(PromotionEditVo promotionEditVo : result) {
      //处理本品赠品一一对应
//      if (Objects.equals(PromotionPolicyEunm.PromotionProductTypeEunm.CURRENT_RELATIONAL_GIFT.getCode(), promotionEditVo.getTemplateVo().getPromotionProduct())) {
//        this.computeOneToOne(promotionEditVo, param);
//        this.limitedCompute(promotionEditVo, param);
//        continue;
//      }
      this.compute(promotionEditVo, param);
      this.limitedCompute(promotionEditVo, param);
    }
    return result;
  }

  /**
   * 本品赠品一一对应计算（特殊）
   * @param promotionEditVo
   * @param param
   */
  private void computeOneToOne(PromotionEditVo promotionEditVo, PromotionQueryReq param) {
    if(CollectionUtil.mapEmpty(promotionEditVo.getProductMap()) || promotionEditVo == null) {
      return;
    }
    List<PromotionProductVo> currents = promotionEditVo.getProductMap().get(PromotionUtil.PRODUCT_CURRENTS);
    if(CollectionUtil.listEmpty(currents)) {
      return;
    }
    List<CalculateHitResultVo> calculateResultBos = Lists.newArrayList();
    for (PromotionProductVo current : currents) {
      PromotionProductVo product = new PromotionProductVo();
      product.setGiftRatio(BigDecimal.ONE);
      product.setProductCode(current.getProductCodeGift());
      product.setProductName(current.getProductNameGift());
      PromotionEditVo promotion = CrmBeanUtil.copy(promotionEditVo, PromotionEditVo.class);
      Map<String, List<PromotionProductVo>> productMap = Maps.newHashMap();
      productMap.put(PromotionUtil.PRODUCT_CURRENTS, Lists.newArrayList(current));
      productMap.put(PromotionUtil.PRODUCT_GIFTS, Lists.newArrayList(product));
      promotion.setProductMap(productMap);
      this.compute(promotion, param);
      CalculateHitResultVo hitResultVo = promotion.getCalculateHitResultVo();
      if(hitResultVo != null) {
        calculateResultBos.add(hitResultVo);
      }
    }
    if(CollectionUtil.listEmpty(calculateResultBos)) {
      return;
    }
    String valueType = null;
    BigDecimal value = null;
    String unitType = null;
    boolean matchState = false;
    String matchMsg = new String();
    List<PromotionRuleEditVo.ControlRow> ladderUsed = Lists.newArrayList();
    CalculateHitResultVo rel = new CalculateHitResultVo();
    for (CalculateHitResultVo bo : calculateResultBos) {
      valueType = bo.getValueType();
      if(bo.getValue() != null) {
        value = (value == null ? BigDecimal.ZERO : value).add(rel.getValue());
      }
      unitType = bo.getUnitType();
      if(bo.isMatchState()) {
        matchState = bo.isMatchState();
      }
      matchMsg = String.join(",", matchMsg, bo.getMatchMsg());
      if(CollectionUtil.listNotEmpty(bo.getLadderUsed())) {
        ladderUsed.addAll(bo.getLadderUsed());
      }
    }
    rel.setValue(value);
    rel.setValueType(valueType);
    rel.setUnitType(unitType);
    rel.setMatchState(matchState);
    rel.setMatchMsg(matchMsg);
    rel.setLadderUsed(ladderUsed);
    promotionEditVo.setCalculateHitResultVo(rel);
  }

  /**
   * 促销计算
   * @param promotionEditVo
   * @param param
   */
  private void compute(PromotionEditVo promotionEditVo, PromotionQueryReq param) {
    List<PromotionRuleVo> calculateRules = promotionEditVo.getRuleMap().get(RuleTypeDynamicEnum.CALCULATE.getObjectName());
    List<PromotionRuleVo> conditionRules = promotionEditVo.getRuleMap().get(RuleTypeDynamicEnum.CONDITION.getObjectName());
    String matchMsg = new String();
    boolean matchState = true;
    //5、起订量筛选
    String moqMatchMsg = moqFilter.moqApply(promotionEditVo, param.getSaleProductVos());
    if(!Objects.equals(MOQ_MATCH_MSG_OK, moqMatchMsg)) {
      promotionEditVo.setMatchState(false);
      matchMsg = String.join(";", matchMsg, moqMatchMsg);
      matchState = false;
    }
    if(CollectionUtil.listEmpty(conditionRules)) {
      promotionEditVo.setMatchState(false);
      matchMsg = String.join(";", matchMsg, "没有到有效的条件规则");
      matchState = false;
    }
    if(CollectionUtil.listEmpty(calculateRules)) {
      promotionEditVo.setMatchState(false);
      matchMsg = String.join(";", matchMsg, "没有到有效的计算规则");
      matchState = false;
    }
    //6、如果起订量不满足或者没有有效的条件和计算规则，则没有必要再继续走下去了
    if(!matchState) {
      promotionEditVo.setMatchMsg(matchMsg);
      return;
    }
    CalculateParamBo calculateParam = new CalculateParamBo();
    PromotionRuleVo calculateRule = calculateRules.get(0);
    PromotionRuleVo conditionRule = conditionRules.get(0);
    PromotionRuleEditVo calculateRuleEditVo = PromotionUtil.getRuleOneCache(calculateRule.getRuleCode(),
            PromotionUtil.getBean(PromotionRuleService.class));
    PromotionRuleEditVo conditionRuleEditVo = PromotionUtil.getRuleOneCache(conditionRule.getRuleCode(),
            PromotionUtil.getBean(PromotionRuleService.class));
    AbstractCalculateComputer calculateComputer = PromotionUtil.getBean(calculateRuleEditVo.getFuncBody(), AbstractCalculateComputer.class);
    CalculateResultBo calculateResult = null;
    if(calculateComputer != null) {
      calculateParam.setConditionRuleFunctionBeanName(conditionRuleEditVo.getFuncBody());
      calculateParam.setLadderList(PromotionUtil.parseLadderArray(conditionRule.getParams()));
      calculateParam.setPrices(Maps.newHashMap());
      calculateParam.setSaleProductVos(param.getSaleProductVos());
      calculateParam.setVariable(promotionEditVo.getUsedQtyUpper() == null ? new BigDecimal(Integer.MAX_VALUE) : promotionEditVo.getUsedQtyUpper());
      calculateParam.setPromotionCode(promotionEditVo.getPromotionPolicyCode());
      calculateParam.setCurrentProductVos(promotionEditVo.getProductMap().get(PromotionUtil.PRODUCT_CURRENTS));
      calculateParam.setGiftProductVos(promotionEditVo.getProductMap().get(PromotionUtil.PRODUCT_GIFTS));
      //7、计算
      calculateResult = calculateComputer.calculateApply(calculateParam);
      //设置赠品赠送数量
      this.buildGiftCount(calculateResult);
      if(calculateResult != null) {
        matchMsg = String.join(";", matchMsg, calculateResult.getMatchMsg());
        matchState = calculateResult.isMatchState();
      } else {
        matchState = false;
        matchMsg = "计算规则执行失败";
      }
    } else {
      matchState = false;
      matchMsg = "没有获取到有效的计算逻辑(相应计算规则函数配置不正确)";
    }
    promotionEditVo.setMatchState(matchState);
    promotionEditVo.setMatchMsg(matchMsg);
    promotionEditVo.setCalculateHitResultVo(CrmBeanUtil.copy(calculateResult, CalculateHitResultVo.class));
  }

  /**
   * 限量计算
   * @param promotionEditVo
   * @param param
   */
  private void limitedCompute(PromotionEditVo promotionEditVo, PromotionQueryReq param) {
    //8、限量规则
    boolean matchState = promotionEditVo.isMatchState();
    String matchMsg = promotionEditVo.getMatchMsg() == null ? new String() : promotionEditVo.getMatchMsg();
    List<PromotionRuleVo> limitedRules = promotionEditVo.getRuleMap().get(RuleTypeDynamicEnum.LIMITED.getObjectName());
    if(!CollectionUtil.listEmpty(limitedRules) && matchState) {
      List<LimitedResultBo> limitedResultBos = Lists.newArrayList();
      for (PromotionRuleVo limited : limitedRules) {
        PromotionRuleEditVo limitedRuleEditVo = PromotionUtil.getRuleOneCache(limited.getRuleCode(),
                PromotionUtil.getBean(PromotionRuleService.class));
        AbstractLimitedComputer limitedComputer = PromotionUtil.getBean(limitedRuleEditVo.getFuncBody(), AbstractLimitedComputer.class);
        if(limitedComputer != null && promotionEditVo.getCalculateHitResultVo() != null) {
          LimitedParamBo paramBo = new LimitedParamBo();
          paramBo.setGiftValue(promotionEditVo.getCalculateHitResultVo().getValue());
          paramBo.setPromotionCode(promotionEditVo.getPromotionPolicyCode());
          paramBo.setLadder(limited.getControlRows());
          paramBo.setAccountCode(param.getAccountCode());
          LimitedResultBo limitedResultBo = limitedComputer.limitedApply(paramBo);
          matchState = limitedResultBo.isValue();
          matchMsg = String.join(";", matchMsg, limitedResultBo.getMatchMsg());
          limitedResultBos.add(limitedResultBo);
        }
      }
      promotionEditVo.setLimitedResultBos(limitedResultBos);
    }
    if(matchMsg.startsWith(";")) {
      matchMsg = matchMsg.substring(1);
    }
    promotionEditVo.setMatchState(matchState);
    promotionEditVo.setMatchMsg(matchMsg);
  }

  /**
   * 根据比例设置赠品数量
   * @param calculateResult
   */
  public void buildGiftCount(CalculateResultBo calculateResult) {
    if(calculateResult == null || CollectionUtil.listEmpty(calculateResult.getGiftProductVos())
    || calculateResult.getValue() == null || StringUtils.isEmpty(calculateResult.getValueType())) {
      return;
    }
    BigDecimal value = calculateResult.getValue();
    String valueType = calculateResult.getValueType();
    List<PromotionProductVo> giftProductVos = calculateResult.getGiftProductVos();
    giftProductVos.forEach(g -> {
      if(AbstractConditionComputer.NUMBER.equals(valueType)) {
        g.setGiftCountMax((g.getGiftRatio() == null ? BigDecimal.ZERO : g.getGiftRatio()).multiply(value).setScale(0, BigDecimal.ROUND_FLOOR));
      } else if(AbstractConditionComputer.AMOUNT.equals(valueType)) {
        g.setGiftAmountMax((g.getGiftRatio() == null ? BigDecimal.ZERO : g.getGiftRatio()).multiply(value));
      }
    });
    BigDecimal sum = BigDecimal.ZERO;
    for (PromotionProductVo gift : giftProductVos) {
      if(AbstractConditionComputer.NUMBER.equals(valueType)) {
        sum = sum.add(gift.getGiftCountMax());
      } else if(AbstractConditionComputer.AMOUNT.equals(valueType)) {
        sum = sum.add(gift.getGiftAmountMax());
      }
    }
    PromotionProductVo scaleProductVo = giftProductVos.get(0);
    if(AbstractConditionComputer.NUMBER.equals(valueType)) {
      scaleProductVo.setGiftCountMax(value.subtract(sum).add(scaleProductVo.getGiftCountMax()));
    }
  }

  /**
   * 根据促销政策编码列表查询促销政策命中情况
   * @param params
   * @return
   */
  @Override
  public Map<String, PromotionEditVo> findHitPromotionMapByPromotionCodes(List<PromotionQueryReq> params) {
    if(CollectionUtil.listEmpty(params)) {
      return Maps.newHashMap();
    }
    params.forEach(p -> ValidateUtils.isTrue(p.getPromotionCodes() != null && p.getPromotionCodes().size() == 1, "每条参数有且只能有一个促销编码"));
    Map<String, PromotionEditVo> promotionEditVoMap = Maps.newHashMap();
    params.forEach(p -> {
      List<PromotionEditVo> promotionEditVos = this.findHitPromotionsByPromotionCodes(p);
      if(CollectionUtil.listEmpty(promotionEditVos)) {
        promotionEditVoMap.put(p.getPromotionCodes().get(0), null);
      } else {
        promotionEditVoMap.put(p.getPromotionCodes().get(0), promotionEditVos.get(0));
      }
    });
    return promotionEditVoMap;
  }

  /**
   * 过滤
   * @param promotionEditVos
   * @param queryReq
   * @return
   */
  protected List<PromotionEditVo> commonFilter(List<PromotionEditVo> promotionEditVos, PromotionQueryReq queryReq) {
    if(CollectionUtil.mapEmpty(commonFilterMap)) {
      return promotionEditVos;
    }
    Set<String> keySet = commonFilterMap.keySet();
    List<String> keyList = keySet.stream().sorted((o1, o2) -> {
      String reg = "^\\d+$";
      Pattern pattern = Pattern.compile(reg);
      Matcher matcher = pattern.matcher(o1);
      Matcher matcher2 = pattern.matcher(o2);
      //默认排最后
      int c1 = matcher.find() ? Integer.parseInt(matcher.group()) : Integer.MAX_VALUE;
      int c2 = matcher2.find() ? Integer.parseInt(matcher2.group()) : Integer.MAX_VALUE;
      return c2 > c1 ? 1 : -1;
    }).collect(Collectors.toList());
    for (String k : keyList) {
      promotionEditVos = commonFilterMap.get(k).filter(promotionEditVos, queryReq);
    }
    return promotionEditVos;
  }

  /**
   * 根据客户查询整单享受促销政策
   * @param param
   * @return
   */
  @Override
  public List<PromotionEditVo> findNoProductPromotionsByAccount(PromotionQueryReq param) {
    //TODO 无本品的情况，产品说后面做
    throw new RuntimeException("功能暂未提供");
  }

  /**
   * 根据客户查询是否有可用政策
   * @param param
   * @return
   */
  @Override
  public Map<String, Boolean> findAvailablePromotionMapByCus(PromotionQueryReq param) {
    if(param == null || StringUtils.isEmpty(param.getAccountCode())) {
      return Maps.newHashMap();
    }
    //1、如果用户类型为空，默认为客户类型
    if(StringUtils.isEmpty(param.getAccountType())) {
      param.setAccountType(AccountTypeDynamicEnum.CUSTOMER.getCode());
    }
    AccountTypeDynamicEnum accountTypeDynamicEnum = AccountTypeDynamicEnum.getAccountTypeEnumsByCode(param.getAccountType());
    if(accountTypeDynamicEnum == null || accountTypeDynamicEnum.getFilerBeanCls() == null) {
      return Maps.newHashMap();
    }
    Map<String, Boolean> result = Maps.newHashMap();
    //2、范围筛选
    AbstractScopeFilter filter = PromotionUtil.getBean(AbstractScopeFilter.class);
    if(filter == null) {
      return Maps.newHashMap();
    }
    List<String> promotionCodes = filter.apply(param);
    if(CollectionUtil.listEmpty(promotionCodes)) {
      return Maps.newHashMap();
    }
    List<PromotionEditVo> promotionEditVos = PromotionUtil.getCacheList(promotionCodes, promotionService);
    for (PromotionEditVo p : promotionEditVos) {
      boolean f = getFlag(p);
      List<PromotionProductVo> promotionProductVos = p.getProductMap().get(PromotionUtil.PRODUCT_CURRENTS);
      if(CollectionUtil.listNotEmpty(promotionProductVos)) {
        promotionProductVos.forEach(product ->
            result.put(product.getProductCode(), f));
      }
    }
    return result;
  }

  // 获取可用的促销政策
  private List<PromotionEditVo> filterAvailable(List<PromotionEditVo> list){
    List<PromotionEditVo> re = Lists.newArrayList();
    if(CollectionUtils.isEmpty(list)){
      return re;
    }
    for (PromotionEditVo p : list) {
      if(getFlag(p)){
        re.add(p);
      }
    }
    return re;
  }

  private Boolean getFlag(PromotionEditVo p) {
    if(p==null|| org.apache.commons.lang3.StringUtils.isBlank(p.getBeginTime())||org.apache.commons.lang3.StringUtils.isBlank(p.getEndTime())){
      return false;
    }
    boolean f = !CrmEnableStatusEnum.ENABLE.getCode().equals(p.getEnableStatus())
        || !CrmEnableStatusEnum.ENABLE.getCode().equals(p.getDelFlag())
        || YesNoEnum.YesNoCodeNumberEnum.YES.getCode(). equals(p.getNotShow());
    if(f){
      return false;
    }

    LocalDateTime begin = LocalDateTimeUtil.parse(p.getBeginTime(),"yyyy-MM-dd HH:mm:ss");
    LocalDateTime end = LocalDateTimeUtil.parse(p.getEndTime(),"yyyy-MM-dd HH:mm:ss");
    LocalDateTime now = LocalDateTime.now();
    long b1 = LocalDateTimeUtil.between(begin, now, ChronoUnit.SECONDS);
    long b2 = LocalDateTimeUtil.between(now, end, ChronoUnit.SECONDS);
    if(b1<0||b2<0){
      return false;
    }
    return true;
  }
}
