package com.biz.crm.promotion.service.npromotion.impl;

import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.nebular.dms.promotion.PromotionRuleEditVo;
import com.biz.crm.promotion.entity.PromotionPolicyRuleEntity;
import com.biz.crm.eunm.dms.RuleTypeDynamicEnum;
import com.biz.crm.promotion.mapper.PromotionPolicyRuleMapper;
import com.biz.crm.promotion.service.npromotion.PromotionRuleService;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionRuleVo;
import com.biz.crm.promotion.util.PromotionUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CommonFieldUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 促销规则逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/4/13 18:39
 */
@Service
@ConditionalOnMissingBean(name = "promotionRuleServiceExtend")
public class PromotionRuleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<PromotionPolicyRuleMapper, PromotionPolicyRuleEntity> implements PromotionRuleService {

  @Resource
  private PromotionPolicyRuleMapper promotionPolicyRuleMapper;

  /**
   * 批量保存促销规则，统一批次只能保存统一促销政策下的全量规则
   * 包含新增、编辑、删除
   * @param ruleVos
   * @param promotionPolicyCode
   * @return
   */
  @Override
  @Transactional
  public List<PromotionRuleVo> saveBatch(List<PromotionRuleVo> ruleVos, String promotionPolicyCode) {
    //1、校验参数非空
    if(CollectionUtil.listEmpty(ruleVos)) {
      return Lists.newArrayList();
    }
    //校验条件规则和计算规则阶梯结构和类型是否匹配
    ruleVos.forEach(rule -> {
      if(rule.getControlRows() != null) {
        rule.setParams(JSONArray.toJSONString(rule.getControlRows()));
      }
      if(RuleTypeDynamicEnum.CONDITION.getCode().equals(rule.getRuleType())
      || RuleTypeDynamicEnum.LIMITED.getCode().equals(rule.getRuleType())) {
        ValidateUtils.validate(rule.getParams(), "条件、限量规则必须传入阶梯控件");
        PromotionUtil.validateLadder(rule.getParams());
      }
    });
    List<PromotionRuleVo> oldRules = this.findByPromotionCode(promotionPolicyCode);
    //2、需要新增的记录
    List<PromotionRuleVo> createRules = ruleVos.stream().filter(r -> StringUtils.isEmpty(r.getId())).collect(Collectors.toList());
    List<PromotionPolicyRuleEntity> createEntities = CrmBeanUtil.copyList(createRules, PromotionPolicyRuleEntity.class);
    this.saveBatch(createEntities);
    //替换保存完成后的记录，用于返回结果
    ruleVos.removeAll(createRules);
    ruleVos.addAll(CrmBeanUtil.copyList(createEntities, PromotionRuleVo.class));
    //3、需要修改的记录
    List<PromotionRuleVo> updateRules = ruleVos.stream().filter(r -> StringUtils.isNotEmpty(r.getId())).collect(Collectors.toList());
    List<PromotionPolicyRuleEntity> updateEntities = CrmBeanUtil.copyList(updateRules, PromotionPolicyRuleEntity.class);
    this.updateBatchById(updateEntities);
    //4、需要删除的id列表
    List<String> deleteIds = CommonFieldUtil.difference(oldRules, ruleVos, "id");
    this.removeByIds(deleteIds);
    return ruleVos;
  }

  /**
   * 根据促销政策编码查询促销规则列表
   * @param promotionPolicyCode
   * @return
   */
  @Override
  public List<PromotionRuleVo> findByPromotionCode(String promotionPolicyCode) {
    if(StringUtils.isEmpty(promotionPolicyCode)) {
      return Lists.newArrayList();
    }
    List<PromotionPolicyRuleEntity> entities = this.promotionPolicyRuleMapper.selectList(Wrappers.<PromotionPolicyRuleEntity>lambdaQuery()
    .eq(PromotionPolicyRuleEntity::getPromotionPolicyCode, promotionPolicyCode));
    if(CollectionUtil.listEmpty(entities)) {
      return Lists.newArrayList();
    }
    List<PromotionRuleVo> vos = CrmBeanUtil.copyList(entities, PromotionRuleVo.class);
    vos.forEach(vo -> {
      if(StringUtils.isNotEmpty(vo.getParams())) {
        vo.setControlRows(JSONArray.parseArray(vo.getParams(), PromotionRuleEditVo.ControlRow.class));
      }
    });
    return vos;
  }
}
