package com.biz.crm.promotion.service.npromotion.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.activiti.ActivitiOperateTypeEnum;
import com.biz.crm.eunm.activiti.Indicator;
import com.biz.crm.eunm.dms.PromotionPolicyEunm;
import com.biz.crm.eunm.dms.ScopeTypeDynamicEnum;
import com.biz.crm.mdm.customer.MdmCustomerMsgFeign;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.activiti.act.ActivitiCallBackVo;
import com.biz.crm.nebular.activiti.vo.ActivitiBusinessVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionEditVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionProductVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionRuleVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionScopeSourceVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionScopeVo;
import com.biz.crm.nebular.dms.promotion.PromotionInfoRespVo;
import com.biz.crm.nebular.dms.promotion.PromotionPolicyQueryVo;
import com.biz.crm.nebular.dms.promotion.PromotionPolicyScopeVo;
import com.biz.crm.nebular.dms.promotion.PromotionPolicyTemplateVo;
import com.biz.crm.nebular.mdm.availablelistrule.AvailableListRuleVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.promotion.entity.PromotionPolicyEntity;
import com.biz.crm.promotion.entity.PromotionPolicyScopeEntity;
import com.biz.crm.promotion.mapper.PromotionPolicyMapper;
import com.biz.crm.promotion.mapper.PromotionPolicyScopeMapper;
import com.biz.crm.promotion.service.PromotionPolicyService;
import com.biz.crm.promotion.service.PromotionPolicyTemplateService;
import com.biz.crm.promotion.service.npromotion.PromotionProductService;
import com.biz.crm.promotion.service.npromotion.PromotionRuleService;
import com.biz.crm.promotion.service.npromotion.PromotionScopeService;
import com.biz.crm.promotion.service.npromotion.PromotionService;
import com.biz.crm.promotion.service.npromotion.validators.PromotionValidator;
import com.biz.crm.promotion.util.PromotionUtil;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.ActivityUtils;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CommonConstant;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.CrmDateUtils;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.UserRedis;
import com.biz.crm.util.UserUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 促销政策逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/4/13 16:43
 */
@Service
@ConditionalOnMissingBean(name = "promotionServiceExtend")
public class PromotionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<PromotionPolicyMapper, PromotionPolicyEntity> implements PromotionService {

  @Resource
  private PromotionPolicyMapper promotionPolicyMapper;
  @Resource
  private PromotionRuleService promotionRuleService;
  @Resource
  private PromotionScopeService promotionScopeService;
  @Resource
  private PromotionProductService promotionProductService;
  @Resource
  private PromotionValidator promotionValidator;
  @Resource
  private PromotionPolicyTemplateService promotionPolicyTemplateService;
  @Resource
  private PromotionPolicyScopeMapper policyScopeMapper;

  @Autowired
  private CrmLogSendUtil crmLogSendUtil;

  @Resource
  private MdmCustomerMsgFeign mdmCustomerMsgFeign;
  @Resource
  private MdmOrgFeign mdmOrgFeign;
  @Resource
  private MdmTerminalFeign mdmTerminalFeign;
  @Resource
  private PromotionPolicyService promotionPolicyService;
  @Autowired
  private RedisService redisService;

  /**
   * 新增促销政策
   * @param createVo
   * @return
   */
  @Override
  @Transactional
  public PromotionEditVo create(PromotionEditVo createVo) {
    //1、校验入参
    this.promotionValidator.validateCreate(createVo);
    createVo.setPromotionPolicyCode(CodeUtil.generateCode());
    if(createVo.getTemplateVo() != null && StringUtils.isNotEmpty(createVo.getTemplateVo().getPromotionType())) {
      createVo.setPromotionType(createVo.getTemplateVo().getPromotionType());
    }
    PromotionPolicyTemplateVo templateVo = this.promotionPolicyTemplateService.findByCode(createVo.getTemplateCode());
    ValidateUtils.validate(templateVo, "没有获取到促销模板");
    createVo.setTemplateVo(templateVo);
    PromotionPolicyEntity entity = CrmBeanUtil.copy(createVo, PromotionPolicyEntity.class);
    //2、保存促销政策
    boolean createFlag = this.save(entity);
    ValidateUtils.isTrue(createFlag, "创建促销政策失败");
    createVo.setId(entity.getId());
    //3、处理促销范围、规则、商品
    this.saveDetails(createVo);
    //通知可购清单更新
    this.promotionPolicyService.countBeginAndEndAffectedCustoms(DateUtil.getDate(DateUtil.DEFAULT_DATE_FORMAT), Lists.newArrayList(createVo.getId()));
    if (Objects.equals(createVo.getTemplateVo().getPromotionProduct(), PromotionPolicyEunm.PromotionProductTypeEunm.CURRENT_RELATIONAL_GIFT.getCode())
    && !Objects.equals(YesNoEnum.YesNoCodeNumberEnum.YES.getCode(), createVo.getNotShow())) {
      this.createSplitPromotion(createVo.getPromotionPolicyCode());
    }

    //日志记录
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    if(menuCodeObj!=null){
      PromotionEditVo newObject = this.findDetailsById(entity.getId());
      crmLogSendUtil.sendForAdd(menuCodeObj == null ? null : menuCodeObj.toString(), entity.getId(),entity.getId(),
              newObject);
    }
    return createVo;
  }

  /**
   * 修改促销政策
   * @param updateVo
   * @return
   */
  @Override
  @Transactional
  public PromotionEditVo update(PromotionEditVo updateVo) {
    //1、校验
    this.promotionValidator.validateUpdate(updateVo);
    PromotionPolicyEntity dbEntity = this.getById(updateVo.getId());
    ValidateUtils.validate(dbEntity, "没有获取到原始记录，请确认传入id是否正确");
    PromotionEditVo oldObject = this.findDetailsById(updateVo.getId());

    PromotionPolicyEntity entity = CrmBeanUtil.copy(updateVo, PromotionPolicyEntity.class);
    //2、更新促销政策
    boolean updateFlag = this.updateById(entity);
    ValidateUtils.isTrue(updateFlag, "更新促销政策失败");
    //3、处理促销范围、规则、商品
    this.saveDetails(updateVo);
    //4、删除缓存
    PromotionUtil.deleteCaches(Lists.newArrayList(updateVo.getPromotionPolicyCode()));
    //通知可购清单更新
    this.promotionPolicyService.countBeginAndEndAffectedCustoms(DateUtil.getDate(DateUtil.DEFAULT_DATE_FORMAT), Lists.newArrayList(updateVo.getId()));

    //编辑日志
    PromotionEditVo newObject = this.findDetailsById(updateVo.getId());
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    if(menuCodeObj!=null){
      crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
              updateVo.getId(),updateVo.getId(),oldObject,newObject);

    }
    return updateVo;
  }

  /**
   * 根据id查询促销政策详情
   * @param id
   * @return
   */
  @Override
  public PromotionEditVo findDetailsById(String id) {
    //1、校验入参
    if(StringUtils.isEmpty(id)) {
      return null;
    }
    //2、查询促销政策
    PromotionPolicyEntity entity = this.getById(id);
    if(entity == null) {
      return null;
    }
    PromotionEditVo editVo = PromotionUtil.getOneCache(entity.getPromotionPolicyCode(), this);
    return editVo;
  }

  /**
   * 根据促销编码查需详情
   * @param promotionCode
   * @return
   */
  @Override
  public PromotionEditVo findDetailsByCode(String promotionCode) {
    //1、校验入参
    if(StringUtils.isEmpty(promotionCode)) {
      return null;
    }
    //2、查询促销政策
    PromotionPolicyEntity entity = this.getOne(Wrappers.<PromotionPolicyEntity>lambdaQuery()
    .eq(PromotionPolicyEntity::getPromotionPolicyCode, promotionCode));
    if(entity == null) {
      return null;
    }
    PromotionEditVo editVo = CrmBeanUtil.copy(entity, PromotionEditVo.class);
    //3、查询促销范围
    List<PromotionScopeVo> scopeVos = this.promotionScopeService.findByPromotionCode(editVo.getPromotionPolicyCode());
    Map<String, List<PromotionScopeVo>> scopeMap = PromotionUtil.formatScopeMap(scopeVos);
    editVo.setScopeMap(scopeMap);
    //4、查询促销规则
    List<PromotionRuleVo> ruleVos = this.promotionRuleService.findByPromotionCode(editVo.getPromotionPolicyCode());
    Map<String, List<PromotionRuleVo>> ruleMap = PromotionUtil.formatRuleMap(ruleVos);
    editVo.setRuleMap(ruleMap);
    //5、查询促销商品
    List<PromotionProductVo> productVos = this.promotionProductService.findByPromotionCode(editVo.getPromotionPolicyCode());
    Map<String, List<PromotionProductVo>> productMap = PromotionUtil.formatProductMap(productVos);
    editVo.setProductMap(productMap);
    //6、查询促销模板主数据
    PromotionPolicyTemplateVo templateVo = this.promotionPolicyTemplateService.findByCode(editVo.getTemplateCode());
    editVo.setTemplateVo(templateVo);
    return editVo;
  }

  /**
   * 根据促销政策编码查询生效促销政策编码列表
   * @param promotionCodes
   * @return
   */
  @Override
  public List<String> findAvailableByCodes(List<String> promotionCodes) {
    if(CollectionUtil.listEmpty(promotionCodes)) {
      return Lists.newArrayList();
    }
    String date = DateUtil.formatDate();
    //日期进行对比，会导致当天的数据无法筛选到，加入详细时间
    date += " 00:00:00";
    List<PromotionPolicyEntity> entities = this.promotionPolicyMapper.selectList(
            Wrappers.<PromotionPolicyEntity>lambdaQuery()
            .in(PromotionPolicyEntity::getPromotionPolicyCode, promotionCodes)
            .le(PromotionPolicyEntity::getBeginTime, date)
            .ge(PromotionPolicyEntity::getEndTime, date)
    );
    if(CollectionUtil.listEmpty(entities)) {
      return Lists.newArrayList();
    }
    return entities.stream().map(PromotionPolicyEntity::getPromotionPolicyCode).collect(Collectors.toList());
  }

  /**
   * 处理促销范围、规则、商品(新增、修改、删除）
   * @param editVo
   */
  private void saveDetails(PromotionEditVo editVo) {
    //1、保存促销范围
    List<PromotionScopeVo> scopeVos = PromotionUtil.parseScopeMap(editVo);
    List<PromotionScopeVo> scopeResults = this.promotionScopeService.saveBatch(scopeVos, editVo.getPromotionPolicyCode());
    editVo.setScopeMap(PromotionUtil.formatScopeMap(scopeResults));
    //2、保存促销规则
    List<PromotionRuleVo> ruleVos = PromotionUtil.parseRuleMap(editVo);
    List<PromotionRuleVo> ruleResults = this.promotionRuleService.saveBatch(ruleVos, editVo.getPromotionPolicyCode());
    editVo.setRuleMap(PromotionUtil.formatRuleMap(ruleResults));
    //3、保存促销商品
    List<PromotionProductVo> productVos = PromotionUtil.parseProductMap(editVo);
    List<PromotionProductVo> productResults = this.promotionProductService.saveBatch(productVos, editVo.getPromotionPolicyCode());
    editVo.setProductMap(PromotionUtil.formatProductMap(productResults));
  }

  /**
   * 通过参数查询促销政策集合(目前需要支持通过促销政策编码集合promotionCodes和是否是查询有效的政策的标识queryEffectiveFlag)
   * @param reqVo
   * @return
   */
  @Override
  public PageResult<PromotionInfoRespVo> findPromotionsByParam(PromotionPolicyQueryVo reqVo) {
    Page<PromotionInfoRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
    page.setSearchCount(reqVo.getNeedPageFlag());
    if(reqVo.getQueryEffectiveFlag()){
      reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
    }
    List<PromotionInfoRespVo> list = promotionPolicyMapper.findPromotionsByParam(page,reqVo);
    if(CollectionUtils.isNotEmpty(list)&&reqVo.getQueryScopeFlag()){
      //查询政策范围数据
      List<String> codes = list.stream().map(PromotionInfoRespVo::getPromotionPolicyCode).collect(Collectors.toList());
      List<PromotionPolicyScopeEntity> policyScopeEntities = policyScopeMapper.selectList(Wrappers.lambdaQuery(PromotionPolicyScopeEntity.class).in(PromotionPolicyScopeEntity::getPromotionPolicyCode, codes));
      Map<String, List<PromotionPolicyScopeEntity>> map = policyScopeEntities.stream().collect(Collectors.groupingBy(PromotionPolicyScopeEntity::getPromotionPolicyCode));
      list.forEach(o->{
        List<PromotionPolicyScopeEntity> scopeEntities = map.get(o.getPromotionPolicyCode());
        if(CollectionUtils.isNotEmpty(scopeEntities)){
          List<PromotionPolicyScopeVo> promotionPolicyScopeVos = CrmBeanUtil.copyList(scopeEntities, PromotionPolicyScopeVo.class);
          o.setScopeList(promotionPolicyScopeVos);
        }else {
          o.setScopeList(Collections.EMPTY_LIST);
        }
      });
    }
    return PageResult.<PromotionInfoRespVo>builder()
            .data(list)
            .count(page.getTotal())
            .build();
  }

  /**
   * 分页查询促销范围数据源
   * @param param
   * @return
   */
  @Override
  public PageResult<PromotionScopeSourceVo> findScopeSourcePage(PromotionScopeSourceVo param) {
    if(param == null) {
      return PageResult.empty();
    }
    //客户
    if(Objects.equals(param.getScopeType(), ScopeTypeDynamicEnum.CUS.getCode())) {
      return this.findCusPage(param);
    }
    //客户组织
    if(Objects.equals(param.getScopeType(), ScopeTypeDynamicEnum.CUS_ORG.getCode())) {
      return this.findOrgPage(param);
    }
    //客户渠道
    if(Objects.equals(param.getScopeType(), ScopeTypeDynamicEnum.CUS_CHANNEL.getCode())){
      return this.findChannelPage(param);
    }
    //终端
    if(Objects.equals(param.getScopeType(), ScopeTypeDynamicEnum.TERMINAL.getCode())) {
      return this.findTerminalPage(param);
    }
    //终端组织
    if(Objects.equals(param.getScopeType(), ScopeTypeDynamicEnum.TERMINAL_ORG.getCode())) {
      return this.findOrgPage(param);
    }
    return PageResult.empty();
  }

  /**
   * 条件分页查询促销政策
   * @param param
   * @return
   */
  @Override
  public PageResult<PromotionEditVo> findPromotionPageByConditions(PromotionEditVo param) {
    if(param == null) {
      param = new PromotionEditVo();
    }
    LambdaQueryWrapper<PromotionPolicyEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.like(StringUtils.isNotEmpty(param.getPromotionPolicyCode()), PromotionPolicyEntity::getPromotionPolicyCode, param.getPromotionPolicyCode())
            .like(StringUtils.isNotEmpty(param.getPromotionPolicyName()), PromotionPolicyEntity::getPromotionPolicyName, param.getPromotionPolicyName())
            .ge(StringUtils.isNotEmpty(param.getBeginTime()), PromotionPolicyEntity::getBeginTime, param.getBeginTime())
            .le(StringUtils.isNotEmpty(param.getEndTime()), PromotionPolicyEntity::getEndTime, param.getEndTime())
            .ne(PromotionPolicyEntity::getNotShow, YesNoEnum.YesNoCodeNumberEnum.YES.getCode())
            .orderByDesc(PromotionPolicyEntity::getCreateDate, PromotionPolicyEntity::getCreateDateSecond);
    Page<PromotionPolicyEntity> pageResult = this.promotionPolicyMapper
            .selectPage(new Page(param.getPageNum(), param.getPageSize()), wrapper);
    return PageResult.<PromotionEditVo>builder().count(pageResult.getTotal())
            .data(CrmBeanUtil.copyList(pageResult.getRecords(), PromotionEditVo.class)).build();
  }


  /**
   * 分页查询渠道
   * @param param
   * @return
   */
  protected PageResult<PromotionScopeSourceVo> findChannelPage(PromotionScopeSourceVo param) {
    //TODO
    return PageResult.empty();
  }

  /**
   * 分页查询终端
   * @param param
   * @return
   */
  protected PageResult<PromotionScopeSourceVo> findTerminalPage(PromotionScopeSourceVo param) {
    MdmTerminalVo p = new MdmTerminalVo();
    p.setTerminalCode(param.getScopeCode());
    p.setTerminalName(param.getScopeName());
    p.setPageNum(param.getPageNum());
    p.setPageSize(param.getPageSize());
    PageResult<MdmTerminalVo> result = mdmTerminalFeign.page(p).getResult();
    List<MdmTerminalVo> list = result.getData() == null ? Lists.newArrayList() : result.getData();
    List<PromotionScopeSourceVo> scopes = Lists.newArrayList();
    list.forEach(li -> {
      PromotionScopeSourceVo scope = new PromotionScopeSourceVo();
      scope.setScopeCode(li.getTerminalCode());
      scope.setScopeName(li.getTerminalName());
      scope.setScopeType(param.getScopeType());
      scopes.add(scope);
    });
    return PageResult.<PromotionScopeSourceVo>builder()
            .count(result.getCount()).data(scopes).build();
  }

  /**
   * 分页查询组织
   * @param param
   * @return
   */
  protected PageResult<PromotionScopeSourceVo> findOrgPage(PromotionScopeSourceVo param) {
    MdmOrgReqVo p = new MdmOrgReqVo();
    p.setOrgCode(param.getScopeCode());
    p.setOrgName(param.getScopeName());
    p.setPageNum(param.getPageNum());
    p.setPageSize(param.getPageSize());
    PageResult<MdmOrgRespVo> result = mdmOrgFeign.pageList(p).getResult();
    List<MdmOrgRespVo> list = result.getData() == null ? Lists.newArrayList() : result.getData();
    List<PromotionScopeSourceVo> scopes = Lists.newArrayList();
    list.forEach(li -> {
      PromotionScopeSourceVo scope = new PromotionScopeSourceVo();
      scope.setScopeCode(li.getOrgCode());
      scope.setScopeName(li.getOrgName());
      scope.setScopeType(param.getScopeType());
      scopes.add(scope);
    });
    return PageResult.<PromotionScopeSourceVo>builder()
            .count(result.getCount()).data(scopes).build();
  }

  /**
   * 分页查询经销商
   * @param param
   * @return
   */
  protected PageResult<PromotionScopeSourceVo> findCusPage(PromotionScopeSourceVo param) {
    MdmCustomerMsgReqVo p = new MdmCustomerMsgReqVo();
    p.setCustomerCode(param.getScopeCode());
    p.setCustomerName(param.getScopeName());
    p.setPageNum(param.getPageNum());
    p.setPageSize(param.getPageSize());
    PageResult<MdmCustomerMsgRespVo> result = mdmCustomerMsgFeign.list(p).getResult();
    List<MdmCustomerMsgRespVo> list = result.getData() == null ? Lists.newArrayList() : result.getData();
    List<PromotionScopeSourceVo> scopes = Lists.newArrayList();
    list.forEach(li -> {
      PromotionScopeSourceVo scope = new PromotionScopeSourceVo();
      scope.setScopeCode(li.getCustomerCode());
      scope.setScopeName(li.getCustomerName());
      scope.setScopeType(param.getScopeType());
      scopes.add(scope);
    });
    return PageResult.<PromotionScopeSourceVo>builder()
            .count(result.getCount()).data(scopes).build();
  }


  @Override
  @Transactional
  public String submit(ActivitiBusinessVo activitiBusinessVo){
    PromotionPolicyEntity promotionPolicyEntity =  checkSubmit(activitiBusinessVo);
    String key = activitiBusinessVo.getBusinessId()+activitiBusinessVo.getTimestamp();
    try {
      UserRedis userRedis = UserUtils.getUser();
      //防止重复提交
      redisService.setSeconds(key,userRedis.getUsername(), RedisParam.TIME10);
      if(StringUtils.isEmpty(activitiBusinessVo.getCallBackFeign())) {
        activitiBusinessVo.setCallBackFeign("promotionFeign");
      }
      String no = ActivityUtils.startProcess(activitiBusinessVo);
      promotionPolicyEntity.setCrmProcessInstanceId(no);
      promotionPolicyEntity.setBpmSubmitDate(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmss));
      promotionPolicyEntity.setBpmSubmitBy(userRedis.getRealname());
      promotionPolicyEntity.setBpmStatus(ActivitiOperateTypeEnum.COMMIT.getCode());
      this.updateById(promotionPolicyEntity);
      return no;
    }finally {
      redisService.del(key);
    }
  }

  /**
   * 审批六回调
   * @param activitiCallBackVo
   */
  @Override
  @Transactional
  public void callback(ActivitiCallBackVo activitiCallBackVo) {
    ValidateUtils.validate(activitiCallBackVo, "审批回调时，参数不能为空");
    ValidateUtils.validate(activitiCallBackVo.getFormNo(), "审批回调时，促销政策id不能为空");
    PromotionPolicyEntity entity = this.getBaseMapper().selectById(activitiCallBackVo.getFormNo());
    ValidateUtils.validate(entity, "没有获取到原始促销政策记录，请确认传入的促销政策id是否正确");
    ValidateUtils.validate(activitiCallBackVo.getProcessState(), "审批状态不能为空");
    //通过
    if(Indicator.CON_BPM_DOING.getCode()==activitiCallBackVo.getProcessState()){
      entity.setBpmStatus("3");
      //创建本品赠品一一对应拆分的促销政策
      this.createSplitPromotion(entity.getPromotionPolicyCode());
    }
    //驳回
    if(Indicator.CON_BPM_PASS.getCode()==activitiCallBackVo.getProcessState()){
      entity.setBpmStatus("4");
    }
    //流程追回
    if(Indicator.CON_BPM_REJECT.getCode()==activitiCallBackVo.getProcessState()){
      entity.setBpmStatus("5");
    }
//    entity.setBpmStatus(Integer.toString(activitiCallBackVo.getProcessState()));
    this.updateById(entity);
    PromotionUtil.deleteCaches(Lists.newArrayList(entity.getPromotionPolicyCode()));
    if(Indicator.CON_BPM_DOING.getCode()==activitiCallBackVo.getProcessState()){
      //创建本品赠品一一对应拆分的促销政策
      this.createSplitPromotion(entity.getPromotionPolicyCode());
    }
  }

  /**
   * 生成拆分促销政策，如果促销政策商品类型不是本品赠品一一对应，则不做处理
   * @param promotionPolicyCode
   */
  @Override
  @Transactional
  public void createSplitPromotion(String promotionPolicyCode) {
    if(StringUtils.isEmpty(promotionPolicyCode)) {
      return;
    }
    PromotionEditVo promotionEditVo = PromotionUtil.getOneCache(promotionPolicyCode, this);
    ValidateUtils.validate(promotionEditVo, "没有获取到原始促销政策记录");
    ValidateUtils.validate(promotionEditVo.getTemplateVo(), "没有获取到该促销政策关联的促销引擎记录");
    //如果不是本品赠品一一对应的促销政策，则不做处理
    if(!Objects.equals(promotionEditVo.getTemplateVo().getPromotionProduct(),
            PromotionPolicyEunm.PromotionProductTypeEunm.CURRENT_RELATIONAL_GIFT.getCode())) {
      return;
    }
    if(CollectionUtil.mapEmpty(promotionEditVo.getProductMap())) {
      return;
    }
    List<PromotionProductVo> productVos = promotionEditVo.getProductMap().get(PromotionUtil.PRODUCT_CURRENTS);
    PromotionProductVo[] currents = new PromotionProductVo[productVos.size()];
    PromotionProductVo[] gifts = new PromotionProductVo[productVos.size()];
    for (int i = 0; i < productVos.size(); i++) {
      PromotionProductVo productVo = productVos.get(i);
      PromotionProductVo current = CrmBeanUtil.copy(productVo, PromotionProductVo.class);
      PromotionProductVo gift = CrmBeanUtil.copy(productVo, PromotionProductVo.class);
      //本品
      current.setProductCodeGift(null);
      current.setProductNameGift(null);
      current.setCurrentProduct(CommonConstant.GLOBAL.YesOrNo.Y.getItemCode());
      current.setGiftRatio(null);
      current.setId(null);
      current.setPromotionPolicyCode(null);
      current.setPromotionPolicyId(null);
      current.setPromotionPolicyName(null);
      currents[i] = current;
      //赠品
      gift.setProductCode(gift.getProductCodeGift());
      gift.setProductName(gift.getProductNameGift());
      gift.setProductCodeGift(null);
      gift.setProductNameGift(null);
      gift.setId(null);
      gift.setPromotionPolicyCode(null);
      gift.setPromotionPolicyId(null);
      gift.setPromotionPolicyName(null);
      gift.setCurrentProduct(CommonConstant.GLOBAL.YesOrNo.N.getItemCode());
      gifts[i] = gift;
    }
    for(int i = 0; i < currents.length; i ++) {
      PromotionEditVo nPromotion = CrmBeanUtil.copy(promotionEditVo, PromotionEditVo.class);
      Map<String, List<PromotionProductVo>> productMap = Maps.newHashMap();
      productMap.put(PromotionUtil.PRODUCT_CURRENTS, Lists.newArrayList(currents[i]));
      productMap.put(PromotionUtil.PRODUCT_GIFTS, Lists.newArrayList(gifts[i]));
      nPromotion.setProductMap(productMap);
      nPromotion.setPromotionPolicyCode(null);
      nPromotion.setId(null);
      nPromotion.setParentCode(promotionEditVo.getPromotionPolicyCode());
      nPromotion.setNotShow(YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
      //设置促销范围
      PromotionUtil.parseScopeMap(nPromotion).forEach(s -> s.setId(null));
      //设置促销规则
      PromotionUtil.parseRuleMap(nPromotion).forEach(r -> r.setId(null));
      this.create(nPromotion);
    }

  }

  public PromotionPolicyEntity checkSubmit(ActivitiBusinessVo activitiBusinessVo){
    AssertUtils.isNotEmpty(activitiBusinessVo.getBusinessId(),"业务主键id[businessId]不能为空");
    AssertUtils.isNotEmpty(activitiBusinessVo.getBusinessNo(),"业务对象编码[businessNo]不能为空");
    AssertUtils.isNotEmpty(activitiBusinessVo.getProcessKey(),"流程key[processKey]不能为空");
    AssertUtils.isNotEmpty(activitiBusinessVo.getProcessKey(),"流程名称[processName]不能为空");
    AssertUtils.isNotEmpty(activitiBusinessVo.getTimestamp(),"时间戳[timestamp]不能为空");
    //检查数据是否已经提交
    PromotionPolicyEntity promotionPolicyEntity =  this.getById(activitiBusinessVo.getBusinessId());
    if(promotionPolicyEntity==null){
      throw new BusinessException("业务单据不存在["+activitiBusinessVo.getBusinessId()+"]");
    }
    if(ActivitiOperateTypeEnum.COMMIT.getCode().equals(promotionPolicyEntity.getBpmStatus())||
            ActivitiOperateTypeEnum.PASS.getCode().equals(promotionPolicyEntity.getBpmStatus())
    ){
      throw new BusinessException("操作失败数据处理审批中或者审批通过");
    }
    String key = activitiBusinessVo.getBusinessId()+activitiBusinessVo.getTimestamp();
    Object o = redisService.get(key);
    if(o!=null){
      throw new BusinessException("流程正在被"+o.toString()+"提交，请勿重复提交");
    }
    return promotionPolicyEntity;
  }

}
