package com.biz.crm.promotion.util;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.base.BusinessException;
import com.biz.crm.constant.dms.RedisPathConstant;
import com.biz.crm.nebular.dms.promotion.PromotionRuleEditVo;
import com.biz.crm.eunm.dms.RuleTypeDynamicEnum;
import com.biz.crm.eunm.dms.ScopeTypeDynamicEnum;
import com.biz.crm.promotion.service.PromotionRuleService;
import com.biz.crm.promotion.service.npromotion.PromotionService;
import com.biz.crm.promotion.service.npromotion.beans.AbstractLadderService;
import com.biz.crm.nebular.dms.npromotion.bo.LadderParseBo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionEditVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionProductVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionRuleVo;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionScopeVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CommonConstant;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;

/**
 * 促销相关工具类
 * @Author: chenrong
 * @Date: 2021/4/12 15:25
 */
@Slf4j
public class PromotionUtil {

  //** 本品对象列表名称 **//
  public static final String PRODUCT_CURRENTS = "currentProducts";
  //** 赠品对象列表名称 **//
  public static final String PRODUCT_GIFTS = "giftProducts";
  //** 促销政策保存在redis中的基础路径 **//
  protected static final String PROMOTION_REDIS_PATH_BASE = "crm:dms:promotion:promotion:";
  //** 促销政策规则redis中的基础历经 **//
  protected static final String PROMOTION_RULE_REDIS_PATH_BASE = "crm:dms:promotion:rule:";

  public static ApplicationContext applicationContext;

  /**
   * 根据促销商品类型转化促销商品参数对象名称
   * @param productType
   * @return
   */
  public static String formatProductObjectName(String productType) {
    if (Objects.equals(productType, CommonConstant.GLOBAL.YesOrNo.Y.getItemCode())) {
      return PRODUCT_CURRENTS;
    } else {
      return PRODUCT_GIFTS;
    }
  }

  /**
   * 解析促销范围参数
   * map -> list
   * @param promotionEditVo
   * @return
   */
  public static List<PromotionScopeVo> parseScopeMap(PromotionEditVo promotionEditVo) {
    //1、校验入参非空
    if (Objects.isNull(promotionEditVo) || CollectionUtil.mapEmpty(promotionEditVo.getScopeMap())) {
      return Lists.newArrayList();
    }
    Map<String, List<PromotionScopeVo>> scopeMap = promotionEditVo.getScopeMap();
    List<PromotionScopeVo> scopeVos = Lists.newArrayList();
    //2、遍历键值对，根据促销范围枚举枚举，设置每条范围数据类型，并且重新统一放入一个list
    scopeMap.forEach((k, v) -> {
      ScopeTypeDynamicEnum scopeTypeDynamicEnum = ScopeTypeDynamicEnum.getScopeTypeEnumsByObjectName(k);
      ValidateUtils.validate(scopeTypeDynamicEnum, "根据促销范围参数名称【%s】没有获取到字典编码，请确认参数是否正确传入", k);
      ValidateUtils.isTrue(Objects.equals(scopeTypeDynamicEnum.getContainName(), k) || Objects.equals(scopeTypeDynamicEnum.getUnContainName(), k),
              "促销范围参数传入错误，没有匹配到参数名：[%s]，正确的参数名应该是scope + 字典编码 + Y|N", k);
      if (CollectionUtil.listNotEmpty(v)) {
        v.forEach(scope -> {
          scope.setContainable(k.substring(k.length() - 1));
          scope.setScopeType(scopeTypeDynamicEnum.getCode());
          scope.setPromotionPolicyId(promotionEditVo.getId());
          scope.setPromotionPolicyCode(promotionEditVo.getPromotionPolicyCode());
          scope.setPromotionPolicyName(promotionEditVo.getPromotionPolicyName());
          scopeVos.add(scope);
        });
      }
    });
    return scopeVos;
  }

  /**
   * 解析促销规则参数
   * @param promotionEditVo
   * @return
   */
  public static List<PromotionRuleVo> parseRuleMap(PromotionEditVo promotionEditVo) {
    //1、校验入参非空
    if (Objects.isNull(promotionEditVo) || CollectionUtil.mapEmpty(promotionEditVo.getRuleMap())) {
      return Lists.newArrayList();
    }
    Map<String, List<PromotionRuleVo>> ruleMap = promotionEditVo.getRuleMap();
    List<PromotionRuleVo> ruleVos = Lists.newArrayList();
    //2、遍历键值对，根据字典匹配每条记录类型，如果统一放到一个list中
    ruleMap.forEach((k, v) -> {
      RuleTypeDynamicEnum ruleTypeDynamicEnum = RuleTypeDynamicEnum.getRuleTypeEnumByObjectName(k);
      ValidateUtils.validate(ruleTypeDynamicEnum, "根据促销规则参数名称【%s】没有获取到字典编码，请确认参数是否正确传入", k);
      ValidateUtils.isTrue(Objects.equals(ruleTypeDynamicEnum.getObjectName(), k),
              "促销规则参数传入错误，没有匹配到参数名：[%s]，正确的参数名应该是 字典编码 + s", k);
      if (CollectionUtil.listNotEmpty(v)) {
        v.forEach(rule -> {
          rule.setRuleType(ruleTypeDynamicEnum.getCode());
          rule.setPromotionPolicyId(promotionEditVo.getId());
          rule.setPromotionPolicyCode(promotionEditVo.getPromotionPolicyCode());
          rule.setPromotionPolicyTemplateId(promotionEditVo.getTemplateId());
          rule.setPromotionPolicyTemplateCode(promotionEditVo.getTemplateCode());
          ruleVos.add(rule);
        });
      }
    });
    return ruleVos;
  }

  /**
   * 解析促销商品参数
   * @param promotionEditVo
   * @return
   */
  public static List<PromotionProductVo> parseProductMap(PromotionEditVo promotionEditVo) {
    if (Objects.isNull(promotionEditVo) || CollectionUtil.mapEmpty(promotionEditVo.getProductMap())) {
      return Lists.newArrayList();
    }
    Map<String, List<PromotionProductVo>> productMap = promotionEditVo.getProductMap();
    List<PromotionProductVo> productVos = Lists.newArrayList();
    productMap.forEach((k, v) -> {
      if (CollectionUtil.listEmpty(v)) {
        return;
      }
      v.forEach(product -> {
        product.setPromotionPolicyId(promotionEditVo.getId());
        product.setPromotionPolicyCode(promotionEditVo.getPromotionPolicyCode());
        product.setPromotionPolicyName(promotionEditVo.getPromotionPolicyName());
        if (Objects.equals(PRODUCT_CURRENTS, k)) {
          product.setCurrentProduct(CommonConstant.GLOBAL.YesOrNo.Y.getItemCode());
          productVos.add(product);
        } else if (Objects.equals(PRODUCT_GIFTS, k)) {
          product.setCurrentProduct(CommonConstant.GLOBAL.YesOrNo.N.getItemCode());
          productVos.add(product);
        }
      });
    });
    return productVos;
  }

  /**
   * 转化促销范围记录，使其成为匹配表单的显示数据结构
   * @param scopeVos
   * @return
   */
  public static Map<String, List<PromotionScopeVo>> formatScopeMap(List<PromotionScopeVo> scopeVos) {
    if (CollectionUtil.listEmpty(scopeVos)) {
      return Maps.newHashMap();
    }
    Map<String, List<PromotionScopeVo>> scopeMap = scopeVos.stream().collect(
            Collectors.groupingBy(scope -> ScopeTypeDynamicEnum.buildObjectName(scope.getScopeType(), scope.getContainable()))
    );
    if (CollectionUtil.mapEmpty(scopeMap)) {
      return Maps.newHashMap();
    }
    return scopeMap;
  }

  /**
   * 转化促销规则记录，使其成为匹配表单的显示数据结构
   * @param ruleVos
   * @return
   */
  public static Map<String, List<PromotionRuleVo>> formatRuleMap(List<PromotionRuleVo> ruleVos) {
    if (CollectionUtil.listEmpty(ruleVos)) {
      return Maps.newHashMap();
    }
    Map<String, List<PromotionRuleVo>> ruleMap = ruleVos.stream().collect(
            Collectors.groupingBy(rule -> RuleTypeDynamicEnum.buildObjectName(rule.getRuleType()))
    );
    if (CollectionUtil.mapEmpty(ruleMap)) {
      return Maps.newHashMap();
    }
    return ruleMap;
  }

  /**
   * 转化促销商品记录，使其成为匹配表单的显示数据结构
   * @param productVos
   * @return
   */
  public static Map<String, List<PromotionProductVo>> formatProductMap(List<PromotionProductVo> productVos) {
    if (CollectionUtil.listEmpty(productVos)) {
      return Maps.newHashMap();
    }
    Map<String, List<PromotionProductVo>> productMap;
//    //特殊情况，兼容老版，本品赠品一一对应
//    List<PromotionProductVo> otherVos = productVos.stream().filter(v -> StringUtils.isNotEmpty(v.getProductCodeGift())).collect(Collectors.toList());
//    //如果存在本品上保存赠品的记录，则表示该促销正式是 本品赠品一一对应
//    if(CollectionUtil.listNotEmpty(otherVos)) {
//      otherVos.forEach(vo -> {
//
//      });
//    }
    productMap = productVos.stream().collect(
            Collectors.groupingBy(product -> formatProductObjectName(product.getCurrentProduct()))
    );
    if (CollectionUtil.mapEmpty(productMap)) {
      return Maps.newHashMap();
    }
    return productMap;
  }

  /**
   * 查询单条促销政策
   * 优先从缓存获取，如果缓存没有，则从数据库中查询，并放入缓存
   * @param promotionCode
   * @param promotionService
   * @return
   */
  public static PromotionEditVo getOneCache(String promotionCode, PromotionService promotionService) {
    if (StringUtils.isEmpty(promotionCode)) {
      return null;
    }
    PromotionEditVo vo = RedisCacheUtil.get(getKey(promotionCode));
    if (Objects.isNull(vo)) {
      vo = promotionService.findDetailsByCode(promotionCode);
      setOneCache(promotionCode, vo);
    }
    return vo;
  }

  /**
   * 根据促销政策编码批量获取缓存对象
   * @param promotionCodes
   * @param promotionService
   * @return
   */
  public static List<PromotionEditVo> getCacheList(List<String> promotionCodes, PromotionService promotionService) {
    if (CollectionUtil.listEmpty(promotionCodes)) {
      return Lists.newArrayList();
    }
    //TODO 因为缓存是在查询的时候单条设置到缓存的，所以批量获取缓存可能存在数据不全的情况，在解决这个问题之前，暂时使用循环获取缓存的方式
    List<PromotionEditVo> list = Lists.newArrayList();
    promotionCodes.forEach(code -> {
      PromotionEditVo oneCache = getOneCache(code, promotionService);
      if (oneCache == null) {
        return;
      }
      list.add(oneCache);
    });
    return list;
  }

  /**
   * 设置单条促销政策缓存
   * @param promotionCode
   * @param vo
   */
  public static void setOneCache(String promotionCode, PromotionEditVo vo) {
    if (StringUtils.isEmpty(promotionCode) || Objects.isNull(vo)) {
      return;
    }
    RedisCacheUtil.setByDays(getKey(promotionCode), vo);
  }

  /**
   * 批量删除缓存
   * @param promotionCodes
   */
  public static void deleteCaches(List<String> promotionCodes) {
    if (CollectionUtil.listEmpty(promotionCodes)) {
      return;
    }
    List<String> keys = Lists.newArrayList();
    promotionCodes.forEach(code -> {
      keys.add(getKey(code));
    });

    RedisCacheUtil.delete(keys);
  }

  /**
   * 设置一条规则缓存
   * @param ruleCode
   * @param ruleVo
   */
  public static void setRuleOneCache(String ruleCode, PromotionRuleEditVo ruleVo) {
    if(StringUtils.isEmpty(ruleCode)) {
      return;
    }
    RedisCacheUtil.setByDays(getRuleKey(ruleCode), ruleVo);
  }

  /**
   * 获取一条规则缓存
   * @param ruleCode
   * @return
   */
  public static PromotionRuleEditVo getRuleOneCache(String ruleCode, PromotionRuleService service) {
    if(StringUtils.isEmpty(ruleCode)) {
      return null;
    }
    PromotionRuleEditVo ruleVo = RedisCacheUtil.get(getRuleKey(ruleCode));
    if(ruleVo == null) {
      ruleVo = service.findByRuleCode(ruleCode);
      setRuleOneCache(ruleCode, ruleVo);
    }
    return ruleVo;
  }

  /**
   * 根据名称和类型获取ioc实例
   * @param name
   * @param cls
   * @return
   */
  public static<T> T getBean(String name, Class<T> cls) {
    T bean;
    try {
      bean = applicationContext.getBean(name,cls);
    } catch (Exception e) {
      bean = null;
      log.error("根据名称和类型获取ioc实例失败", e);
    }
    return bean;
  }

  /**
   * 根据类型从ioc中获取生效的实例
   * @param cls
   * @return
   */
  public static<T> T getBean(Class<T> cls) {
    T bean;
    try {
      bean = applicationContext.getAutowireCapableBeanFactory().getBean(cls);
    } catch (Exception e) {
      bean = null;
      log.error("根据类型从ioc中获取生效的实例失败", e);
    }
    return bean;
  }

  /**
   * 阶梯解析
   * @param text
   * @return
   */
  public static List<PromotionRuleEditVo.ControlRow> parseLadderArray(String text) {
    if(StringUtils.isEmpty(text)) {
      return Lists.newArrayList();
    }
    return JSONArray.parseArray(text, PromotionRuleEditVo.ControlRow.class);
  }

  /**
   * 获取促销政策缓存key
   * @param key
   * @return
   */
  private static String getKey(String key) {
    if (StringUtils.isEmpty(key)) {
      key = "none";
    }
    return PROMOTION_REDIS_PATH_BASE.concat(key);
  }

  /**
   * 获取促销规则缓存key
   * @param key
   * @return
   */
  private static String getRuleKey(String key) {
    if (StringUtils.isEmpty(key)) {
      key = "none";
    }
    return PROMOTION_RULE_REDIS_PATH_BASE.concat(key);
  }

  /**
   * 获取赠品key
   * @param key
   * @return
   */
  private static String getGiftKey(String key) {
    if(StringUtils.isEmpty(key)) {
      key = "none";
    }
    return RedisPathConstant.PROMOTION_GIFT_REDIS_PATH_BASE.concat(key);
  }

  /**
   * 从redis种获取赠品使用情况
   * @param promotionCode
   * @return
   */
  public static BigDecimal getGift(String accountCode, String promotionCode) {
    if(StringUtils.isEmpty(promotionCode) || StringUtils.isEmpty(accountCode)) {
      return BigDecimal.ZERO;
    }
    BigDecimal s = RedisCacheUtil.get(getGiftKey(accountCode.concat("-").concat(promotionCode)));
    if(s == null) {
      return BigDecimal.ZERO;
    }
    return s;
  }

  /**
   * 校验阶梯控件格式和类型是否匹配
   * @param text
   */
  public static void validateLadder(String text) {
    List<PromotionRuleEditVo.ControlRow> controlRows = parseLadderArray(text);
    controlRows.forEach(controlRow -> {
      validateLadder(controlRow);
    });
  }
  /**
   * 校验阶梯控件格式和类型是否匹配
   * @param controlRow
   */
  public static void validateLadder(PromotionRuleEditVo.ControlRow controlRow) {
    LadderParseBo ladderParseBo = parseLadderByType(controlRow.getControls(), controlRow.getControlType());
    ValidateUtils.validate(ladderParseBo, "控件格式与类型不匹配");
  }

  /**
   * 根据控件类型解析控件
   * 一共提供了8中类型，
   * 如果有拓展枚举，不同类型，请重写{@link AbstractLadderService}parseLadderByType方法，追加一个类型处理 即可
   * @param controls
   * @param type
   * @return
   */
  public static LadderParseBo parseLadderByType(List<PromotionRuleEditVo.KeyValParamControl> controls, String type) {
    try {
      AbstractLadderService ladderService = getBean(AbstractLadderService.class);
      return ladderService.parseLadderByType(controls, type);
    }catch (Exception e) {
      log.error(e.getMessage(), e);
      throw new BusinessException("控件格式与类型不匹配", e);
    }
  }

  /**
   * 将Object类型转化为指定类型
   * @param value
   * @return
   */
  public static BigDecimal parseBigDecimalFromObj(Object value, Class cls) {
    if(value == null) {
      return null;
    }
    return new BigDecimal((String) value);
  }

  /**
   * 将Object类型转化为指定类型
   * @param value
   * @return
   */
  public static String parseStringFromObj(Object value, Class cls) {
    if(value == null) {
      return null;
    }
    return (String) value;
  }

}
