package com.biz.crm.salecontract.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.dms.salecontract.ContractTemplateVo;
import com.biz.crm.nebular.dms.salecontract.ContractTupleVo;
import com.biz.crm.salecontract.entity.ContractTemplateEntity;
import com.biz.crm.salecontract.mapper.ContractTemplateMapper;
import com.biz.crm.salecontract.service.ContractTemplateService;
import com.biz.crm.salecontract.service.ContractTemplateTupleMappingService;
import com.biz.crm.salecontract.util.ContractTemplateUtil;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import java.util.List;
import java.util.Objects;
import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 合同模板逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/2/5 11:27
 */
@ConditionalOnMissingBean(name = "contractTemplateServiceImpl")
@Service(value = "contractTemplateService")
public class ContractTemplateServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<ContractTemplateMapper, ContractTemplateEntity> implements ContractTemplateService {

  @Resource
  private ContractTemplateMapper contractTemplateMapper;
  @Resource
  private ContractTemplateTupleMappingService contractTemplateTupleMappingService;
  @Autowired
  private CrmLogSendUtil crmLogSendUtil;
  /**
   * 新增合同模板
   * 1、校验入参
   * 2、保存模板
   * 3、保存元组模板关系
   * @param contractTemplateVo
   * @return
   */
  @Override
  @Transactional
  public ContractTemplateVo add(ContractTemplateVo contractTemplateVo) {
    //1、
    ContractTemplateUtil.validateAdd(contractTemplateVo);
    contractTemplateVo.setCode(CodeUtil.generateCode());
    ContractTemplateEntity entity = CrmBeanUtil.copy(contractTemplateVo, ContractTemplateEntity.class);
    this.validateName(contractTemplateVo.getName(), null);
    //2、
    entity.setId(null);
    this.contractTemplateMapper.insert(entity);
    contractTemplateVo.setId(entity.getId());
    //3、
    this.contractTemplateTupleMappingService.replace(contractTemplateVo.getTuples(), contractTemplateVo.getCode());
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    ContractTemplateVo logVo = this.findDetailsByCode(contractTemplateVo.getCode());
    crmLogSendUtil.sendForAdd(menuCodeObj.toString(),logVo.getId(),logVo.getCode(),logVo);
    return contractTemplateVo;
  }

  /**
   * 校验模板名称是否重复
   * @param name
   * @param id
   */
  private void validateName(String name, String id) {
    if(StringUtils.isEmpty(name)) {
      return;
    }
    List<ContractTemplateEntity> entities = this.contractTemplateMapper.selectList(
            Wrappers.<ContractTemplateEntity>query().eq("name", name)
            .ne(StringUtils.isNotEmpty(id), "id", id)
    );
    ValidateUtils.isTrue(CollectionUtil.listEmpty(entities), "模板名称【%s】重复", name);
  }

  /**
   * 编辑合同模板
   * 1、校验入参
   * 2、修改模板
   * 3、更新模板/元组映射
   * @param contractTemplateVo
   * @return
   */
  @Override
  @Transactional
  public ContractTemplateVo edit(ContractTemplateVo contractTemplateVo) {
    //1、
    ContractTemplateUtil.validateEdit(contractTemplateVo);
    this.validateName(contractTemplateVo.getName(), contractTemplateVo.getId());
    //2、
    ContractTemplateEntity entity = CrmBeanUtil.copy(contractTemplateVo, ContractTemplateEntity.class);
    ContractTemplateVo oldObject = this.findDetailsByCode(contractTemplateVo.getCode());
    boolean editResult = this.updateById(entity);
    ValidateUtils.isTrue(editResult, "编辑模板失败，请检查id是否正确传入");
    //3、
    this.contractTemplateTupleMappingService.replace(contractTemplateVo.getTuples(), contractTemplateVo.getCode());
    ContractTemplateVo newObject = this.findDetailsByCode(contractTemplateVo.getCode());
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newObject.getId(),newObject.getCode(),oldObject,newObject);
    return contractTemplateVo;
  }

  /**
   * 根据编码查询合同模板
   * @param code
   * @return
   */
  @Override
  public ContractTemplateVo findByCode(String code) {
    if(StringUtils.isEmpty(code)) {
      return null;
    }
    QueryWrapper wrapper = Wrappers.query().eq("code", code);
    ContractTemplateEntity entity = this.contractTemplateMapper.selectOne(wrapper);
    return CrmBeanUtil.copy(entity, ContractTemplateVo.class);
  }

  /**
   * 分页条件查询合同模板
   * @param contractTemplateVo
   * @return
   */
  @Override
  public PageResult<ContractTemplateVo> findPageByConditions(ContractTemplateVo contractTemplateVo) {
    QueryWrapper wrapper = Wrappers.query()
            .like(!Objects.isNull(contractTemplateVo.getContractType()), "contract_type", contractTemplateVo.getContractType())
            .like(StringUtils.isNotEmpty(contractTemplateVo.getName()), "name", contractTemplateVo.getName())
            .like(StringUtils.isNotEmpty(contractTemplateVo.getEnableStatus()), "enable_status", contractTemplateVo.getEnableStatus())
            .like(StringUtils.isNotEmpty(contractTemplateVo.getOrgCode()), "org_code", contractTemplateVo.getOrgCode())
            .like(StringUtils.isNotEmpty(contractTemplateVo.getOrgName()), "org_name", contractTemplateVo.getOrgName())
            .orderByDesc("create_date").orderByDesc("create_date_second");
    Page<ContractTemplateVo> page = new Page<>(contractTemplateVo.getPageNum(), contractTemplateVo.getPageSize());
    List<ContractTemplateVo> pageList = this.contractTemplateMapper.findPageByConditions(page, wrapper);
    return PageResult.<ContractTemplateVo>builder().data(pageList).count(page.getTotal()).build();
  }

  /**
   * 根据模板编码查询详情
   * @param code 模板编码
   * @return
   */
  @Override
  public ContractTemplateVo findDetailsByCode(String code) {
    if(StringUtils.isEmpty(code)) {
      return null;
    }
    QueryWrapper wrapper = Wrappers.query().eq("code", code);
    ContractTemplateEntity entity = this.getOne(wrapper);
    ContractTemplateVo vo = CrmBeanUtil.copy(entity, ContractTemplateVo.class);
    if(vo == null) {
      return null;
    }
    List<ContractTupleVo> tupleVos = this.contractTemplateTupleMappingService.findTupleByTemplateCode(code);
    vo.setTuples(tupleVos);
    return vo;
  }

  /**
   * 批量启用模板
   * @param ids
   */
  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    if(CollectionUtil.listEmpty(ids)) {
      return;
    }
    UpdateWrapper<ContractTemplateEntity> wrapper = Wrappers.<ContractTemplateEntity>update()
            .set("enable_status", CrmEnableStatusEnum.ENABLE.getCode())
            .in("id", ids);
    this.update(wrapper);
  }

  /**
   * 批量禁用模板
   * @param ids
   */
  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    if(CollectionUtil.listEmpty(ids)) {
      return;
    }
    UpdateWrapper<ContractTemplateEntity> wrapper = Wrappers.<ContractTemplateEntity>update()
            .set("enable_status", CrmEnableStatusEnum.DISABLE.getCode())
            .in("id", ids);
    this.update(wrapper);
  }
}
