package com.biz.crm.salecontract.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.dms.salecontract.ContractTemplateTupleMappingVo;
import com.biz.crm.nebular.dms.salecontract.ContractTupleVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalVo;
import com.biz.crm.salecontract.entity.ContractTemplateEntity;
import com.biz.crm.salecontract.entity.ContractTemplateTupleMappingEntity;
import com.biz.crm.salecontract.mapper.ContractTemplateMapper;
import com.biz.crm.salecontract.mapper.ContractTemplateTupleMappingMapper;
import com.biz.crm.salecontract.service.ContractTemplateTupleMappingService;
import com.biz.crm.salecontract.util.ContractTemplateUtil;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 合同模板元组逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/2/5 11:52
 */
@ConditionalOnMissingBean(name = "contractTemplateTupleMappingServiceImpl")
@Service(value = "contractTemplateTupleMappingService")
public class ContractTemplateTupleMappingServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<ContractTemplateTupleMappingMapper, ContractTemplateTupleMappingEntity> implements ContractTemplateTupleMappingService {

  @Resource
  private ContractTemplateTupleMappingMapper contractTemplateTupleMappingMapper;

  /**
   * 新增/更新模板元组映射
   * 1、校验入参
   * 2、处理映射关系
   * @param tupleVos
   * @param templateCode
   * @return
   */
  @Override
  @Transactional
  public List<ContractTemplateTupleMappingVo> replace(List<ContractTupleVo> tupleVos, String templateCode) {
    //1、
    ValidateUtils.validate(templateCode, "新增/更新模板元组映射时，模板编码不能为空");
    if (CollectionUtil.listEmpty(tupleVos)) {
      return Lists.newArrayList();
    }
    //2、
    List<ContractTemplateTupleMappingVo> dbMappings = this.findMappingByTemplateCode(templateCode);
    List<ContractTemplateTupleMappingVo> dbEnableMapping = dbMappings.stream().filter(mapping -> Objects.equals(CrmEnableStatusEnum.ENABLE.getCode(), mapping.getDelFlag())).collect(Collectors.toList());
    List<ContractTemplateTupleMappingVo> dbDisabledMapping = dbMappings.stream().filter(mapping -> Objects.equals(CrmEnableStatusEnum.DISABLE.getCode(), mapping.getDelFlag())).collect(Collectors.toList());
    List<ContractTemplateTupleMappingVo> mappingVos = ContractTemplateUtil.buildTemplateTupleMappingVo(tupleVos, templateCode);
    //需要新增的映射
    List<String> addTupleCodes = this.difference(mappingVos, dbMappings);
    List<ContractTemplateTupleMappingVo> addMappings = mappingVos.stream().filter(mapping -> addTupleCodes.contains(mapping.getTupleCode())).collect(Collectors.toList());
    addMappings.forEach(mapping -> mapping.setIndexCode(CodeUtil.getCode()));
    this.saveBatch(CrmBeanUtil.copyList(addMappings, ContractTemplateTupleMappingEntity.class));
    //需要修改的映射
    List<ContractTemplateTupleMappingVo> updateTuples = this.beMixed(dbMappings, mappingVos);
    this.updateBatchById(CrmBeanUtil.copyList(updateTuples, ContractTemplateTupleMappingEntity.class));
    //需要禁用的映射
    List<String> disabledTupleCodes = this.difference(dbEnableMapping, mappingVos);
    List<ContractTemplateTupleMappingVo> disabledMappings = dbEnableMapping.stream().filter(mapping -> disabledTupleCodes.contains(mapping.getTupleCode())).collect(Collectors.toList());
    disabledMappings.forEach(mapping -> mapping.setDelFlag(CrmEnableStatusEnum.DISABLE.getCode()));
    this.updateBatchById(CrmBeanUtil.copyList(disabledMappings, ContractTemplateTupleMappingEntity.class));
    //需要启用的映射
    List<String> enableTupleCodes = Lists.newArrayList(mappingVos.stream().collect(Collectors.toMap(ContractTemplateTupleMappingVo::getTupleCode, a -> a, (a, b) -> b)).keySet());
    List<ContractTemplateTupleMappingVo> enableMappings = dbDisabledMapping.stream().filter(mapping -> enableTupleCodes.contains(mapping.getTupleCode())).collect(Collectors.toList());
    enableMappings.forEach(mapping -> mapping.setDelFlag(CrmEnableStatusEnum.ENABLE.getCode()));
    this.updateBatchById(CrmBeanUtil.copyList(enableMappings, ContractTemplateTupleMappingEntity.class));
    return mappingVos;
  }

  /**
   * 列表(a)与列表(b)取交集
   * 只有a、b列表同时存在某个特定字段值相同，才会被返回到结果列表
   * @param source
   * @param target
   * @return
   */
  private List<ContractTemplateTupleMappingVo> beMixed(List<ContractTemplateTupleMappingVo> source, List<ContractTemplateTupleMappingVo> target) {
    Map<String, ContractTemplateTupleMappingVo> sourceMap = source.stream().collect(Collectors.toMap(ContractTemplateTupleMappingVo::getTupleCode, a -> a, (c, d) -> d));
    Map<String, ContractTemplateTupleMappingVo> targetMap = target.stream().collect(Collectors.toMap(ContractTemplateTupleMappingVo::getTupleCode, a -> a, (c, d) -> d));
    List<ContractTemplateTupleMappingVo> result = Lists.newArrayList();
    sourceMap.forEach((k, v) -> {
      if(targetMap.containsKey(k)) {
        ContractTemplateTupleMappingVo targetValue = targetMap.get(k);
        targetValue.setId(v.getId());
        result.add(targetValue);
      }
    });
    return result;
  }

  /**
   * 列表（a）与列表（b）求差集（c）
   * c = a - b
   * 1、构建源id列表
   * 2、构建目标id列表
   * 3、求差集
   * @param dbVos
   * @param vos
   * @return
   */
  private List<String> difference(List<ContractTemplateTupleMappingVo> dbVos, List<ContractTemplateTupleMappingVo> vos) {
    Set<String> dbTupleCodes = dbVos.stream().collect(Collectors.groupingBy(ContractTemplateTupleMappingVo::getTupleCode)).keySet();
    Set<String> tupleCodes = vos.stream().filter(vo -> com.biz.crm.util.StringUtils.isNotEmpty(vo.getTupleCode()))
            .collect(Collectors.groupingBy(ContractTemplateTupleMappingVo::getTupleCode)).keySet();
    Set<String> differenceIds = Sets.difference(dbTupleCodes, tupleCodes);
    if(!CollectionUtil.collectionNotEmpty(differenceIds)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(differenceIds);
  }

  /**
   * 根据合同模板编码查询合同元组
   * @param templateCode
   * @return
   */
  @Override
  public List<ContractTupleVo> findTupleByTemplateCode(String templateCode) {
    if(StringUtils.isEmpty(templateCode)) {
      return Lists.newArrayList();
    }
    List<ContractTupleVo> contractTupleVos = this.contractTemplateTupleMappingMapper.findTupleByTemplateCode(templateCode);
    return contractTupleVos;
  }

  /**
   * 根据模板编码查询模板/元组映射列表
   * @param templateCode
   * @return
   */
  @Override
  public List<ContractTemplateTupleMappingVo> findMappingByTemplateCode(String templateCode) {
    if(StringUtils.isEmpty(templateCode)) {
      return Lists.newArrayList();
    }
    QueryWrapper wrapper = Wrappers.query().eq("template_code", templateCode);
    List<ContractTemplateTupleMappingEntity> entities = this.contractTemplateTupleMappingMapper.selectList(wrapper);
    return CrmBeanUtil.copyList(entities, ContractTemplateTupleMappingVo.class);
  }

  /**
   * 根据元组编码查询映射列表
   * @param tupleCode
   * @return
   */
  @Override
  public List<ContractTemplateTupleMappingVo> findByTupleCode(String tupleCode) {
    if(StringUtils.isEmpty(tupleCode)) {
      return Lists.newArrayList();
    }
    QueryWrapper wrapper = Wrappers.<ContractTemplateTupleMappingEntity>query().eq("tuple_code", tupleCode);
    List<ContractTemplateTupleMappingEntity> entities = this.list(wrapper);
    if(CollectionUtil.listEmpty(entities)) {
      return Lists.newArrayList();
    }
    return CrmBeanUtil.copyList(entities, ContractTemplateTupleMappingVo.class);
  }
}
