package com.biz.crm.salecontract.service.listener.impl;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.eunm.dms.SaleGoalEunm;
import com.biz.crm.nebular.dms.salecontract.SaleContractVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalVo;
import com.biz.crm.salecontract.service.listener.AbstractContractModelListener;
import com.biz.crm.salegoal.service.SaleGoalService;
import com.biz.crm.util.BeanCopyUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import java.util.List;
import javax.annotation.Resource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 销量目标
 * @Author: chenrong
 * @Date: 2021/2/24 10:27
 */
@Service("contractSaleGoalListener")
@ConditionalOnMissingBean(name = "saleGoalContractListenerExtend")
public class ContractSaleGoalListener implements AbstractContractModelListener {

  @Resource
  private SaleGoalService saleGoalService;

  /**
   * 保存
   * @param data         对象数据（列表）
   * @param contractCode 合同编码
   * @param jsonName     数据对象名称，一般情况下不会用到，如果多个模块实现同一个接口，该参数可以做区分使用
   * @param saleContractVo
   * @param indexCode
   * @return
   */
  @Override
  @Transactional
  public JSONArray add(JSONArray data, String contractCode, String jsonName, SaleContractVo saleContractVo, String indexCode) {
    //1、校验入参
    if(CollectionUtil.jsonArrayEmpty(data)) {
      return new JSONArray();
    }
    ValidateUtils.validate(contractCode, "合同编码不能为空");
    //2、解析参数
    List<SaleGoalVo> saleGoalVos = BeanCopyUtil.parseByJSONArray(data.getJSONArray(0), SaleGoalVo.class);
    if(CollectionUtil.listNotEmpty(saleGoalVos)) {
      saleGoalVos.forEach(vo -> this.buildBaseInfo(vo, saleContractVo));
    }
    //3、保存数据
    saleGoalVos = saleGoalService.replace(saleGoalVos, contractCode, indexCode, saleContractVo);
    return BeanCopyUtil.formatJSONArray(saleGoalVos);
  }

  /**
   * 编辑
   * @param data         对象数据（列表）
   * @param contractCode 合同编码
   * @param jsonName     数据对象名称，一般情况下不会用到，如果多个模块实现同一个接口，该参数可以做区分使用
   * @param saleContractVo
   * @param indexCode
   * @return
   */
  @Override
  @Transactional
  public JSONArray edit(JSONArray data, String contractCode, String jsonName, SaleContractVo saleContractVo, String indexCode) {
    //1、校验入参
    ValidateUtils.validate(contractCode, "合同编码不能为空");
    List<SaleGoalVo> saleGoalVos = BeanCopyUtil.parseByJSONArray(data.getJSONArray(0), SaleGoalVo.class);
    if(CollectionUtil.listNotEmpty(saleGoalVos)) {
      saleGoalVos.forEach(vo -> this.buildBaseInfo(vo, saleContractVo));
    }
    //2、更新数据
    saleGoalVos = saleGoalService.replace(saleGoalVos, contractCode, indexCode, saleContractVo);
    return BeanCopyUtil.formatJSONArray(saleGoalVos);
  }

  /**
   * 根据合同查询
   * @param contractCode 合同编码
   * @param jsonName     数据对象名称，一般情况下不会用到，如果多个模块实现同一个接口，该参数可以做区分使用
   * @param indexCode
   * @return
   */
  @Override
  public JSONArray findByContractCode(String contractCode, String jsonName, String indexCode) {
    if(StringUtils.isEmpty(contractCode)) {
      return new JSONArray();
    }
    List<SaleGoalVo> saleGoalVos = this.saleGoalService.findDetailsByContractCodeAndIndexCode(contractCode, indexCode);
    return new JSONArray(BeanCopyUtil.formatJSONArray(saleGoalVos));
  }

  /**
   * 从合同中获取基础数据
   * @param vo
   * @param saleContractVo
   */
  private void buildBaseInfo(SaleGoalVo vo, SaleContractVo saleContractVo) {
    if (vo != null && saleContractVo != null) {
      vo.setObjectType(SaleGoalEunm.ObjectType.CUSTOMER.getCode());
      vo.setCusCode(saleContractVo.getCusCode());
      vo.setCusName(saleContractVo.getCusName());
      vo.setCusChannelCode(saleContractVo.getChannelCode());
      vo.setCusChannelName(saleContractVo.getChannelName());
      vo.setCusOrgCode(saleContractVo.getOrgCode());
      vo.setCusOrgName(saleContractVo.getOrgName());
    }
  }
}
