package com.biz.crm.salegoal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.dms.SaleGoalEunm;
import com.biz.crm.nebular.dms.salecontract.SaleContractVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalItemVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalRatioVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalVo;
import com.biz.crm.salegoal.entity.SaleGoalEntity;
import com.biz.crm.salegoal.mapper.SaleGoalMapper;
import com.biz.crm.salegoal.service.SaleGoalItemService;
import com.biz.crm.salegoal.service.SaleGoalRatioService;
import com.biz.crm.salegoal.service.SaleGoalService;
import com.biz.crm.salegoal.utils.SaleGoalItemUtil;
import com.biz.crm.salegoal.utils.SaleGoalUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/9/15 16:11
 **/
@ConditionalOnMissingBean(name = "saleGoalServiceImpl")
@Service(value = "saleGoalService")
public class SaleGoalServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SaleGoalMapper, SaleGoalEntity> implements SaleGoalService {

  @Resource
  private SaleGoalMapper saleGoalMapper;

  @Resource
  private SaleGoalItemService saleGoalItemService;

  @Resource
  private SaleGoalRatioService saleGoalRatioService;

  @Autowired
  private CrmLogSendUtil crmLogSendUtil;
  /**
   * 1、验证
   * 2、组装
   * 3、分批次保存并且保存月目标
   * @param saleGoalVos
   */
  @Transactional
  @Override
  public List<SaleGoalVo> addBatch(List<SaleGoalVo> saleGoalVos) {
    //1
    this.validateForAddBatch(saleGoalVos);

    //2
    Map<String, SaleGoalItemVo> saleGoalItemMap = new HashMap<>(saleGoalVos.size());
    List<SaleGoalEntity> entities = new ArrayList<>(saleGoalVos.size());
    for (SaleGoalVo vo : saleGoalVos) {
      SaleGoalItemVo goalItemVo = vo.getSaleGoalItemVo();
      ValidateUtils.validate(goalItemVo, "目标明细不能为空");
      SaleGoalEntity entity = SaleGoalUtil.packageEntityForAdd(vo);
      goalItemVo.setSaleGoalId(entity.getId());
      goalItemVo.setTaskType(entity.getTaskType());
      entities.add(entity);
      saleGoalItemMap.put(entity.getId(), goalItemVo);
    }

    //3
    List<SaleGoalEntity> currentEntities = new ArrayList<>();
    List<SaleGoalItemVo> currentItemVos = new ArrayList<>(saleGoalVos.size());
    int entityI = 0;
    for (SaleGoalEntity entity : entities) {
      entityI++;
      currentEntities.add(entity);
      currentItemVos.add(saleGoalItemMap.get(entity.getId()));
      if (currentEntities.size() == 500 || entityI == entities.size()) {
        this.saveBatch(currentEntities);
        saleGoalItemService.addBatch(currentItemVos);
        currentEntities.clear();
        currentItemVos.clear();
      }
    }
    List<SaleGoalVo> resultVos = CrmBeanUtil.copyList(entities, SaleGoalVo.class);
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    for(SaleGoalVo vo : resultVos){
      crmLogSendUtil.sendForAdd(menuCodeObj.toString(),vo.getId(),vo.getId(),vo);
    }
    return CrmBeanUtil.copyList(entities, SaleGoalVo.class);
  }

  /**
   * AddBatch的验证方法
   * 1、验证参数
   * 2、设置onlyKey值并验证是否数据库存在
   * 3、构建并抛出已存在数据的相关异常信息
   * @param saleGoalVos
   */
  private void validateForAddBatch(List<SaleGoalVo> saleGoalVos) {
    //1
    SaleGoalUtil.validateParamForAddBatch(saleGoalVos);

    //2
    SaleGoalUtil.buildOnlyKey(saleGoalVos);
    List<String> params = saleGoalVos.stream().map(SaleGoalVo::getOnlyKey).collect(Collectors.toList());
    //避免大数据量导入，所以分批次查询
    List<String> currentParam = new ArrayList<>();
    List<SaleGoalEntity> exsits = new ArrayList<>();
    int i = 0;
    Integer type = saleGoalVos.get(0).getType();
    Integer objectType = saleGoalVos.get(0).getObjectType();
    for (String param : params) {
      i++;
      currentParam.add(param);
      if (currentParam.size() == 500 || i == params.size()) {
        QueryWrapper<SaleGoalEntity> wrapper = new QueryWrapper<>();
        wrapper.in("only_key", currentParam);
        List<SaleGoalEntity> entities = saleGoalMapper.selectList(wrapper);
        if (!CollectionUtils.isEmpty(entities)) {
          exsits.addAll(entities);
        }
        currentParam.clear();
      }
    }
    //3、
    if (!CollectionUtils.isEmpty(exsits)) {
      StringBuilder stb = new StringBuilder();
      for (SaleGoalEntity entity : exsits) {
        if(SaleGoalEunm.ObjectType.CUSTOMER.getCode().equals(objectType)) {
          stb.append(entity.getCusName()).append("-");
        } else if(SaleGoalEunm.ObjectType.ORGANIZATION.getCode().equals(objectType)) {
          stb.append(entity.getOrgName()).append("-");
        } else if(SaleGoalEunm.ObjectType.TERMINAL.getCode().equals(objectType)) {
          stb.append(entity.getTerminalName()).append("-");
        }
        if (type.intValue() == SaleGoalEunm.GoalType.GOODS.getCode().intValue()) {
          stb.append(entity.getGoodsName()).append("-").append(entity.getTargetYear()).append(";");
        } else if (type.intValue() == SaleGoalEunm.GoalType.PRODUCT_LEVEL.getCode().intValue()) {
          stb.append(entity.getProductLevelName()).append("-").append(entity.getTargetYear()).append(";");
        } else {
          stb.append(entity.getTargetYear()).append(";");
        }
      }
      StringBuilder msgPre = new StringBuilder("销量目标已经存在,年份为：");
      if (type.intValue() != SaleGoalEunm.GoalType.ROUTINE.getCode().intValue()) {
        msgPre = new StringBuilder("销量目标已经存在,产品/产品层级和年份为: ");
      }
      throw new BusinessException(msgPre.append(stb).toString());
    }
  }

  /**
   * 1、验证
   * 2、更新总价
   * 3、更新月目标详情
   * @param saleGoalVo
   */
  @Transactional
  @Override
  public void edit(SaleGoalVo saleGoalVo) {
    //1
    ValidateUtils.validate(saleGoalVo.getId(), "请指定要编辑的销售目标!");
    SaleGoalEntity entity = saleGoalMapper.selectById(saleGoalVo.getId());
    ValidateUtils.validate(entity, "您要编辑的销售目标不存在或者已经被删除!");
    SaleGoalVo oldObject = findById(saleGoalVo.getId());
    //2
    saleGoalVo.getSaleGoalItemVo().setTaskType(saleGoalVo.getTaskType());
    entity.setTargetNum(SaleGoalItemUtil.sumSingleItem(saleGoalVo.getSaleGoalItemVo()).getYearlySum());
    entity.setName(saleGoalVo.getName());
    entity.setTaskType(saleGoalVo.getTaskType());
    saleGoalMapper.updateById(entity);

    //3
    saleGoalItemService.update(saleGoalVo.getSaleGoalItemVo());
    SaleGoalVo newObject = findById(saleGoalVo.getId());
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), saleGoalVo.getId(),saleGoalVo.getId(),oldObject,newObject);

  }

  /**
   * 1、查询主干信息
   * 2、查询月份详情
   * @param id
   * @return
   */
  @CrmDictMethod
  @Override
  public SaleGoalVo findById(String id) {
    SaleGoalVo vo = null;
    //1
    ValidateUtils.validate(id, "请指定需要查询的销售目标！");
    SaleGoalEntity entity = saleGoalMapper.selectById(id);
    if (null == entity) {
      return vo;
    }
    vo = new SaleGoalVo();
    BeanUtils.copyProperties(entity, vo);
    vo.setTargetNumStr(vo.getTargetNum().stripTrailingZeros().toPlainString());

    //2
    vo.setSaleGoalItemVo(saleGoalItemService.findBySaleGoalId(vo.getId()));

    return vo;
  }

  @CrmDictMethod
  @Override
  public PageResult<SaleGoalVo> list(SaleGoalVo saleGoalVo) {
    QueryWrapper<SaleGoalVo> wrapper = Wrappers.<SaleGoalVo>query()
            .like(!StringUtils.isEmpty(saleGoalVo.getCusName()), "cus_name", saleGoalVo.getCusName())
            .like(!StringUtils.isEmpty(saleGoalVo.getCusCode()), "cus_code", saleGoalVo.getCusCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getCusOrgName()), "cus_org_name", saleGoalVo.getCusOrgName())
            .like(!StringUtils.isEmpty(saleGoalVo.getCusOrgCode()), "cus_org_code", saleGoalVo.getCusOrgCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getCusChannelName()), "cus_channel_name", saleGoalVo.getCusChannelName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getCusChannelCode()), "cus_channel_code", saleGoalVo.getCusChannelCode())
            .eq(null != saleGoalVo.getType(), "type", saleGoalVo.getType())
            .eq(null != saleGoalVo.getTargetYear(), "target_year", saleGoalVo.getTargetYear())
            .eq(!StringUtils.isEmpty(saleGoalVo.getName()), "name", saleGoalVo.getName())
            .like(!StringUtils.isEmpty(saleGoalVo.getGoodsName()), "goods_name", saleGoalVo.getGoodsName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getGoodsCode()), "goods_code", saleGoalVo.getGoodsCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getProductLevelName()), "product_level_name", saleGoalVo.getProductLevelName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getProductLevelCode()), "product_level_code", saleGoalVo.getProductLevelCode())

            .like(!StringUtils.isEmpty(saleGoalVo.getOrgCode()), "org_code", saleGoalVo.getOrgCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getOrgName()), "org_name", saleGoalVo.getOrgName())
            .eq(!Objects.isNull(saleGoalVo.getObjectType()), "object_type", saleGoalVo.getObjectType())
            .eq(!Objects.isNull(saleGoalVo.getTaskType()), "task_type", saleGoalVo.getTaskType())
            .like(!StringUtils.isEmpty(saleGoalVo.getTerminalCode()), "terminal_code", saleGoalVo.getTerminalCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getTerminalName()), "terminal_name", saleGoalVo.getTerminalName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getContractCode()), "contract_code", saleGoalVo.getContractCode())
            .orderByDesc("create_date", "create_date_second");
    Page<SaleGoalVo> page = PageUtil.buildPage(saleGoalVo.getPageNum(), saleGoalVo.getPageSize());
    List<SaleGoalVo> list = saleGoalMapper.list(page, wrapper);
    if (!CollectionUtils.isEmpty(list)) {
      Map<String, SaleGoalItemVo> detailMap = saleGoalItemService.findBySaleGoalIds(list.stream().map(SaleGoalVo::getId).collect(Collectors.toList()));
      for (SaleGoalVo vo : list) {
        vo.setTargetNumStr(vo.getTargetNum().stripTrailingZeros().toPlainString());
        vo.setSaleGoalItemVo(detailMap.get(vo.getId()));
      }
    }
    return PageResult.<SaleGoalVo>builder()
            .data(list)
            .count(page.getTotal())
            .build();
  }

  @Transactional
  @Override
  public void delByIds(ArrayList<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    List<SaleGoalVo> vos = this.findByIds(ids);
    ValidateUtils.validate(vos,"请指定需要查询的销售目标！");
    saleGoalMapper.deleteBatchIds(ids);
    vos.forEach(vo -> {
      crmLogSendUtil.sendForDel(menuCodeObj.toString(),vo.getId(),vo.getId(),vo);
    });
  }

  @Transactional
  @Override
  public void delByParam(SaleGoalVo saleGoalVo) {
    if (null == saleGoalVo) {
      throw new BusinessException("请指定查询条件!");
    }
    QueryWrapper<SaleGoalEntity> wrapper = Wrappers.<SaleGoalEntity>query()
            .like(!StringUtils.isEmpty(saleGoalVo.getCusName()), "cus_name", saleGoalVo.getCusName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getCusCode()), "cus_code", saleGoalVo.getCusCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getCusOrgName()), "cus_org_name", saleGoalVo.getCusOrgName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getCusOrgCode()), "cus_org_code", saleGoalVo.getCusOrgCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getCusChannelName()), "cus_channel_name", saleGoalVo.getCusChannelName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getCusChannelCode()), "cus_channel_code", saleGoalVo.getCusChannelCode())
            .eq(null != saleGoalVo.getType(), "type", saleGoalVo.getType())
            .eq(null != saleGoalVo.getTargetYear(), "target_year", saleGoalVo.getTargetYear())
            .like(!StringUtils.isEmpty(saleGoalVo.getGoodsName()), "goods_name", saleGoalVo.getGoodsName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getGoodsCode()), "goods_code", saleGoalVo.getGoodsCode())
            .like(!StringUtils.isEmpty(saleGoalVo.getProductLevelName()), "product_level_name", saleGoalVo.getProductLevelName())
            .eq(!StringUtils.isEmpty(saleGoalVo.getProductLevelCode()), "product_level_code", saleGoalVo.getProductLevelCode());
    List<SaleGoalEntity> entities = saleGoalMapper.selectList(wrapper);
    List<SaleGoalVo> copies = CrmBeanUtil.copyList(entities, SaleGoalVo.class);
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    saleGoalMapper.delete(wrapper);
    copies.forEach(copy -> crmLogSendUtil.sendForDel(menuCodeObj.toString(),copy.getId(),copy.getId(),copy));

  }

  /**
   * 根据合同编码跟新销售目标
   * @param saleGoalVos
   * @param contractCode
   * @param indexCode
   * @param saleContractVo
   * @return
   */
  @Override
  public List<SaleGoalVo> replace(List<SaleGoalVo> saleGoalVos, String contractCode, String indexCode, SaleContractVo saleContractVo) {
    ValidateUtils.validate(contractCode, "合同编码不能为空");
    //1、删除老数据
//    this.deleteByContractCode(contractCode);
    //2、参数判空
    if(CollectionUtil.listEmpty(saleGoalVos)) {
      return Lists.newArrayList();
    }
    //按分摊比例分配到月份
    saleGoalVos.forEach(vo -> {
      SaleGoalRatioVo ratioVo = this.saleGoalRatioService.findMonthRatio(vo.getTargetYear(), saleContractVo);
      SaleGoalUtil.sharingMonthByYear(ratioVo, vo);
    });
    List<SaleGoalVo> dbVos = this.findDetailsByContractCodeAndIndexCode(contractCode, indexCode);
    //3、分别处理新增、修改、删除数据
    //需要创建的记录
    List<SaleGoalVo> addVos = saleGoalVos.stream().filter(vo -> StringUtils.isEmpty(vo.getId())).collect(Collectors.toList());
    addVos.forEach(sale -> {
      sale.setContractCode(contractCode);
      sale.setIndexCode(indexCode);
    });
    if(CollectionUtil.listNotEmpty(addVos)) {
      this.addBatch(addVos);
    }
    //需要修改的记录
    List<SaleGoalVo> updateVos = saleGoalVos.stream().filter(vo -> !StringUtils.isEmpty(vo.getId())).collect(Collectors.toList());
    updateVos.forEach(vo -> this.edit(vo));
    //需要删除的记录
    List<String> deleteIds = this.difference(dbVos, saleGoalVos);
    this.delByIds(new ArrayList<>(deleteIds));
    return saleGoalVos;
  }

  /**
   * 数据库中的公式列表（a）与前端传入的公式列表（b）求差集（c）
   * c = a - b
   * 1、构建源id列表
   * 2、构建前端传入id列表
   * 3、求差集
   * @param dbVos
   * @param vos
   * @return
   */
  private List<String> difference(List<SaleGoalVo> dbVos, List<SaleGoalVo> vos) {
    Set<String> dbIds = dbVos.stream().collect(Collectors.groupingBy(SaleGoalVo::getId)).keySet();
    Set<String> ids = vos.stream().filter(vo -> com.biz.crm.util.StringUtils.isNotEmpty(vo.getId()))
            .collect(Collectors.groupingBy(SaleGoalVo::getId)).keySet();
    Set<String> differenceIds = Sets.difference(dbIds, ids);
    if(!CollectionUtil.collectionNotEmpty(differenceIds)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(differenceIds);
  }

  /**
   * 根据合同编码删除
   * 1、校验入参
   * 2、获取原始销售目标记录
   * 3、删除相关明细记录
   * 4、删除销售目标
   * @param contractCode
   */
  @Override
  @Transactional
  public void deleteByContractCode(String contractCode) {
    //1、
    if(StringUtils.isEmpty(contractCode)) {
      return;
    }
    //2、
    List<SaleGoalEntity> entities = this.saleGoalMapper.selectList(Wrappers.<SaleGoalEntity>query().eq("contract_code", contractCode));
    if(CollectionUtil.listEmpty(entities)) {
      return;
    }
    //3、
    ArrayList<String> goalIds = Lists.newArrayList(entities.stream().collect(Collectors.toMap(SaleGoalEntity::getId, a->a)).keySet());
    this.saleGoalItemService.deleteByGoalIds(goalIds);
    //4、
    this.delByIds(goalIds);
  }

  /**
   * 根据合同编码查询详情
   * @param contractCode
   * @param indexCode
   * @return
   */
  @Override
  public List<SaleGoalVo> findDetailsByContractCodeAndIndexCode(String contractCode, String indexCode) {
    if(StringUtils.isEmpty(contractCode)) {
      return Lists.newArrayList();
    }
    List<SaleGoalEntity> entities = this.saleGoalMapper.selectList(
            Wrappers.<SaleGoalEntity>query().eq("contract_code", contractCode)
                    .eq("index_code", indexCode)
    .orderByAsc("create_date").orderByAsc("create_date_second"));
    List<SaleGoalVo> saleGoalVos = CrmBeanUtil.copyList(entities, SaleGoalVo.class);
    if(CollectionUtil.listEmpty(saleGoalVos)) {
      return Lists.newArrayList();
    }
    saleGoalVos.forEach(vo -> {
      SaleGoalItemVo itemVo = this.saleGoalItemService.findBySaleGoalId(vo.getId());
      vo.setSaleGoalItemVo(itemVo);
    });
    return saleGoalVos;
  }

  private List<SaleGoalVo> findByIds(ArrayList<String> ids) {
    List<SaleGoalVo> vos = null;
    ValidateUtils.validate(ids,"请指定需要查询的销售目标！");
    List<SaleGoalEntity> entities = this.lambdaQuery()
            .in(SaleGoalEntity::getId, ids)
            .list();
    if(null == entities){
      return vos;
    }
    vos = CrmBeanUtil.copyList(entities, SaleGoalVo.class);
    Map<String, SaleGoalItemVo> detailMap = saleGoalItemService.findBySaleGoalIds(ids);
    for (SaleGoalVo vo : vos) {
      vo.setTargetNumStr(vo.getTargetNum() == null ? null : vo.getTargetNum().stripTrailingZeros().toPlainString());
      vo.setSaleGoalItemVo(detailMap.get(vo.getId()));
    }
    return vos;
  }
}
