package com.biz.crm.contract.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.contract.entity.ContractEntity;
import com.biz.crm.contract.mapper.ContractMapper;
import com.biz.crm.contract.service.ContractFileService;
import com.biz.crm.contract.service.ContractSalesAreaService;
import com.biz.crm.contract.service.ContractService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.nebular.dms.contract.ContractSalesAreaVo;
import com.biz.crm.nebular.dms.contract.ContractVo;
import com.biz.crm.nebular.dms.contract.UpdateEndTimeByIdsReq;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import io.jsonwebtoken.lang.Collections;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@ConditionalOnMissingBean(name = "contractServiceExpandImpl")
@Service(value = "contractService")
public class ContractServiceImpl<M extends BaseMapper<T>,T> implements ContractService {

    @Resource
    private ContractMapper contractMapper;
    @Resource
    private ContractFileService contractFileService;
    @Resource
    private ContractSalesAreaService contractSalesAreaService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 1、验证主表参数
     * 2、组装保存
     * 3、保存文件
     * 4、保存销售范围
     * @param contractVo
     */
    @Override
    @Transactional
    public void add(ContractVo contractVo) {
        //1
        ValidateUtils.validate(contractVo.getType(), "请指定合同类型");
//        ValidateUtils.validate(contractVo.getYear(), "请指定合同年度");
        ValidateUtils.validate(contractVo.getCode(), "请输入合同编号后再提交");
        ValidateUtils.validate(contractVo.getCusCode(), "请指定客户编号");
        ValidateUtils.validate(contractVo.getCusName(), "请指定客户名称");
        ValidateUtils.validate(contractVo.getStartTime(), "请指定有效时间段");
        ValidateUtils.validate(contractVo.getEndTime(), "请指定有效时间段");
        List<ContractEntity> entities = contractMapper.selectByMap(new HashMap<String, Object>(){
            {
                put("code", contractVo.getCode());
            }
        });
        if(!CollectionUtils.isEmpty(entities)){
            throw new BusinessException("您输入的合同编号" + contractVo.getCode() + "已存在");
        }

        //2
        ContractEntity contractEntity = new ContractEntity();
        BeanUtils.copyProperties(contractVo, contractEntity);
        contractMapper.insert(contractEntity);

        //3
        contractFileService.replace(contractEntity, contractVo.getFiles());

        //4
        contractSalesAreaService.replace(contractEntity, contractVo.getSalesAreas(),contractVo.getSalesAreasNoContain());
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), contractEntity.getId(),contractVo.getCusName(),contractVo);
    }

    /**
     * 1、验证主表参数
     * 2、组装保存
     * 3、保存文件
     * 4、保存销售范围
     * @param contractVo
     */
    @Override
    @Transactional
    public void edit(ContractVo contractVo) {

        //1
        ValidateUtils.validate(contractVo.getId(),"请指定要编辑的资料");
//        ValidateUtils.validate(contractVo.getType(),"请指定合同类型");
        ValidateUtils.validate(contractVo.getYear(),"请指定合同年度");
        ValidateUtils.validate(contractVo.getCode(),"请输入合同编号后再提交");
        ValidateUtils.validate(contractVo.getCusCode(),"请指定客户编号");
        ValidateUtils.validate(contractVo.getCusName(),"请指定客户名称");
        ValidateUtils.validate(contractVo.getStartTime(),"请指定有效时间段");
        ValidateUtils.validate(contractVo.getEndTime(),"请指定有效时间段");
        ContractEntity entity = contractMapper.selectById(contractVo.getId());
        ValidateUtils.validate(entity,"您要编辑的资料不存在或者已经被删除!");
        QueryWrapper wrapper = Wrappers.query();
        wrapper.eq("code", contractVo.getCode());
        ContractEntity contractEntity = this.contractMapper.selectOne(wrapper);
        ValidateUtils.isTrue(contractEntity == null || Objects.equals(contractEntity.getCode(), contractVo.getCode())
        && Objects.equals(contractEntity.getId(), contractVo.getId()), "该编码已存在，请重新输入");
        boolean codeChange = !contractVo.getCode().equals(entity.getCode());
        if(codeChange){
            List<ContractEntity> entities = contractMapper.selectByMap(new HashMap<String, Object>(){
                {
                    put("code", contractVo.getCode());
                }
            });
            if(!CollectionUtils.isEmpty(entities) && entities.size() > 1){
                throw new BusinessException("您输入的合同编号" + contractVo.getCode() + "已存在");
            }
        }

        //2
        ContractVo oldObject = this.findById(contractVo.getId());
        BeanUtils.copyProperties(contractVo,entity);
        contractMapper.updateById(entity);

        //3
        contractFileService.replace(entity, contractVo.getFiles());

        //4
        contractSalesAreaService.replace(entity, contractVo.getSalesAreas(),contractVo.getSalesAreasNoContain());

        ContractVo newObject = this.findById(contractVo.getId());
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                entity.getId(),entity.getCusName(),oldObject,newObject);
    }

    /**
     * 1、先查询主信息并且组装
     * 2、查询文件信息并且组装
     * 3、查询销售范围信息并且组装
     * @param id
     * @return
     */
    @CrmDictMethod
    @Override
    public ContractVo findById(String id) {
        ContractVo vo = null;
        //1
        ValidateUtils.validate(id, "请指定需要查询的资料！");
        ContractEntity entity = contractMapper.selectById(id);
        if(null == entity){
            return vo;
        }
        vo = new ContractVo();
        BeanUtils.copyProperties(entity,vo);
        this.buildCusChannelName(vo);

        //2
        vo.setFiles(contractFileService.findByContractId(id));

        //3
        Map<String,List<ContractSalesAreaVo>> map = contractSalesAreaService.findByContractId(id);
        vo.setSalesAreas(map.get("contain"));
        vo.setSalesAreasNoContain(map.get("noContain"));
        this.buildCusChannelName(vo);
        return vo;
    }

    /**
     * 设置客户渠道名称
     * @param vos
     * @return
     */
    private void buildCusChannelName(ContractVo... vos) {
        Map<String, String> channelDict = DictUtil.dictMap("channel");
        if(!CollectionUtil.mapNotEmpty(channelDict) || vos == null) {
            return;
        }
        Arrays.asList(vos).forEach(vo -> vo.setCusChannelName(this.buildCusChannelName(vo.getCusChannelCode(), channelDict)));
    }

    /**
     * 根据字典获取客户渠道名称
     * @param cusChannelCode
     * @param channelDict
     * @return
     */
    private String buildCusChannelName(String cusChannelCode, Map<String, String> channelDict) {
        if(StringUtils.isEmpty(cusChannelCode) || !CollectionUtil.mapNotEmpty(channelDict)) {
            return null;
        }
        String[] cusChannelCodes = cusChannelCode.split(",");
        StringBuilder builder = new StringBuilder();
        Arrays.asList(cusChannelCodes).forEach( code -> builder.append(channelDict.get(code)).append(","));
        return builder.substring(0, builder.length() > 0 ? builder.length()-1 : 0);
    }

    /**
     * @param contractVo
     * @return
     */
    @CrmDictMethod
    @Override
    public PageResult<ContractVo> list(ContractVo contractVo) {
        QueryWrapper<ContractVo> wrapper = Wrappers.<ContractVo>query()
                .eq(!StringUtils.isEmpty(contractVo.getType()),"type", contractVo.getType())
                .like(!StringUtils.isEmpty(contractVo.getCusChannelCode()),"cus_channel_code", contractVo.getCusChannelCode())
                .like(!StringUtils.isEmpty(contractVo.getCode()),"code", contractVo.getCode())
                .like(!StringUtils.isEmpty(contractVo.getCusCode()),"cus_code", contractVo.getCusCode())
                .like(!StringUtils.isEmpty(contractVo.getCusName()),"cus_name", contractVo.getCusName())
                .ge(!StringUtils.isEmpty(contractVo.getStartTime()),"start_time", contractVo.getStartTime())
                .le(!StringUtils.isEmpty(contractVo.getEndTime()),"end_time", contractVo.getEndTime())
                .like(!StringUtils.isEmpty(contractVo.getSaleCompanyCode()), "sale_company_code", contractVo.getSaleCompanyCode())
                .in(!CollectionUtil.listEmpty(contractVo.getOfficeCodes()), "ext33", contractVo.getOfficeCodes())
//                .like(!StringUtils.isEmpty(contractVo.getSaleCompanyName()), "sale_company_name", contractVo.getSaleCompanyName())
//                .like(!StringUtils.isEmpty(contractVo.getCusName()),"cus_name", contractVo.getCusName())
                .orderByDesc("create_date", "create_date_second");
        Page<ContractVo> page = PageUtil.buildPage(contractVo.getPageNum(),contractVo.getPageSize());
        List<ContractVo> contractVos = contractMapper.list(page, wrapper);
        if(!CollectionUtil.listEmpty(contractVos)) {
            this.buildCusChannelName(contractVos.toArray(new ContractVo[contractVos.size()]));
        }
        return PageResult.<ContractVo>builder()
                .data(contractVos)
                .count(page.getTotal())
                .build();
    }

    @Override
    @Transactional
    public void delByIds(ArrayList<String> ids) {
        //1
        if(Collections.isEmpty(ids)){
            return;
        }

        //2
        QueryWrapper<ContractVo> wrapper = Wrappers.<ContractVo>query()
                .in("id", ids);
        Page<ContractVo> page = PageUtil.buildPage(1,1000);
        List<ContractVo> contractVos = contractMapper.list(page, wrapper);
        contractMapper.deleteBatchIds(ids);

        //3
        contractSalesAreaService.delByContractIds(ids);


        if(CollectionUtils.isEmpty(contractVos)){
            return;
        }
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for(ContractVo vo : contractVos){
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),
                    vo.getId(),vo.getCusName(),vo);
        }
    }

    /**
     * 1、删除合同主干信息
     * 2、删除合同销售范围信息
     * @param contractVo
     */
    @Override
    @Transactional
    public void delByParam(ContractVo contractVo) {
        if(null == contractVo){
            throw new BusinessException("请指定查询条件!");
        }
        QueryWrapper<ContractEntity> wrapper = Wrappers.<ContractEntity>query()
                .eq(!StringUtils.isEmpty(contractVo.getType()),"type", contractVo.getType())
                .eq(!StringUtils.isEmpty(contractVo.getCusChannelCode()),"cus_channel_code", contractVo.getCusChannelCode())
                .like(!StringUtils.isEmpty(contractVo.getCode()),"code", contractVo.getCode())
                .like(!StringUtils.isEmpty(contractVo.getCusCode()),"cus_code", contractVo.getCusCode());
        List<ContractEntity> entities = contractMapper.selectList(wrapper);
        if(CollectionUtils.isEmpty(entities)){
            return ;
        }
        List<String> ids = entities.stream()
                .map(ContractEntity::getId)
                .collect(Collectors.toList());

        //1
        contractMapper.delete(wrapper);

        //2
        contractSalesAreaService.delByContractIds(ids);

        if(CollectionUtils.isEmpty(entities)){
            return;
        }
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for(ContractEntity entity : entities){
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),
                    entity.getId(),entity.getCusName(),entity);
        }
    }

    @CrmDictMethod
    @Override
    public List<ContractVo> findByConditions(ContractVo contractVo) {
        QueryWrapper<ContractVo> wrapper = Wrappers.<ContractVo>query()
                .eq(!StringUtils.isEmpty(contractVo.getType()),"type", contractVo.getType())
                .eq(!StringUtils.isEmpty(contractVo.getCusChannelCode()),"cus_channel_code", contractVo.getCusChannelCode())
                .like(!StringUtils.isEmpty(contractVo.getCode()),"code", contractVo.getCode())
                .like(!StringUtils.isEmpty(contractVo.getCusCode()),"cus_code", contractVo.getCusCode())
                .ge(!StringUtils.isEmpty(contractVo.getStartTime()),"start_time", contractVo.getStartTime())
                .le(!StringUtils.isEmpty(contractVo.getEndTime()),"end_time", contractVo.getEndTime())
                .like(!StringUtils.isEmpty(contractVo.getSaleCompanyCode()), "sale_company_code", contractVo.getSaleCompanyCode())
//                .like(!StringUtils.isEmpty(contractVo.getSaleCompanyName()), "sale_company_name", contractVo.getSaleCompanyName())
                .like(!StringUtils.isEmpty(contractVo.getExt9()), "ext9", contractVo.getExt9())
                .orderByDesc("create_date", "create_date_second");
        List<ContractVo> contractVos = contractMapper.findByConditions(wrapper);
        if(CollectionUtil.listEmpty(contractVos)) {
            return Lists.newArrayList();
        }
        if(!CollectionUtil.listEmpty(contractVos)) {
            this.buildCusChannelName(contractVos.toArray(new ContractVo[contractVos.size()]));
        }
        return contractVos;
    }

    /**
     * 1、验证
     * 2、根据条件更新
     * @param updateEndTimeByIdsReq
     */
    @Override
    public void updateEndTimeByIds(UpdateEndTimeByIdsReq updateEndTimeByIdsReq) {
        //1
        if(CollectionUtils.isEmpty(updateEndTimeByIdsReq.getIds())){
            return;
        }
        if(StringUtils.isEmpty(updateEndTimeByIdsReq.getEndTime())){
            throw new BusinessException("请指定要修改时间！");
        }

        //2
        contractMapper.updateEndTimeByIds(updateEndTimeByIdsReq);
    }
}
