package com.biz.crm.minbuynumofproduct.utils;

import com.biz.crm.base.BusinessException;
import com.biz.crm.minbuynumofproduct.entity.MinBuyNumOfProductEntity;
import com.biz.crm.nebular.dms.minbuynumofproduct.MinBuyNumOfProductVo;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.UUIDGenerator;
import com.biz.crm.util.ValidateUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 起订量工具类
 *
 * @Author: huojia
 * @DateTime: 2021/1/25 18:02
 **/
public class MinBuyNumOfProductUtil {

    /**
     * 验证参数
     *
     * @param minBuyNumOfProductVos
     * @author: huojia
     * @date: 2021/1/25 18:04
     * @return: void
     */
    public static void validParamBatch(ArrayList<MinBuyNumOfProductVo> minBuyNumOfProductVos) {
        minBuyNumOfProductVos.stream()
                .forEach(minBuyNumOfProductVo -> validParam(minBuyNumOfProductVo));
    }

    /**
     * 验证修改方法的参数
     *
     * @param minOrderVo
     * @author: huojia
     * @date: 2021/1/26 15:00
     * @return: void
     */
    public static void validParam(MinBuyNumOfProductVo minOrderVo) {
        ValidateUtils.validate(minOrderVo.getProductCode(), "产品编码不能为空！");
        ValidateUtils.validate(minOrderVo.getProductName(), "产品名称不能为空！");
        boolean flag = false;
        if (!StringUtils.isEmpty(minOrderVo.getOrgCode())
                && !StringUtils.isEmpty(minOrderVo.getOrgName())) {
            flag = true;
        }
        if (!StringUtils.isEmpty(minOrderVo.getCusCode())
                && !StringUtils.isEmpty(minOrderVo.getCusName())) {
            flag = true;
        }
        if (!StringUtils.isEmpty(minOrderVo.getTerminalCode())
                && !StringUtils.isEmpty(minOrderVo.getTerminalName())) {
            flag = true;
        }
        if (!flag) {
            throw new BusinessException("请选择组织、客户或终端！");
        }
        ValidateUtils.validate(minOrderVo.getVerificationMode(), "请指定验证方式！");
        ValidateUtils.validate(minOrderVo.getMinOrderQuantity(), "起订量不能为空！");
    }

    /**
     * 组装最小起订量的实体
     *
     * @param minOrderVos
     * @author: huojia
     * @date: 2021/1/26 9:41
     * @return: java.util.List<com.biz.crm.order.entity.MinOrderEntity>
     */
    public static List<MinBuyNumOfProductEntity> assembleEntities(ArrayList<MinBuyNumOfProductVo> minOrderVos) {
        List<MinBuyNumOfProductEntity> minOrderEntities = CrmBeanUtil.copyList(minOrderVos, MinBuyNumOfProductEntity.class);
        return minOrderEntities.stream()
                .map(minOrderEntity -> {
                    minOrderEntity.setId(UUIDGenerator.generate());
                    minOrderEntity.setOnlyKey(getOnlyKey(minOrderEntity));
                    return minOrderEntity;
                })
                .collect(Collectors.toList());
    }

    /**
     * 获取onlyKey，作为assembleEntities的子方法
     *
     * @param minOrderEntity
     * @author: huojia
     * @date: 2021/1/26 9:46
     * @return: java.lang.String
     */
    private static String getOnlyKey(MinBuyNumOfProductEntity minOrderEntity) {
        StringBuilder key = new StringBuilder();
        if (!StringUtils.isEmpty(minOrderEntity.getOrgCode())) {
            key.append(minOrderEntity.getOrgCode());
        } else if (!StringUtils.isEmpty(minOrderEntity.getCusCode())) {
            key.append(minOrderEntity.getCusCode());
        } else {
            key.append(minOrderEntity.getTerminalCode());
        }
        key.append(minOrderEntity.getProductCode());
        return key.toString();
    }

    /**
     * 构建分页查询条件
     *
     * @param minOrderVo
     * @author: huojia
     * @date: 2021/1/26 17:25
     * @return: org.elasticsearch.index.query.BoolQueryBuilder
     */
    public static BoolQueryBuilder buildQuery(MinBuyNumOfProductVo minOrderVo) {
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        if (!StringUtils.isEmpty(minOrderVo.getProductCode())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productCode", minOrderVo.getProductCode()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getProductName())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productName", minOrderVo.getProductName()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getOrgCode())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("orgCode", minOrderVo.getOrgCode()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getOrgName())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("orgName", minOrderVo.getOrgName()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getCusCode())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusCode", minOrderVo.getCusCode()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getCusName())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusName", minOrderVo.getCusName()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getTerminalCode())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalCode", minOrderVo.getTerminalCode()));
        }
        if (!StringUtils.isEmpty(minOrderVo.getTerminalName())) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalName", minOrderVo.getTerminalName()));
        }
        if (minOrderVo.getEffectiveFlag() != null) {
            boolQueryBuilder.must(QueryBuilders.termQuery("effectiveFlag", minOrderVo.getEffectiveFlag()));
        }
        return boolQueryBuilder;
    }
}