package com.biz.crm.promotion.service.component.validator.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.dms.PromotionPolicyEunm;
import com.biz.crm.nebular.dms.promotion.*;
import com.biz.crm.promotion.entity.PromotionPolicyEntity;
import com.biz.crm.promotion.entity.PromotionPolicyTemplateEntity;
import com.biz.crm.promotion.mapper.PromotionPolicyTemplateMapper;
import com.biz.crm.promotion.service.component.validator.PromotionPolicyEditValidator;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmDateUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;


/**
 * 政策编辑基础校验
 *  @author: luoqi
 *  @Date: 2021-1-6 16:47
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
@Component("promotionPolicyEditBaseValidator")
@Order(1)
public class PromotionPolicyEditBaseValidator implements PromotionPolicyEditValidator {
    @Resource
    private PromotionPolicyTemplateMapper promotionPolicyTemplateMapper;
    /**
     * 校验
     *
     * @param vo
     */
    @Override
    public void validate(PromotionPolicyEditReqVo vo) throws BusinessException{
        this.checkParams(vo);
    }

    /**
     * 校验参数
     *
     * @author: luoqi
     * @Date: 2020-10-15 17:42
     * @version: V1.0
     * @Description:
     */
    protected void checkParams(PromotionPolicyEditReqVo vo) {
        AssertUtils.isNotNull(vo, "无效参数，请重试");
        AssertUtils.isNotEmpty(vo.getFirstOrder(), "首单优惠不能为空!");
        AssertUtils.isNotEmpty(vo.getBeginTime(), "有效期开始日期不能为空!");
        AssertUtils.isNotEmpty(vo.getEndTime(), "有效期结束日期不能为空!");
        AssertUtils.isNotEmpty(vo.getMultipleable(), "是否校验不能为空");
        if (vo.getBeginTime().length() == 10) {
            vo.setBeginTime(vo.getBeginTime() + " " + CrmDateUtils.TIME_STR_00);
        }
        if (vo.getEndTime().length() == 10) {
            vo.setEndTime(vo.getEndTime() + " " + CrmDateUtils.TIME_STR_235959);
        }
        if (LocalDateTime.parse(vo.getBeginTime(), CrmDateUtils.yyyyMMddHHmmss)
                .compareTo(LocalDateTime.parse(vo.getEndTime(), CrmDateUtils.yyyyMMddHHmmss)) > 0) {
            throw new BusinessException("政策有效期开始日期不能大于结束日期");
        }
        AssertUtils.isNotEmpty(vo.getDescription(), "促销描述不能为空!");
        AssertUtils.isNotEmpty(vo.getPromotionPolicyCode(), "促销编码不能为空!");
        AssertUtils.isNotEmpty(vo.getPromotionPolicyName(), "促销名称不能为空!");
        AssertUtils.isNotEmpty(vo.getTemplateId(), "模板ID不能为空!");
        AssertUtils.isNotEmpty(vo.getTemplateCode(), "模板编码不能为空!");
        AssertUtils.isNotEmpty(vo.getTemplateName(), "模板名称不能为空!");
        if(StringUtils.isBlank(vo.getChanel())){
            vo.setChanel(PromotionPolicyEntity.ALL);
        }
        if(StringUtils.isBlank(vo.getSaleCompanyCode())){
            vo.setSaleCompanyCode(PromotionPolicyEntity.ALL);
            vo.setSaleCompanyName(PromotionPolicyEntity.ALL);
        }
        this.checkProductParams(vo);
        this.checkRuleParams(vo);
        this.checkScopeParams(vo);

    }

    /**
     * 规则检查
     * @param vo
     */
    protected void checkRuleParams(PromotionPolicyEditReqVo vo) {
        PromotionPolicyRule rule = vo.getPromotionPolicyRule();
        AssertUtils.isNotNull(rule, "促销规则不能为空!");
        AssertUtils.isNotEmpty(rule.getRulesAsCalculate(), "促销规则-计算规则不能为空!");
        AssertUtils.isNotEmpty(rule.getRulesAsCondition(), "促销规则-满足条件不能为空!");
        List<PromotionPolicyRuleEditVo> allRules = vo.takeAllRules();
        allRules.forEach(v -> {
//            AssertUtils.isNotEmpty(v.getPromotionPolicyTemplateCode(),"规则模板编码不能为空!");
//            AssertUtils.isNotEmpty(v.getPromotionPolicyTemplateId(),"规则模板ID不能为空!");
//            AssertUtils.isNotEmpty(v.getPromotionPolicyCode(),"政策编码不能为空!");
//            AssertUtils.isNotEmpty(v.getPromotionPolicyId(),"政策ID不能为空!");
            AssertUtils.isNotEmpty(v.getPromotionRuleId(), "规则ID不能为空!");
//            AssertUtils.isNotEmpty(v.getRuleCode(),"规则编码不能为空!");
//            AssertUtils.isNotEmpty(v.getRuleType(),"规则类型不能为空!");
//            AssertUtils.isNotEmpty(v.getPromotionType(),"规则促销类型不能为空!");
        });
    }

    /**
     * 促销商品检查
     * @param vo
     */
    protected void checkProductParams(PromotionPolicyEditReqVo vo) {
        PromotionPolicyProduct product = vo.getPromotionPolicyProduct();
        AssertUtils.isNotNull(product, "促销商品不能为空!");
        AssertUtils.isNotEmpty(product.getProductsCurrentY(), "促销商品本品不能为空!");
        PromotionPolicyTemplateEntity templateEntity = this.promotionPolicyTemplateMapper.selectById(vo.getTemplateId());
        AssertUtils.isNotNull(templateEntity, "不存在的政策模板[" + vo.getTemplateCode() + "]");
        //本品赠品一一对应
        boolean currentRelationalGift = PromotionPolicyEunm.PromotionProductTypeEunm.CURRENT_RELATIONAL_GIFT.getCode().equals(templateEntity.getPromotionProduct());
        if(currentRelationalGift){
            //本品赠品一一对应时，不需要传赠品列表，直接取本品上配置的赠品数据
            vo.getPromotionPolicyProduct().setProductsCurrentN(Lists.newArrayList());
        }
        List<PromotionPolicyProductEditVo> allProducts = vo.takeAllProducts();
        allProducts.forEach(v -> {
            AssertUtils.isNotEmpty(v.getProductCode(), "促销商品-商品编码不能为空!");
            AssertUtils.isNotEmpty(v.getProductName(), "促销商品-商品名称不能为空!");
            if(currentRelationalGift){
                AssertUtils.isNotEmpty(v.getProductCodeGift(), "促销商品-赠品编码不能为空!");
                AssertUtils.isNotEmpty(v.getProductNameGift(), "促销商品-赠品名称不能为空!");
            }
        });
    }

    /**
     * 促销范围检查
     * @param vo
     */
    protected void checkScopeParams(PromotionPolicyEditReqVo vo) {
        PromotionPolicyScope product = vo.getPromotionPolicyScope();
//        if(null == product){
//            //当用户没有指定范围时，可以生成默认范围对象
//            vo.buildDefToEmptyScope();
//        }
        List<PromotionPolicyScopeEditVo> allScopes = vo.takeAllScopes();
//        if(CollectionUtils.isEmpty(allScopes)){
////            //当用户没有指定范围时，可以生成默认范围对象
////            vo.buildDefToEmptyScope();
////        }
        allScopes.forEach(v -> {
            AssertUtils.isNotEmpty(v.getScopeCode(), "促销范围-编码不能为空!");
            AssertUtils.isNotEmpty(v.getScopeName(), "促销范围-名称不能为空!");
            AssertUtils.isNotEmpty(v.getScopeType(), "促销范围-范围类型不能为空!");
        });
    }
}
