package com.biz.crm.promotion.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.dms.PromotionEnum;
import com.biz.crm.nebular.dms.promotion.policy.req.DmsPromotionPolicyOrderDetailReqVo;
import com.biz.crm.promotion.entity.DmsPromotionPolicyOrderDetailEntity;
import com.biz.crm.promotion.entity.PromotionPolicyEntity;
import com.biz.crm.promotion.mapper.DmsPromotionPolicyOrderDetailMapper;
import com.biz.crm.promotion.mapper.PromotionPolicyMapper;
import com.biz.crm.promotion.service.IDmsPromotionPolicyOrderDetailService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import org.springframework.transaction.annotation.Transactional;

/**
 * 促销政策订单明细接口实现
 *
 * @author lf
 * @date 2020-12-26 13:34:30
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="DmsPromotionPolicyOrderDetailServiceExpandImpl")
public class DmsPromotionPolicyOrderDetailServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<DmsPromotionPolicyOrderDetailMapper, DmsPromotionPolicyOrderDetailEntity> implements IDmsPromotionPolicyOrderDetailService {

    @Resource
    private DmsPromotionPolicyOrderDetailMapper dmsPromotionPolicyOrderDetailMapper;
    @Resource
    private PromotionPolicyMapper promotionPolicyMapper;
    @Autowired
    private RedissonUtil redissonUtil;

//    @Override
//    public Boolean lockPromotionPolicyOrderDetailList(List<DmsPromotionPolicyOrderDetailReqVo> reqVoList) {
//        Set<String> promotionPolicyIdList = reqVoList.stream().map(DmsPromotionPolicyOrderDetailReqVo::getPromotionPolicyId).collect(Collectors.toSet());
//        if (promotionPolicyIdList.size()>0){
//            throw new BusinessException("存在多个不同的促销政策id");
//        }
//        reqVoList.forEach(data->{
//            this.checkParam(data);
//        });
//        //促销政策锁
//        String promotionPolicyIdLock = reqVoList.get(0).getPromotionPolicyId();
//        Boolean flag = false;
//        Boolean lockFlag = false;
//        //获取锁
//        RLock rLock = redissonUtil.createLock(promotionPolicyIdLock);
//        try {
//            if (rLock.tryLock()){
//                lockFlag = true;
//                reqVoList.forEach(reqVo->{
//                    //业务代码实现
//                    BigDecimal surplusUsedQty = promotionPolicyMapper.selectPromotionPolicySurplusUsedQty(reqVo.getPromotionPolicyId());
//                    if (reqVo.getUsedQty().compareTo(surplusUsedQty)==1){
//                        throw new BusinessException("促销政策使用量大于剩余使用量");
//                    }
//                    DmsPromotionPolicyOrderDetailEntity entity = CrmBeanUtil.copy(reqVo,DmsPromotionPolicyOrderDetailEntity.class);
//                    entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
//                    entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
//                    this.save(entity);
//                });
//                flag = true;
//            }else {
//                throw new BusinessException("此促销政策正在使用,请稍等....");
//            }
//        }finally {
//            if (lockFlag){
//                redissonUtil.unLock(rLock);
//            }
//        }
//        return flag;
//    }

    /**
     * 锁定促销政策使用明细
     * @param reqVo
     * @return
     */
    @Override
    @Transactional
    public Boolean lockPromotionPolicyOrderDetail(DmsPromotionPolicyOrderDetailReqVo reqVo) {
        reqVo.setPolicyOrderDetailStatus(PromotionEnum.POLICY_ORDER_DETAIL.locked.getVal());
        this.checkParam(reqVo);
        //促销政策锁
        String promotionPolicyIdLock = reqVo.getPromotionPolicyId();
        Boolean flag = false;
        Boolean lockFlag = false;
        //获取锁
        RLock rLock = redissonUtil.createLock(promotionPolicyIdLock);
        try {
            if (rLock.tryLock()){
                lockFlag = true;
                //业务代码实现
                LambdaQueryWrapper<PromotionPolicyEntity> lambdaQuery = new QueryWrapper<PromotionPolicyEntity>().lambda().eq(PromotionPolicyEntity::getId,reqVo.getPromotionPolicyId());
                PromotionPolicyEntity promotionPolicyEntity = promotionPolicyMapper.selectOne(lambdaQuery);
                if (promotionPolicyEntity == null ){
                    throw new BusinessException("未查询到促销政策信息");
                }
                //判断使用量存在
                if (null != promotionPolicyEntity.getUsedQtyUpper()){
                    BigDecimal usedQty = dmsPromotionPolicyOrderDetailMapper.selectPromotionPolicyUsedQty(reqVo.getPromotionPolicyId());
                    BigDecimal surplusUsedQty = promotionPolicyEntity.getUsedQtyUpper().subtract(usedQty);
                    if (reqVo.getUsedQty().compareTo(surplusUsedQty)==1){
                        throw new BusinessException("促销政策使用量大于剩余使用量");
                    }
                }
                DmsPromotionPolicyOrderDetailEntity entity = CrmBeanUtil.copy(reqVo,DmsPromotionPolicyOrderDetailEntity.class);
                entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
                entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                entity.setHitDetail(JSONObject.toJSONString(reqVo.getHitDetail()));
                this.save(entity);
                flag = true;
            }else {
                throw new BusinessException("此促销政策正在使用,请稍等....");
            }
        }finally {
            if (lockFlag){
                redissonUtil.unLock(rLock);
            }
        }
        return flag;
    }

    /**
     * 批量锁定促销明细使用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional
    public Boolean lockPromotionPolicyOrderDetailBatch(List<DmsPromotionPolicyOrderDetailReqVo> reqVo) {
        if(CollectionUtil.listEmpty(reqVo)) {
            return false;
        }
        for (DmsPromotionPolicyOrderDetailReqVo vo : reqVo) {
            ValidateUtils.isTrue(this.lockPromotionPolicyOrderDetail(vo), "锁定失败订单号【%s】政策编码【%s】",
                    vo.getOrderNo(), vo.getPromotionCode());
        }
        return true;
    }

    /**
     * 提交促销政策使用明细
     * @param orderNo
     * @return
     */
    @Override
    public Boolean commitPromotionPolicyOrderDetail(String orderNo) {
        if (!StringUtils.isNotEmpty(orderNo)){
            throw new BusinessException("订单编码为空");
        }
        Boolean flag = false;
        Boolean lock = false;
        //查询已锁定的数据
        List<DmsPromotionPolicyOrderDetailEntity> entityList = this.lambdaQuery().
                eq(DmsPromotionPolicyOrderDetailEntity::getOrderNo,orderNo)
                .eq(DmsPromotionPolicyOrderDetailEntity::getPolicyOrderDetailStatus,PromotionEnum.POLICY_ORDER_DETAIL.locked).list();
        //查询是否为空
        if (!CollectionUtil.listNotEmptyNotSizeZero(entityList)){
            flag = true;
            return flag;
        }
        //获取锁
        RLock rLock = redissonUtil.createLock(orderNo);
        try {
            if (rLock.tryLock()){
                lock = true;
                entityList.forEach(data->{
                    data.setPolicyOrderDetailStatus(PromotionEnum.POLICY_ORDER_DETAIL.commit.getVal());
                });
                this.updateBatchById(entityList);
                flag = true;
            }else {
                throw new BusinessException("正在进行提交促销政策使用明细,请稍等");
            }
        }finally {
            if (lock){
                redissonUtil.unLock(rLock);
            }
        }
        return flag;
    }

    @Override
    public Boolean lockAndSubmitPromotionPolicyOrderDetail(DmsPromotionPolicyOrderDetailReqVo reqVo) {
        if(this.lockPromotionPolicyOrderDetail(reqVo)){
            return this.commitPromotionPolicyOrderDetail(reqVo.getOrderNo());
        }
        return false;
    }

    /**
     * 解除锁定促销政策使用明细
     * @param orderNo
     * @return
     */
    @Override
    public Boolean unlockPromotionPolicyOrderDetail(String orderNo) {
        if (!StringUtils.isNotEmpty(orderNo)){
            throw new BusinessException("订单编码为空");
        }
        Boolean flag = false;
        Boolean lock = false;
        //查询已锁定的数据
        List<DmsPromotionPolicyOrderDetailEntity> entityList = this.lambdaQuery().
                eq(DmsPromotionPolicyOrderDetailEntity::getOrderNo,orderNo)
                .eq(DmsPromotionPolicyOrderDetailEntity::getPolicyOrderDetailStatus,PromotionEnum.POLICY_ORDER_DETAIL.locked).list();
        //查询是否为空
        if (!CollectionUtil.listNotEmptyNotSizeZero(entityList)){
            flag = true;
            return flag;
        }
        //获取锁
        RLock rLock = redissonUtil.createLock(orderNo);
        try {
            if (rLock.tryLock()){
                lock = true;
                entityList.forEach(data->{
                    data.setPolicyOrderDetailStatus(PromotionEnum.POLICY_ORDER_DETAIL.unlock.getVal());
                });
                this.updateBatchById(entityList);
                flag = true;
            }else {
                throw new BusinessException("正在进行提交促销政策使用明细,请稍等");
            }
        }finally {
            if (lock){
                redissonUtil.unLock(rLock);
            }
        }
        return flag;
    }

    /**
     * 释放促销政策使用明细
     * @param orderNo
     * @return
     */
    @Override
    public Boolean rollbackPromotionPolicyOrderDetail(String orderNo) {
        if (!StringUtils.isNotEmpty(orderNo)){
            throw new BusinessException("订单编码为空");
        }
        Boolean flag = false;
        Boolean lock = false;
        //查询已锁定的数据
        List<DmsPromotionPolicyOrderDetailEntity> entityList = this.lambdaQuery().
                eq(DmsPromotionPolicyOrderDetailEntity::getOrderNo,orderNo)
                .eq(DmsPromotionPolicyOrderDetailEntity::getPolicyOrderDetailStatus,PromotionEnum.POLICY_ORDER_DETAIL.locked).list();
        //查询是否为空
        if (!CollectionUtil.listNotEmptyNotSizeZero(entityList)){
            flag = true;
            return flag;
        }
        //获取锁
        RLock rLock = redissonUtil.createLock(orderNo);
        try {
            if (rLock.tryLock()){
                lock = true;
                entityList.forEach(data->{
                    data.setPolicyOrderDetailStatus(PromotionEnum.POLICY_ORDER_DETAIL.rollback.getVal());
                });
                this.updateBatchById(entityList);
                flag = true;
            }else {
                throw new BusinessException("正在进行提交促销政策使用明细,请稍等");
            }
        }finally {
            if (lock){
                redissonUtil.unLock(rLock);
            }
        }
        return flag;
    }

    /**
     * 校验参数信息
     */
    private void checkParam(DmsPromotionPolicyOrderDetailReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getPolicyOrderDetailStatus(),"促销政策使用明细状态为空");
        AssertUtils.isNotEmpty(reqVo.getPromotionPolicyId(),"促销政策id为空");
        AssertUtils.isNotEmpty(reqVo.getOrderId(),"订单id为空");
        AssertUtils.isNotEmpty(reqVo.getOrderNo(),"订单号为空");
        AssertUtils.isNotEmpty(reqVo.getCustomerCode(),"客户编码为空");
        AssertUtils.isNotEmpty(reqVo.getCustomerName(),"客户名称为空");
        if (reqVo.getUsedQty()==null){
            throw new BusinessException("促销政策id为空");
        }
        if (!CollectionUtil.listNotEmptyNotSizeZero(reqVo.getHitDetail())){
            throw new BusinessException("政策明细数据为空");
        }
        AssertUtils.isNotEmpty(reqVo.getUsedQtyType(),"政策使用量类型为空");
        AssertUtils.isNotEmpty(reqVo.getUsedQtyUnit(),"政策使用量单位为空");
    }
}
