package com.biz.crm.promotion.service.npromotion.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.promotion.entity.PromotionPolicyProductEntity;
import com.biz.crm.promotion.mapper.PromotionPolicyProductMapper;
import com.biz.crm.promotion.service.npromotion.PromotionProductService;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionProductVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CommonConstant;
import com.biz.crm.util.CommonFieldUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 促销商品逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/4/13 18:37
 */
@Service
@ConditionalOnMissingBean(name = "promotionProductServiceExtend")
public class PromotionProductServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<PromotionPolicyProductMapper, PromotionPolicyProductEntity> implements PromotionProductService {

  @Resource
  private PromotionPolicyProductMapper promotionPolicyProductMapper;

  /**
   * 批量保存促销商品
   * 包含新增、修改、删除操作
   * @param productVos
   * @param promotionPolicyCode
   * @return
   */
  @Override
  @Transactional
  public List<PromotionProductVo> saveBatch(List<PromotionProductVo> productVos, String promotionPolicyCode) {
    //1、校验参数非空
    if(CollectionUtil.listEmpty(productVos)) {
      return Lists.newArrayList();
    }
    List<PromotionProductVo> oldProducts = this.findByPromotionCode(promotionPolicyCode);
    //2、需要新增的记录
    List<PromotionProductVo> createProducts = productVos.stream().filter(r -> StringUtils.isEmpty(r.getId())).collect(Collectors.toList());
    List<PromotionPolicyProductEntity> createEntities = CrmBeanUtil.copyList(createProducts, PromotionPolicyProductEntity.class);
    this.saveBatch(createEntities);
    //替换保存完成后的记录，用于返回结果
    productVos.removeAll(createProducts);
    productVos.addAll(CrmBeanUtil.copyList(createEntities, PromotionProductVo.class));
    //3、需要修改的记录
    List<PromotionProductVo> updateProducts = productVos.stream().filter(r -> StringUtils.isNotEmpty(r.getId())).collect(Collectors.toList());
    List<PromotionPolicyProductEntity> updateEntities = CrmBeanUtil.copyList(updateProducts, PromotionPolicyProductEntity.class);
    this.updateBatchById(updateEntities);
    //4、需要删除的id列表
    List<String> deleteIds = CommonFieldUtil.difference(oldProducts, productVos, "id");
    this.removeByIds(deleteIds);
    return productVos;
  }

  /**
   * 根据促销政策编码查询促销商品列表
   * @param promotionPolicyCode
   * @return
   */
  @Override
  public List<PromotionProductVo> findByPromotionCode(String promotionPolicyCode) {
    if(StringUtils.isEmpty(promotionPolicyCode)) {
      return Lists.newArrayList();
    }
    List<PromotionPolicyProductEntity> entities = this.promotionPolicyProductMapper.selectList(Wrappers.<PromotionPolicyProductEntity>lambdaQuery()
    .eq(PromotionPolicyProductEntity::getPromotionPolicyCode, promotionPolicyCode));
    return CrmBeanUtil.copyList(entities, PromotionProductVo.class);
  }

  /**
   * 根据商品（本品）编码查询促销政策编码列表
   * @param productCode
   * @return
   */
  @Override
  public List<String> findPromotionCodesByProductCode(String productCode) {
    if(StringUtils.isEmpty(productCode)) {
      return Lists.newArrayList();
    }
    LambdaQueryWrapper wrapper = Wrappers.<PromotionPolicyProductEntity>lambdaQuery()
            .eq(PromotionPolicyProductEntity::getCurrentProduct, CommonConstant.GLOBAL.YesOrNo.Y.getItemCode())
            .eq(PromotionPolicyProductEntity::getProductCode, productCode);
    List<PromotionPolicyProductEntity> entities = this.promotionPolicyProductMapper.selectList(wrapper);
    if(CollectionUtil.listEmpty(entities)) {
      return Lists.newArrayList();
    }
    Set<String> promotionCodes = entities.stream().map(PromotionPolicyProductEntity::getPromotionPolicyCode).collect(Collectors.toSet());
    if(!CollectionUtil.collectionNotEmpty(promotionCodes)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(promotionCodes);
  }
}
