package com.biz.crm.salecontract.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.dms.salecontract.ContractTemplateTupleMappingVo;
import com.biz.crm.nebular.dms.salecontract.ContractTupleVo;
import com.biz.crm.salecontract.entity.ContractTupleEntity;
import com.biz.crm.salecontract.mapper.ContractTupleMapper;
import com.biz.crm.salecontract.service.ContractTemplateTupleMappingService;
import com.biz.crm.salecontract.service.ContractTupleService;
import com.biz.crm.salecontract.service.listener.AbstractContractModelListener;
import com.biz.crm.salecontract.util.ContractTupleUtil;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 合同元组逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/2/5 9:50
 */
@ConditionalOnMissingBean(name = "contractTupleServiceImpl")
@Service(value = "contractTupleService")
public class ContractTupleServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<ContractTupleMapper, ContractTupleEntity> implements ContractTupleService {

  @Resource
  private ContractTupleMapper contractTupleMapper;
  @Resource
  private ApplicationContext applicationContext;
  @Autowired
  private CrmLogSendUtil crmLogSendUtil;
  @Resource
  private ContractTemplateTupleMappingService mappingService;
  /**
   * 新增一条合同元组
   * 1、校验入参
   * 2、对象名称验重
   * 3、保存元组
   * 4、写日志
   * @param contractTupleVo
   * @return
   */
  @Override
  @Transactional
  public ContractTupleVo add(ContractTupleVo contractTupleVo) {
    //1、
    ContractTupleUtil.validateAdd(contractTupleVo);
    ValidateUtils.validate(this.getTupleBean(contractTupleVo.getBeanName()), "没有获取到名称为%s的bean对象", contractTupleVo.getBeanName());
    //2、
    List<ContractTupleEntity> entities = this.contractTupleMapper.selectList(Wrappers.<ContractTupleEntity>query().eq("json_name", contractTupleVo.getJsonName()));
    ValidateUtils.isTrue(CollectionUtil.listEmpty(entities), "模块字段名%s已存在，请重新输入", contractTupleVo.getJsonName());
    //3、
    contractTupleVo.setCode(CodeUtil.generateCode());
    ContractTupleEntity entity = CrmBeanUtil.copy(contractTupleVo, ContractTupleEntity.class);
    this.contractTupleMapper.insert(entity);
    contractTupleVo.setId(entity.getId());
    //4、
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    crmLogSendUtil.sendForAdd(menuCodeObj.toString(),contractTupleVo.getId(),contractTupleVo.getCode(),
            this.findByCode(contractTupleVo.getCode()));
    return contractTupleVo;
  }

  /**
   * 编辑一条合同元组
   * 1、校验入参
   * 2、修改元组记录
   * 3、写日志
   * @param contractTupleVo
   * @return
   */
  @Override
  @Transactional
  public ContractTupleVo edit(ContractTupleVo contractTupleVo) {
    //1、
    ContractTupleUtil.validateEdit(contractTupleVo);
    ValidateUtils.validate(this.getTupleBean(contractTupleVo.getBeanName()), "没有获取到名称为%s的bean对象", contractTupleVo.getBeanName());
    ContractTupleVo dbVo = this.findByCode(contractTupleVo.getCode());
    ValidateUtils.validate(dbVo, "没有获取到原始记录，不能编辑");
    //2、
    ContractTupleEntity entity = CrmBeanUtil.copy(contractTupleVo, ContractTupleEntity.class);
    this.contractTupleMapper.updateById(entity);
    //3、
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    ContractTupleVo newObject = this.findByCode(contractTupleVo.getCode());
    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), contractTupleVo.getId(),contractTupleVo.getCode(),dbVo,newObject);
    return contractTupleVo;
  }

  /**
   * 根据编码查询
   * @param code
   * @return
   */
  @Override
  public ContractTupleVo findByCode(String code) {
    if(StringUtils.isEmpty(code)) {
      return null;
    }
    ContractTupleEntity entity = this.contractTupleMapper.selectOne(Wrappers.<ContractTupleEntity>query().eq("code", code));
    return CrmBeanUtil.copy(entity, ContractTupleVo.class);
  }

  /**
   * 分页条件查询合同元组
   * @param contractTupleVo
   * @return
   */
  @Override
  public PageResult<ContractTupleVo> findPageByConditions(ContractTupleVo contractTupleVo) {
    Page<ContractTupleVo> page = new Page(contractTupleVo.getPageNum(), contractTupleVo.getPageSize());
    QueryWrapper wrapper = Wrappers.query()
            .like(StringUtils.isNotEmpty(contractTupleVo.getCode()), "code", contractTupleVo.getCode())
            .like(StringUtils.isNotEmpty(contractTupleVo.getName()), "name", contractTupleVo.getName())
            .like(StringUtils.isNotEmpty(contractTupleVo.getRelativePath()), "relative_path", contractTupleVo.getRelativePath())
            .like(StringUtils.isNotEmpty(contractTupleVo.getRouterPath()), "router_path", contractTupleVo.getRouterPath())
            .eq(StringUtils.isNotEmpty(contractTupleVo.getEnableStatus()), "enable_status", contractTupleVo.getEnableStatus());
    List<ContractTupleVo> pageList = this.contractTupleMapper.findPageByConditions(page, wrapper);
    return PageResult.<ContractTupleVo>builder().data(pageList).count(page.getTotal()).build();
  }

  /**
   * 获取元组对应bean对象
   * @param beanName
   * @return
   */
  @Override
  public AbstractContractModelListener getTupleBean(String beanName) {
    ValidateUtils.validate(beanName, "bean名称不能为空");
    AbstractContractModelListener contractListenerService;
    try {
      contractListenerService = this.applicationContext.getBean(beanName, AbstractContractModelListener.class);
    } catch (Exception e) {
      throw new BusinessException("没有获取到名称为".concat(beanName).concat("的bean对象"), e);
    }
    return contractListenerService;
  }

  /**
   * 批量启用
   * @param ids
   */
  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    ValidateUtils.notEmpty(ids, "请至少选择一条数据");
    UpdateWrapper wrapper = Wrappers.<ContractTupleEntity>update()
            .set("enable_status", CrmEnableStatusEnum.ENABLE.getCode())
            .in("id", ids);
    this.update(wrapper);
  }

  /**
   * 批量禁用
   * @param ids
   */
  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    ValidateUtils.notEmpty(ids, "请至少选择一条数据");
    UpdateWrapper wrapper = Wrappers.<ContractTupleEntity>update()
            .set("enable_status", CrmEnableStatusEnum.DISABLE.getCode())
            .in("id", ids);
    this.update(wrapper);
  }

  /**
   * 批量删除
   * @param ids
   */
  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    ValidateUtils.notEmpty(ids, "请至少选择一条数据");
    ids.forEach(id -> this.delete(id));
  }

  /**
   * 删除
   * 1、校验入参
   * 2、校验元组是否被模板使用过
   * 3、删除记录
   * @param id
   */
  @Override
  @Transactional
  public void delete(String id) {
    //1、
    ValidateUtils.validate(id, "id不能为空");
    ContractTupleEntity entity = this.contractTupleMapper.selectById(id);
    ValidateUtils.validate("没有获取到id为【%s】的原始元组记录", id);
    //2、
    List<ContractTemplateTupleMappingVo> mappingVos = mappingService.findByTupleCode(entity.getCode());
    Set<String> templateCodeSet = mappingVos.stream().collect(Collectors.toMap(ContractTemplateTupleMappingVo::getTemplateCode, a -> a, (a, b) -> b)).keySet();
    String templateCodeStr = String.join(",", templateCodeSet);
    ValidateUtils.isTrue(CollectionUtil.listEmpty(mappingVos), "元组【%s】已被模板【%s】使用过，不能删除", entity.getName(), templateCodeStr);
    //3、
    this.contractTupleMapper.deleteById(id);
  }
}
