package com.biz.crm.salecontract.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.contractupdatingfiles.service.ContractUpdatingFilesService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.dms.contractupdatingfiles.ContractUpdatingFilesVo;
import com.biz.crm.nebular.dms.salecontract.ContractTemplateVo;
import com.biz.crm.nebular.dms.salecontract.ContractTupleVo;
import com.biz.crm.nebular.dms.salecontract.SaleContractVo;
import com.biz.crm.salecontract.entity.SaleContractEntity;
import com.biz.crm.salecontract.mapper.SaleContractMapper;
import com.biz.crm.salecontract.service.ContractTemplateService;
import com.biz.crm.salecontract.service.ContractTupleService;
import com.biz.crm.salecontract.service.SaleContractService;
import com.biz.crm.salecontract.service.extral.SaleContractValidator;
import com.biz.crm.salecontract.service.listener.AbstractContractModelListener;
import com.biz.crm.salecontract.util.SaleContractUtil;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import java.util.List;
import javax.annotation.Resource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 合同逻辑处理实现类
 * @Author: chenrong
 * @Date: 2021/2/17 14:26
 */
@Service("saleContractService")
@ConditionalOnMissingBean(name = "saleContractServiceImpl")
public class SaleContractServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SaleContractMapper, SaleContractEntity> implements SaleContractService {

  @Resource
  private SaleContractMapper saleContractMapper;
  @Resource
  private ContractTemplateService contractTemplateService;
  @Resource
  private ContractTupleService contractTupleService;
  @Resource
  private ContractUpdatingFilesService contractUpdatingFilesService;
  @Autowired
  private CrmLogSendUtil crmLogSendUtil;
  @Autowired(required = false)
  private List<SaleContractValidator> saleContractValidators;

  /**
   * 新增合同
   * 1、校验入参
   * 2、设置编码
   * 3、获取并校验模板包含的元组信息
   * 4、根据元组信息触发相关数据保存接口
   * 5、保存合同
   * 6、更新文件
   * @param saleContractVo
   * @return
   */
  @Override
  @Transactional
  public SaleContractVo add(SaleContractVo saleContractVo) {
    //1、
    if(CollectionUtil.listNotEmpty(saleContractValidators)) {
      saleContractValidators.forEach(saleContractValidator -> saleContractValidator.validateAdd(saleContractVo));
    }
    SaleContractUtil.validateAdd(saleContractVo);
    //2、
    saleContractVo.setCode(CodeUtil.generateCode());
    //3、
    ContractTemplateVo contractTemplateVo = this.contractTemplateService.findDetailsByCode(saleContractVo.getTemplateCode());
    ValidateUtils.validate(contractTemplateVo, "没有获取到模板");
    ValidateUtils.notEmpty(contractTemplateVo.getTuples(), "没有获取到模板元组列表");
    //5、
    SaleContractEntity entity = CrmBeanUtil.copy(saleContractVo, SaleContractEntity.class);
    boolean addResult = this.save(entity);
    ValidateUtils.isTrue(addResult, "新增合同失败");
    saleContractVo.setId(entity.getId());
    List<ContractTupleVo> tupleVos = contractTemplateVo.getTuples();
    JSONObject data = saleContractVo.getData();
    //4、
    tupleVos.forEach(tuple -> {
      AbstractContractModelListener contractListenerService = this.contractTupleService.getTupleBean(tuple.getBeanName());
      if (contractListenerService != null && data.getJSONArray(tuple.getJsonName()) != null && data.getJSONArray(tuple.getJsonName()).size() > 0) {
        JSONArray jsonArray = data.getJSONArray(tuple.getJsonName());
        ValidateUtils.validate(CollectionUtil.listEmpty(jsonArray) || jsonArray.size() == 1, "模块%s只能传入一条记录", tuple.getName());
        JSONArray result = contractListenerService.add(jsonArray, saleContractVo.getCode(), tuple.getJsonName(), saleContractVo, tuple.getIndexCode());
        data.put(tuple.getJsonName(), result);
      }
    });
    //6、
    if(CollectionUtil.listNotEmpty(saleContractVo.getFilesVo())) {
      this.contractUpdatingFilesService.replace(saleContractVo.getFilesVo(), saleContractVo.getCode());
    }
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    SaleContractVo logVo = this.findDetailByCode(entity.getCode());
    crmLogSendUtil.sendForAdd(menuCodeObj.toString(), logVo.getId(),logVo.getCode(),logVo);
    return saleContractVo;
  }

  /**
   * 编辑合同
   * 1、校验入参
   * 2、获取并校验模板包含的元组信息
   * 3、根据元组信息触发各模块数据更新接口
   * 4、更新合同
   * 5、更新文件
   * @param saleContractVo
   * @return
   */
  @Override
  @Transactional
  public SaleContractVo edit(SaleContractVo saleContractVo) {
    //1、
    if(CollectionUtil.listNotEmpty(saleContractValidators)) {
      saleContractValidators.forEach(saleContractValidator -> saleContractValidator.validateEdit(saleContractVo));
    }
    SaleContractUtil.validateEdit(saleContractVo);
    //2、
    ContractTemplateVo contractTemplateVo = this.contractTemplateService.findDetailsByCode(saleContractVo.getTemplateCode());
    ValidateUtils.validate(contractTemplateVo, "没有获取到模板");
    ValidateUtils.notEmpty(contractTemplateVo.getTuples(), "没有获取到模板元组列表");
    List<ContractTupleVo> tupleVos = contractTemplateVo.getTuples();
    JSONObject data = saleContractVo.getData();
    SaleContractVo oldObject = this.findDetailByCode(saleContractVo.getCode());
    //3、
    tupleVos.forEach(tuple -> {
      AbstractContractModelListener contractListenerService = this.contractTupleService.getTupleBean(tuple.getBeanName());
      if (contractListenerService != null && data.getJSONArray(tuple.getJsonName()) != null && data.getJSONArray(tuple.getJsonName()).size() > 0) {
        JSONArray jsonArray = data.getJSONArray(tuple.getJsonName());
        ValidateUtils.validate(CollectionUtil.listEmpty(jsonArray) || jsonArray.size() == 1, "模块%s只能传入一条记录", tuple.getName());
        JSONArray result = contractListenerService.edit(jsonArray, saleContractVo.getCode(), tuple.getJsonName(), saleContractVo, tuple.getIndexCode());
        data.put(tuple.getJsonName(), result);
      }
    });
    //4、
    boolean editResult = this.updateById(CrmBeanUtil.copy(saleContractVo, SaleContractEntity.class));
    ValidateUtils.validate(editResult, "更新合同失败");
    //5、
    if(CollectionUtil.listNotEmpty(saleContractVo.getFilesVo())) {
      this.contractUpdatingFilesService.replace(saleContractVo.getFilesVo(), saleContractVo.getCode());
    }
    SaleContractVo newObject = saleContractVo;
    Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
            newObject.getId(),newObject.getCode(),oldObject,newObject);
    return saleContractVo;
  }

  /**
   * 根据合同编码查询详情
   * 1、获取合同主数据
   * 2、获取模板数据
   * 3、根据元组信息获取各模块数据
   * 4、获取附件
   * @param code 合同编码
   * @return
   */
  @Override
  public SaleContractVo findDetailByCode(String code) {
    if (StringUtils.isEmpty(code)) {
      return null;
    }
    //1、
    QueryWrapper wrapper = Wrappers.query().eq("code", code);
    SaleContractEntity entity = this.getOne(wrapper);
    SaleContractVo vo = CrmBeanUtil.copy(entity, SaleContractVo.class);
    if (vo == null) {
      return null;
    }
    //2、
    ContractTemplateVo templateVo = this.contractTemplateService.findDetailsByCode(vo.getTemplateCode());
    ValidateUtils.validate(templateVo, "没有获取到原始模板");
    vo.setContractTemplateVo(templateVo);
    List<ContractTupleVo> tupleVos = templateVo.getTuples();
    //3、
    if (CollectionUtil.listNotEmpty(tupleVos)) {
      JSONObject data = new JSONObject();
      tupleVos.forEach(tuple -> {
        AbstractContractModelListener listenerService = this.contractTupleService.getTupleBean(tuple.getBeanName());
        JSONArray result = listenerService.findByContractCode(code, tuple.getJsonName(), tuple.getIndexCode());
        data.put(tuple.getJsonName(), result);
      });
      vo.setData(data);
    }
    //4、
    List<ContractUpdatingFilesVo> filesVos = this.contractUpdatingFilesService.findByContractCode(code);
    vo.setFilesVo(filesVos);
    return vo;
  }

  /**
   * 条件分页查询合同
   * @param saleContractVo
   * @return
   */
  @Override
  public PageResult<SaleContractVo> findContractPageByConditions(SaleContractVo saleContractVo) {
    Page<SaleContractVo> page = new Page<>(saleContractVo.getPageNum(), saleContractVo.getPageSize());
    QueryWrapper<SaleContractVo> wrapper = Wrappers.<SaleContractVo>query()
            .like(StringUtils.isNotEmpty(saleContractVo.getCode()), "code", saleContractVo.getCode())
            .eq(StringUtils.isNotEmpty(saleContractVo.getContractType()), "contract_type", saleContractVo.getContractType())
            .like(StringUtils.isNotEmpty(saleContractVo.getCusCode()), "cus_code", saleContractVo.getCusCode())
            .like(StringUtils.isNotEmpty(saleContractVo.getCusName()), "cus_name", saleContractVo.getCusName())
            .like(StringUtils.isNotEmpty(saleContractVo.getChannelCode()), "channel_code", saleContractVo.getChannelCode())
            .like(StringUtils.isNotEmpty(saleContractVo.getChannelName()), "channel_name", saleContractVo.getChannelName())
            .eq(saleContractVo.getSignState() != null, "sign_state", saleContractVo.getSignState())
            .orderByDesc("create_date").orderByDesc("create_date_second");
    List<SaleContractVo> pageList = this.saleContractMapper.findContractPageByConditions(page, wrapper);
    return PageResult.<SaleContractVo>builder().data(pageList).count(page.getTotal()).build();
  }

  /**
   * 批量启用合同
   * @param ids
   */
  @Override
  public void enableBatch(List<String> ids) {
    if(CollectionUtil.listEmpty(ids)) {
      return;
    }
    UpdateWrapper<SaleContractEntity> wrapper = Wrappers.<SaleContractEntity>update()
            .set("enable_status", CrmEnableStatusEnum.ENABLE.getCode())
            .in("id", ids);
    this.update(wrapper);
  }

  /**
   * 批量禁用
   * @param ids
   */
  @Override
  public void disableBatch(List<String> ids) {
    if(CollectionUtil.listEmpty(ids)) {
      return;
    }
    UpdateWrapper<SaleContractEntity> wrapper = Wrappers.<SaleContractEntity>update()
            .set("enable_status", CrmEnableStatusEnum.DISABLE.getCode())
            .in("id", ids);
    this.update(wrapper);
  }

  /**
   * 客户签约
   * 1、校验入参
   * 2、校验数据是否合法
   * 3、修改签约状态
   * @param vo
   * @return
   */
  @Override
  @Transactional
  public SaleContractVo sign(SaleContractVo vo) {
    //1、
    ValidateUtils.validate(vo, "合同参数不能为空");
    //2、
    SaleContractVo dbVo = this.findDetailByCode(vo.getCode());
    ValidateUtils.validate(dbVo, "没有获取到原始合同记录");
    ValidateUtils.isTrue(!YesNoEnum.YesNoCodeNumberEnum.YES.getCode().equals(dbVo.getSignState()), "该合同已签约，请不要重新操作");
    //3、
    dbVo.setSignState(YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
    boolean signResult = this.updateById(CrmBeanUtil.copy(dbVo, SaleContractEntity.class));
    ValidateUtils.isTrue(signResult, "签约失败，可能是数据异常，请联系管理员检查后重试");
    return dbVo;
  }
}
