package com.biz.crm.salegoal.utils;

import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.dms.SaleGoalEunm;
import com.biz.crm.nebular.dms.salegoal.SaleGoalItemVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalRatioVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalVo;
import com.biz.crm.salegoal.entity.SaleGoalEntity;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.FieldHandleUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.UUIDGenerator;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.springframework.beans.BeanUtils;
import org.springframework.util.CollectionUtils;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/9/15 16:23
 **/
public class SaleGoalUtil {
  //每个对象维度对应的唯一约束字段
  private static final Map<Integer, List<String>> OBJECT_TYPE_FILED_MAP = Maps.newHashMap();
  //每个销售类型对应的唯一约束字段
  private static final Map<Integer, List<String>> GOAL_TYPE_FILED_MAP = Maps.newHashMap();

  static {
    //定义一个存放各类型字段的列表，为了日后可拓展，所以这里不用字符串，而用列表
    List<String> fieldList;
    //每个对象维度对应的唯一约束字段
    fieldList = Lists.newArrayList("cusCode");
    OBJECT_TYPE_FILED_MAP.put(SaleGoalEunm.ObjectType.CUSTOMER.getCode(), fieldList);
    fieldList = Lists.newArrayList("orgCode");
    OBJECT_TYPE_FILED_MAP.put(SaleGoalEunm.ObjectType.ORGANIZATION.getCode(), fieldList);
    fieldList = Lists.newArrayList("terminalCode");
    OBJECT_TYPE_FILED_MAP.put(SaleGoalEunm.ObjectType.TERMINAL.getCode(), fieldList);
    //每个销售类型对应的唯一约束字段
    GOAL_TYPE_FILED_MAP.put(SaleGoalEunm.GoalType.ROUTINE.getCode(), Lists.newArrayList());
    fieldList = Lists.newArrayList("goodsCode");
    GOAL_TYPE_FILED_MAP.put(SaleGoalEunm.GoalType.GOODS.getCode(), fieldList);
    fieldList = Lists.newArrayList("productLevelCode");
    GOAL_TYPE_FILED_MAP.put(SaleGoalEunm.GoalType.PRODUCT_LEVEL.getCode(), fieldList);
  }

  /**
   * 验证添加方法的参数
   * @param saleGoalVo
   */
  public static void validateAddParam(SaleGoalVo saleGoalVo) {
//    ValidateUtils.validate(saleGoalVo.getCusCode(), "客户编码不能为空!");
//    ValidateUtils.validate(saleGoalVo.getCusName(), "客户名称不能为空!");
//        ValidateUtils.validate(saleGoalVo.getCusOrgCode(),"客户所属组织编码不能为空!");
//        ValidateUtils.validate(saleGoalVo.getCusOrgName(),"客户所属组织名称不能为空!");
//        ValidateUtils.validate(saleGoalVo.getCusChannelCode(),"客户所属渠道编码不能为空!");
//        ValidateUtils.validate(saleGoalVo.getCusChannelName(),"客户所属渠道名称不能为空!");
    ValidateUtils.validate(saleGoalVo.getTargetYear(), "目标年份不能为空!");
    ValidateUtils.validate(saleGoalVo.getType(), "销量目标类型不能为空!");
    if (saleGoalVo.getType().intValue() == SaleGoalEunm.GoalType.PRODUCT_LEVEL.getCode().intValue()) {
      ValidateUtils.validate(saleGoalVo.getProductLevelCode(), "产品层级编码不能为空!");
      ValidateUtils.validate(saleGoalVo.getProductLevelName(), "产品层级名称不能为空!");
    }
    if (saleGoalVo.getType().intValue() == SaleGoalEunm.GoalType.GOODS.getCode().intValue()) {
      ValidateUtils.validate(saleGoalVo.getGoodsCode(), "商品编码不能为空!");
      ValidateUtils.validate(saleGoalVo.getGoodsName(), "商品名称不能为空!");
//            ValidateUtils.validate(saleGoalVo.getProductSpec(),"商品规格不能为空!");
//            ValidateUtils.validate(saleGoalVo.getProductLevelCode(),"产品层级编码不能为空!");
//            ValidateUtils.validate(saleGoalVo.getProductLevelName(),"产品层级名称不能为空!");
    }
  }

  /**
   * 验证参数(为addBatch方法)
   * 1、验证目标类型必须是同一个
   * 2、根据不同的对象维度，分别验证经销商、组织编码、终端编码必须是同一个
   * 3、验证详细参数
   * 4、验证参数中产品是否重复
   * @param saleGoalVos
   */
  public static void validateParamForAddBatch(List<SaleGoalVo> saleGoalVos) {
    if (CollectionUtils.isEmpty(saleGoalVos)) {
      return;
    }

    //1
    Integer type = saleGoalVos.get(0).getType();
    saleGoalVos.forEach(saleGoalVo -> {
      ValidateUtils.validate(saleGoalVo.getObjectType(), "终端类型不能为空");
      ValidateUtils.validate(saleGoalVo.getType(), "目标类型不能为空");
    });

    if (!CollectionUtils.isEmpty(
            saleGoalVos.stream().filter(vo -> !Objects.equals(type, vo.getType())).collect(Collectors.toList())
    )) {
      throw new BusinessException("暂不支持跨目标类型批量操作！");
    }

    //2
    Set<Integer> objectTypes = saleGoalVos.stream().map(SaleGoalVo::getObjectType).collect(Collectors.toSet());
    ValidateUtils.isTrue(objectTypes.size() == 1, "暂时仅支持同一个对象维度批量操作");
    switch (SaleGoalEunm.getObjectTypeByCode(objectTypes.iterator().next())) {
      case CUSTOMER:
        Set<String> cusCodes = saleGoalVos.stream().map(SaleGoalVo::getCusCode).collect(Collectors.toSet());
        ValidateUtils.isTrue(cusCodes.size() == 1, "暂时仅支持同一客户批量操作");
        ValidateUtils.validate(cusCodes.iterator().next(), "客户编码不能为空");
        break;
      case ORGANIZATION:
        Set<String> orgCodes = saleGoalVos.stream().map(SaleGoalVo::getOrgCode).collect(Collectors.toSet());
        ValidateUtils.isTrue(orgCodes.size() == 1, "暂时仅支持同一组织批量操作");
        ValidateUtils.validate(orgCodes.iterator().next(), "组织编码不能为空");
        break;
      case TERMINAL:
        Set<String> terminalCodes = saleGoalVos.stream().map(SaleGoalVo::getTerminalCode).collect(Collectors.toSet());
        ValidateUtils.isTrue(terminalCodes.size() == 1, "暂时仅支持同一组织批量操作");
        ValidateUtils.validate(terminalCodes.iterator().next(), "终端编码不能为空");
        break;
    }

    //3
    for (SaleGoalVo vo : saleGoalVos) {
      validateAddParam(vo);
    }

    //4
    Map<String, Object> map = new HashMap<>(saleGoalVos.size());//验证重复用的map
    for (SaleGoalVo vo : saleGoalVos) {
      //验重
      String key = "";
      String repeat = "";
      if (vo.getType().intValue() == SaleGoalEunm.GoalType.GOODS.getCode().intValue()) {
        key = new StringBuilder(vo.getGoodsCode()).append(vo.getTargetYear()).toString();
        repeat = vo.getGoodsName();
      } else if (vo.getType().intValue() == SaleGoalEunm.GoalType.PRODUCT_LEVEL.getCode().intValue()) {
        key = new StringBuilder(vo.getProductLevelCode()).append(vo.getTargetYear()).toString();
        repeat = vo.getProductLevelName();
      } else {
        key = vo.getTargetYear().toString();
      }
      if (map.get(key) != null) {
        StringBuilder msg = new StringBuilder("目标重复");
        if (!StringUtils.isEmpty(repeat)) {
          msg.append(",产品/产品层级:").append(repeat);
        }
        msg.append(",目标年份:").append(vo.getTargetYear());
        throw new BusinessException(msg.toString());
      }
      map.put(key, key);
    }
  }


  /**
   * 组装实体对象，为add方法
   * @return
   */
  public static SaleGoalEntity packageEntityForAdd(SaleGoalVo vo) {
    //设置唯一约束字段值
    buildOnlyKey(vo);
    SaleGoalEntity entity = new SaleGoalEntity();
    BeanUtils.copyProperties(vo, entity);
    entity.setId(UUIDGenerator.generate());
    if (vo.getType().intValue() == SaleGoalEunm.GoalType.PRODUCT_LEVEL.getCode().intValue()) {
      entity.setGoodsCode(null);
      entity.setGoodsName(null);
      entity.setProductSpec(null);
      entity.setOnlyKey(new StringBuilder(vo.getProductLevelCode()).append(",").append(vo.getTargetYear()).toString());
    }
    if (vo.getType().intValue() == SaleGoalEunm.GoalType.ROUTINE.getCode().intValue()) {
      entity.setGoodsCode(null);
      entity.setGoodsName(null);
      entity.setProductSpec(null);
      entity.setProductLevelCode(null);
      entity.setProductLevelName(null);
      entity.setOnlyKey(vo.getTargetYear().toString());
    }
    //全年总目标
    vo.getSaleGoalItemVo().setTaskType(vo.getTaskType());
    entity.setTargetNum(SaleGoalItemUtil.sumSingleItem(vo.getSaleGoalItemVo()).getYearlySum());
    return entity;
  }

  /**
   * 构建销售目标列表每个对象的onlyKey字段值
   * @param saleGoalVos
   */
  public static void buildOnlyKey(List<SaleGoalVo> saleGoalVos) {
    if (CollectionUtil.listEmpty(saleGoalVos)) {
      return;
    }
    saleGoalVos.forEach(sale -> buildOnlyKey(sale));
  }

  /**
   * 构建单个销售目标对象onlyKey字段值
   * @param sale
   */
  private static String buildOnlyKey(SaleGoalVo sale) {
    if (sale == null) {
      return null;
    }
    List<String> filedNames = Lists.newArrayList();
    filedNames.addAll(OBJECT_TYPE_FILED_MAP.get(sale.getObjectType()));
    filedNames.addAll(GOAL_TYPE_FILED_MAP.get(sale.getType()));
    filedNames.add("targetYear");
    Map<String, Object> valueMap = FieldHandleUtil.fieldsValue(sale, filedNames.toArray(new String[filedNames.size()]));
    Collection<Object> values = valueMap.values();
    List<String> valueList = Lists.newArrayList();
    values.forEach(v -> {
      ValidateUtils.validate(v, "%s不能为空");
      valueList.add(v.toString());
    });
    String onlyKey = String.join(",", valueList.toArray(new String[valueList.size()]));
    sale.setOnlyKey(onlyKey);
    return onlyKey;
  }

  /**
   * 构建一个默认的比例
   * @return
   */
  public static SaleGoalRatioVo packageDefaultRatioVo() {
    SaleGoalRatioVo ratioVo = new SaleGoalRatioVo();
    try {
      Class cls = ratioVo.getClass();
      Field[] fields = cls.getDeclaredFields();
      for (Field f : fields) {
        f.setAccessible(true);
        f.set(ratioVo, BigDecimal.valueOf(0.083));
      }
      Field field = cls.getDeclaredField("firstMonthRatio");
      field.setAccessible(true);
      field.set(ratioVo, BigDecimal.valueOf(0.087));
    } catch (IllegalAccessException | NoSuchFieldException e) {
      e.printStackTrace();
    }
    return ratioVo;
  }

  /**
   * 年目标量分摊导月度
   * @param ratioVo
   * @param vo
   */
  public static void sharingMonthByYear(SaleGoalRatioVo ratioVo, SaleGoalVo vo) {
    ValidateUtils.validate(vo, "销量目标不能为空");
    ValidateUtils.validate(vo.getSaleGoalItemVo(), "销量目标明细不能为空");
    SaleGoalEunm.SharingType sharingType = SaleGoalEunm.getSharingTypeByCode(vo.getSharingType());
    ValidateUtils.validate(sharingType, "销量目标分摊类型传入不正确，请检查");
    ValidateUtils.validate(ratioVo, "比例不能为空");
    SaleGoalItemVo itemVo = vo.getSaleGoalItemVo();
    //如果前端没有填入年总量，则不分摊
    if(itemVo.getYearlySum() == null || itemVo.getYearlySum().compareTo(BigDecimal.ZERO) == 0) {
      return;
    }
    switch (sharingType) {
      //不分摊
      case NONE:
        break;
        //分摊到季度
      case QUARTERLY:
        itemVo.setFirstQuarterNum(itemVo.getYearlySum().multiply(ratioVo.getFirstMonthRatio()));
        itemVo.setSecondQuarterNum(itemVo.getYearlySum().multiply(ratioVo.getSecondMonthRatio()));
        itemVo.setThirdQuarterNum(itemVo.getYearlySum().multiply(ratioVo.getThirdMonthRatio()));
        itemVo.setFourthQuarterNum(itemVo.getYearlySum().multiply(ratioVo.getFourthMonthRatio()));
        break;
        //分摊到月度
      case MONTHLY:
        itemVo.setJanTargetNum(itemVo.getYearlySum().multiply(ratioVo.getFirstMonthRatio()));
        itemVo.setFebTargetNum(itemVo.getYearlySum().multiply(ratioVo.getSecondMonthRatio()));
        itemVo.setMarTargetNum(itemVo.getYearlySum().multiply(ratioVo.getThirdMonthRatio()));
        itemVo.setAprTargetNum(itemVo.getYearlySum().multiply(ratioVo.getFourthMonthRatio()));
        itemVo.setMayTargetNum(itemVo.getYearlySum().multiply(ratioVo.getFifthMonthRatio()));
        itemVo.setJunTargetNum(itemVo.getYearlySum().multiply(ratioVo.getSixthMonthRatio()));
        itemVo.setJulTargetNum(itemVo.getYearlySum().multiply(ratioVo.getSeventhMonthRatio()));
        itemVo.setAugTargetNum(itemVo.getYearlySum().multiply(ratioVo.getEighthMonthRatio()));
        itemVo.setSepTargetNum(itemVo.getYearlySum().multiply(ratioVo.getNinthMonthRatio()));
        itemVo.setOctTargetNum(itemVo.getYearlySum().multiply(ratioVo.getTenthMonthRatio()));
        itemVo.setNovTargetNum(itemVo.getYearlySum().multiply(ratioVo.getEleventhMonthRatio()));
        itemVo.setDecTargetNum(itemVo.getYearlySum().multiply(ratioVo.getTwelfthMonthRatio()));
        break;
      default: break;
    }
  }
}
