package com.biz.crm.util;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.biz.crm.nebular.dms.contract.ContractVo;
import com.biz.crm.nebular.dms.salegoal.SaleGoalVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;

/**
 * bean对象拷贝工具类
 * @Author: chenrong
 * @Date: 2020/12/3 10:27
 */
public class BeanCopyUtil {

  private static final Logger LOGGER = LoggerFactory.getLogger(BeanCopyUtil.class);

  /**
   * 拷贝单个bean对象
   * @param source
   * @param cls
   * @return
   */
  public static<T> T copyBen(Object source, Class<T> cls) {
    if(source == null) {
      return null;
    }
    T target = null;
    try {
      target = cls.newInstance();
      BeanUtils.copyProperties(source, target);
    } catch (InstantiationException e) {
      LOGGER.error(e.getMessage(), e);
    } catch (IllegalAccessException e) {
      LOGGER.error(e.getMessage(), e);
    }
    ValidateUtils.validate(target, "对象拷贝失败");
    return target;
  }

  /**
   * 拷贝List
   * @param source
   * @param cls
   * @return
   */
  public static<T> List<T> copyList(Collection source, Class<T> cls) {
    if(!CollectionUtil.collectionNotEmpty(source)){
      return Lists.newArrayList();
    }
    List<T> list = Lists.newArrayList();
    source.forEach(f -> list.add(copyBen(f, cls)));
    return list;
  }

  /**
   * 拷贝Set
   * @param source
   * @param cls
   * @return
   */
  public static<T> Set<T> copySet(Collection source, Class<T> cls) {
    if(!CollectionUtil.collectionNotEmpty(source)){
      return Sets.newHashSet();
    }
    Set<T> set = Sets.newLinkedHashSet();
    source.forEach(f -> set.add(copyBen(f, cls)));
    return set;
  }

  /**
   * 对象转map
   * @param o
   * @return
   */
  public static Map<String, Object> objToMap(Object o) {
    Map<String, Object> result = Maps.newTreeMap();
    try {
      Class c = o.getClass();
      for (Field field : c.getDeclaredFields()) {
        field.setAccessible(true);
        result.put(field.getName(), field.get(o));
      }
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return Maps.newHashMap();
    }
    return result;
  }

  /**
   * 根据json对象解析java对象
   * @param jsonObject
   * @param <T>
   * @return
   */
  public static<T> T parseByJSONObject(JSONObject jsonObject, Class<T> cls) {
    if(jsonObject == null) {
      return null;
    }
    T object = jsonObject.toJavaObject(cls);
    return object;
  }

  /**
   * 根据json对象列表解析java对象列表
   * @param jsonArray
   * @param cls
   * @param <T>
   * @return
   */
  public static<T> List<T> parseByJSONArray(JSONArray jsonArray, Class<T> cls) {
    if(jsonArray == null || jsonArray.size() == 0) {
      return Lists.newArrayList();
    }
    String jsonStr = jsonArray.toJSONString();
    List<T> result = JSONArray.parseArray(jsonStr, cls);
    return result;
  }

  /**
   * 解析{@link JSONArray} 为java对象列表
   * @param data
   * @return
   */
  public static<T> JSONArray formatJSONArray(List<T> data) {
    if(CollectionUtil.listEmpty(data)) {
      return new JSONArray();
    }
    return JSONArray.parseArray(JSONArray.toJSONString(data));
  }
}

