package com.biz.crm.util;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.nebular.mdm.CrmBaseVo;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.biz.crm.nebular.mdm.constant.DictConstant.FEE_POOL_REDUCE_RED_TRICK_RULE;

/**
 * 处理，继承自{@link CrmBaseVo}的公共字段
 * @Author: chenrong
 * @Date: 2020/11/24 10:23
 */
public class FieldHandleUtil {

  private static final Logger LOGGER = LoggerFactory.getLogger(FieldHandleUtil.class);

  /**
   * 新增时初始化公共字段，一旦出错就抛异常，避免数据错乱
   * @param vo
   */
  public static void initCreateFields(Object vo) {
    if (vo == null || !(vo instanceof CrmBaseVo || vo instanceof CrmBaseEntity)) {
      return;
    }
    try {
      String[] timeStr = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()).split(" ");
      UserRedis user = UserUtils.getUser();
      if (user == null) {
        return;
      }
      Map<String, Object> attributes = Maps.newHashMap();
      attributes.put("delFlag", "1");
      attributes.put("enableStatus", "1");
      attributes.put("createDate", timeStr[0]);
      attributes.put("createDateSecond", timeStr[1]);
      attributes.put("createCode", user.getCustcode());
      attributes.put("createName", user.getCustname());
      attributes.put("createOrgCode", user.getOrgcode());
      attributes.put("createOrgName", user.getOrgname());
      attributes.put("createPosCode", user.getPoscode());
      attributes.put("createPosName", user.getPosname());
      setValue(attributes, vo);
    } catch (Exception e) {
      throw new BusinessException("处理公共字段失败", e);
    }
  }

  /**
   * 设置字段值
   * @param attributes
   * @param vo
   * @throws IllegalAccessException
   */
  private static void setValue(Map<String, Object> attributes, Object vo) throws IllegalAccessException {
    List<Field> fields = getFields(vo.getClass());
    Map<String, Field> fieldMap = Maps.newHashMap();
    fields.forEach(f -> fieldMap.put(f.getName(), f));
    for (Map.Entry<String, Object> entry : attributes.entrySet()) {
      Field field = fieldMap.get(entry.getKey());
      if (field == null) {
        continue;
      }
      field.setAccessible(true);
      field.set(vo, entry.getValue());
    }
  }


  /**
   * 获取所有字段，包括所有父类
   * @param c
   * @return
   */
  public static List<Field> getFields(Class c) {
    List<Field> fields = Lists.newArrayList();
    while (c != null) {
      fields.addAll(Arrays.asList(c.getDeclaredFields()));
      c = c.getSuperclass();
    }
    return fields;
  }

  /**
   * 修改时初始化公共字段，一旦出错就抛异常，避免数据错乱
   * @param vo
   */
  public static void initUpdateFields(Object vo) {
    if (vo == null || !(vo instanceof CrmBaseVo || vo instanceof CrmBaseEntity)) {
      return;
    }
    try {
      String[] timeStr = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()).split(" ");
      UserRedis user = UserUtils.getUser();
      if (user == null) {
        return;
      }
      Map<String, Object> attributes = Maps.newHashMap();
      attributes.put("updateDate", timeStr[0]);
      attributes.put("updateDateSecond", timeStr[1]);
      attributes.put("updateCode", user.getCustcode());
      attributes.put("updateName", user.getCustname());
      attributes.put("updateOrgCode", user.getOrgcode());
      attributes.put("updateOrgName", user.getOrgname());
      attributes.put("updatePosCode", user.getPoscode());
      attributes.put("updatePosName", user.getPosname());
      setValue(attributes, vo);
    } catch (Exception e) {
      LOGGER.error("data:{}", vo);
      throw new BusinessException("处理公共字段失败", e);
    }
  }

  /**
   * 获取对象中指定的字段值
   * @param o
   * @param fieldNames
   * @return
   */
  public static Map<String, Object> fieldsValue(Object o, String... fieldNames) {
    Map<String, Object> result = Maps.newHashMap();
    try {
      Class c = o.getClass();
      for (String fieldName : fieldNames) {
        Field field = c.getDeclaredField(fieldName);
        field.setAccessible(true);
        result.put(fieldName, field.get(o));
      }
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return Maps.newHashMap();
    }
    return result;
  }

  /**
   * java字段名转化成数据库字段名
   * @param sourceName
   * @return
   */
  public static String fieldHumpToLine(String sourceName) {
    Pattern humpPattern = Pattern.compile("[A-Z]");
    Matcher matcher = humpPattern.matcher(sourceName);
    StringBuffer sb = new StringBuffer();
    while (matcher.find()) {
      matcher.appendReplacement(sb, "_" + matcher.group(0).toLowerCase());
    }
    matcher.appendTail(sb);
    return sb.toString();
  }

  /**
   * 数据库字段名转化成java字段名
   * @param sourceName
   * @return
   */
  public static String fieldLineToHump(String sourceName) {
    Pattern linePattern = Pattern.compile("_(\\w)");
    Matcher matcher = linePattern.matcher(sourceName);
    StringBuffer sb = new StringBuffer();
    while (matcher.find()) {
      matcher.appendReplacement(sb, matcher.group(1).toUpperCase());
    }
    matcher.appendTail(sb);
    return sb.toString();
  }

  /**
   * 初始化为null的，{@link BigDecimal}类型字段
   * @param v
   */
  public static void initDecimalZero(Object v) {
    Class c = v.getClass();
    try {
      Field[] fields = c.getDeclaredFields();
      if (fields == null) {
        return;
      }
      for (Field field : fields) {
        field.setAccessible(true);
        if (field.getType() == BigDecimal.class && field.get(v) == null) {
          field.set(v, BigDecimal.ZERO);
        }
      }
    } catch (IllegalAccessException e) {
      LOGGER.error("data:{}", v);
      throw new BusinessException("处理空数字字段异常", e);
    }
  }

  /**
   * 根据字典排序规则构建查询排序条件
   * @param wrapper
   * @param indexDictCode
   * @return
   */
  public static QueryWrapper buildIndexByIndexDictCode(QueryWrapper wrapper, String indexDictCode) {
    ValidateUtils.validate(wrapper, "根据字典排序规则构建查询排序条件时，条件封装对象必须传入");
    List<DictDataVo> dictDataVos = DictUtil.getChildrenList(FEE_POOL_REDUCE_RED_TRICK_RULE, indexDictCode);
//    ValidateUtils.notEmpty(dictDataVos, "没有获取到扣减、冲销顺序字典数据，请确认字典是否配置，或者子类字典编码是否传递正确");
    //如果没有获取到，默认按时间排序
    if(CollectionUtil.listEmpty(dictDataVos)) {
      DictDataVo vo1 = new DictDataVo();
      vo1.setDictCode("create_date");
      vo1.setDictValue("asc");
      vo1.setDictSort(1);
      DictDataVo vo2 = new DictDataVo();
      vo2.setDictCode("create_date_second");
      vo2.setDictValue("asc");
      vo2.setDictSort(2);
      dictDataVos = Lists.newArrayList(vo1, vo2);
    }
    dictDataVos = dictDataVos.stream().sorted(Comparator.comparing(DictDataVo::getDictSort)).collect(Collectors.toList());
    dictDataVos.forEach(d -> wrapper.orderBy(true, Objects.equals("desc", d.getDictValue()) ? false : true, d.getDictCode()));
    return wrapper;
  }

}
