package com.biz.crm.pool.service;

import com.baomidou.mybatisplus.extension.service.IService;
import com.biz.crm.common.PageResult;
import com.biz.crm.nebular.fee.pool.req.*;
import com.biz.crm.nebular.fee.pool.resp.*;
import com.biz.crm.pool.model.FeePoolEntity;
import org.springframework.web.bind.annotation.RequestBody;

import java.math.BigDecimal;
import java.util.List;

/**
 * 费用池主表接口
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
public interface FeePoolService extends IService<FeePoolEntity> {

    /**
     * 费用池主表分页列表
     *
     * @param reqVo
     * @return
     */
    PageResult<FeePoolPageRespVo> findFeePoolPageList(FeePoolPageReqVo reqVo);

    /**
     * 费用池主表分页列表，根据配置编码
     *
     * @param reqVo
     * @return
     */
    PageResult<FeePoolPageRespVo> findFeePoolPageListByConfigCode(FeePoolPageReqVo reqVo);

    /**
     * 保存费用池
     *
     * @param reqVo
     * @return
     */
    void saveFeePool(FeePoolReqVo reqVo);

    /**
     * 根据费用池编号获取费用池主表详情
     *
     * @param poolCode
     * @return
     */
    FeePoolRespVo queryByPoolCode(String poolCode);

    /**
     * 查询费用池费用
     *
     * @param reqVo
     * @return
     */
    FeePoolAmountQueryRespVo queryPoolAmount(FeePoolAmountQueryReqVo reqVo);

    /**
     * 上账
     *
     * @param reqVo
     */
    void account(FeePoolAccountReqVo reqVo);

    /**
     * 上账费用池维度，有费用池编号更新，没有费用池编号新增
     *
     * @param reqVo
     */
    void accountFeePool(FeePoolAdjustReqVo reqVo);

    /**
     * 上账（批量）
     *
     * @param list
     */
    void accountBatch(List<FeePoolAccountReqVo> list);

    /**
     * 回滚操作
     *
     * @param rollbackCode
     */
    void rollback(String rollbackCode);

    /**
     * 调整，按照维度
     *
     * @param reqVo
     */
    void adjust(FeePoolAdjustReqVo reqVo);

    /**
     * 调整，按照费用池编号
     *
     * @param reqVo
     */
    void adjustByPoolCode(FeePoolAdjustReqVo reqVo);

    /**
     * 使用费用池
     *
     * @param reqVo
     */
    void usePool(FeePoolUseReqVo reqVo);

    /**
     * 使用费用池（批量）
     *
     * @param list
     */
    void usePoolBatch(List<FeePoolUseReqVo> list);

    /**
     * 先退回使用金额，再重新使用费用池金额
     *
     * @param reqVo
     */
    void usePoolAfterBackByFromCode(FeePoolUseReqVo reqVo);

    /**
     * 先退回使用金额，再重新使用费用池金额（批量）
     *
     * @param list
     */
    void usePoolAfterBackByFromCodeBatch(List<FeePoolUseReqVo> list);

    /**
     * 回退使用金额，根据来源单号
     *
     * @param reqVo
     */
    void backUseByFromCode(FeePoolUseBackReqVo reqVo);

    /**
     * 回退使用金额，根据来源单号（批量）
     *
     * @param list
     */
    void backUseByFromCodeBatch(List<FeePoolUseBackReqVo> list);

    /**
     * 通过费用池编码扣减
     *
     * @param reqVo
     */
    void useByPoolCode(FeePoolAdjustReqVo reqVo);

    /**
     * 通过费用池编码占用
     *
     * @param reqVo
     */
    void occupyByPoolCode(FeePoolAdjustReqVo reqVo);

    /**
     * 通过费用池编码冻结
     *
     * @param reqVo
     */
    void freezeByPoolCode(FeePoolAdjustReqVo reqVo);

    /**
     * 通过费用池编码解冻
     *
     * @param reqVo
     */
    void unfreezeByPoolCode(FeePoolAdjustReqVo reqVo);

    /**
     * 全量回退，根据来源单号
     *
     * @param reqVo
     */
    void backAllUseByFromCode(FeePoolUseBackAllReqVo reqVo);

    /**
     * 全量回退，根据来源单号（批量）
     *
     * @param list
     */
    void backAllUseByFromCodeBatch(List<FeePoolUseBackAllReqVo> list);

    /**
     * 占用费用池
     *
     * @param reqVo
     */
    void occupyPool(FeePoolUseReqVo reqVo);

    /**
     * 占用费用池（批量）
     *
     * @param list
     */
    void occupyPoolBatch(List<FeePoolUseReqVo> list);

    /**
     * 全量回退占用，根据来源单号
     *
     * @param reqVo
     */
    void backAllOccupyByFromCode(FeePoolOccupyBackAllReqVo reqVo);

    /**
     * 全量回退占用，根据来源单号（批量）
     *
     * @param list
     */
    void backAllOccupyByFromCodeBatch(List<FeePoolOccupyBackAllReqVo> list);

    /**
     * 占用变更为使用，根据来源单号
     *
     * @param reqVo
     */
    void occupyToUseByFromCode(FeePoolOccupyToUseReqVo reqVo);

    /**
     * 占用变更为使用，根据来源单号（批量）
     *
     * @param list
     */
    void occupyToUseByFromCodeBatch(List<FeePoolOccupyToUseReqVo> list);

    /**
     * 根据客户编码和费用池类型查询客户的金额汇总
     *
     * @param poolGroup    费用池分组
     * @param customerCode 客户编码
     * @param poolType     费用池类型
     * @return
     */
    FeePoolAmountRespVo queryAmountByCustomerCode(String poolGroup, String customerCode, String poolType);

    /**
     * 查询月度金额汇总
     *
     * @param poolGroup    费用池分组
     * @param customerCode 客户编码
     * @param poolType     费用池类型
     * @param yearMonth    年月 yyyy-MM
     * @return
     */
    FeePoolMonthAmountRespVo queryMonthAmountByCustomerCode(String poolGroup, String customerCode, String poolType, String yearMonth);

    /**
     * 增加总金额和可使用金额（单表操作）
     *
     * @param poolCode 费用池编号
     * @param amount   金额
     */
    void addTotalAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 增加已使用金额，减少可使用金额（单表操作）
     *
     * @param poolCode
     * @param amount
     */
    void addHasUseAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 减少已使用金额，增加可使用金额（单表操作）
     *
     * @param poolCode
     * @param amount
     */
    void subtractHasUseAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 增加占用金额，减少可使用金额（单表操作）
     *
     * @param poolCode
     * @param amount
     */
    void addOccupyAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 减少占用金额，增加可使用金额（单表操作）
     *
     * @param poolCode
     * @param amount
     */
    void subtractOccupyAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 增加冻结金额，减少可使用金额（单表操作）
     *
     * @param poolCode
     * @param amount
     */
    void addFreezeAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 减少冻结金额，增加可使用金额（单表操作）
     *
     * @param poolCode
     * @param amount
     */
    void subtractFreezeAmountByPoolCode(String poolCode, BigDecimal amount);

    /**
     * 回退折扣已使用金额
     *
     * @param operationType 操作类型
     * @param fromCode      来源单号
     * @param fromDesc      来源描述
     * @param backAmount    折扣回退金额
     * @param remarks       备注
     * @param fileList      附件
     */
    void backDiscountUseAmount(String operationType, String fromCode, String fromDesc, BigDecimal backAmount, String remarks, List<FeePoolFileReqVo> fileList);

    /**
     * 回退货补使用金额
     *
     * @param operationType 操作类型
     * @param fromCode      来源单号
     * @param fromDesc      来源描述
     * @param itemList      货补回退明细
     * @param remarks       备注
     * @param fileList      附件
     */
    void backGoodsUseAmount(String operationType, String fromCode, String fromDesc, List<FeePoolGoodsUseBackItemReqVo> itemList, String remarks, List<FeePoolFileReqVo> fileList);
}

