package com.biz.crm.pool.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.fee.FeePoolGroupEnum;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailLogPageReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailLogReqVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolDetailLogPageRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolFileRespVo;
import com.biz.crm.pool.mapper.FeePoolDetailLogMapper;
import com.biz.crm.pool.model.FeePoolDetailLogEntity;
import com.biz.crm.pool.service.FeePoolDetailLogService;
import com.biz.crm.pool.service.FeePoolFileService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.fee.FeePoolConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 费用池明细操作记录表接口实现
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "FeePoolDetailLogServiceExpandImpl")
public class FeePoolDetailLogServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<FeePoolDetailLogMapper, FeePoolDetailLogEntity> implements FeePoolDetailLogService {

    @Resource
    private FeePoolDetailLogMapper feePoolDetailLogMapper;

    @Resource
    private FeePoolFileService feePoolFileService;

    @Override
    public PageResult<FeePoolDetailLogPageRespVo> findFeePoolDetailLogList(FeePoolDetailLogPageReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getBeginDate())) {
            reqVo.setBeginDate(reqVo.getBeginDate() + " 00:00:00");
        }
        if (StringUtils.isNotEmpty(reqVo.getEndDate())) {
            reqVo.setEndDate(reqVo.getEndDate() + " 23:59:59");
        }
        if (StringUtils.isNotEmpty(reqVo.getPoolGroup())) {
            reqVo.setPoolGroup(FeePoolGroupEnum.DEFAULT.getValue());
        }
        Page<FeePoolDetailLogPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<FeePoolDetailLogPageRespVo> list = feePoolDetailLogMapper.findFeePoolDetailLogList(page, reqVo);
        this.queryAndSetFileList(list);
        return PageResult.<FeePoolDetailLogPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<FeePoolDetailLogPageRespVo> findFeePoolDetailLogListByConfigCode(FeePoolDetailLogPageReqVo reqVo) {
        String configCode = reqVo.getConfigCode();
        Assert.hasText(configCode, "缺失配置编码");

        String poolGroup = FeePoolConfigUtil.getConfigQueryPoolGroup(configCode);
        Assert.hasText(poolGroup, "配置【" + configCode + "】未配置查询的费用池分组");
        reqVo.setPoolGroup(poolGroup);

        List<String> poolTypeList = FeePoolConfigUtil.getConfigQueryPoolTypeList(configCode);
        Assert.notEmpty(poolTypeList, "配置【" + configCode + "】未配置查询的费用池类型");
        reqVo.setPoolTypeList(poolTypeList);

        List<String> operationTypeList = FeePoolConfigUtil.getConfigQueryOperationTypeList(configCode);
        Assert.notEmpty(operationTypeList, "配置【" + configCode + "】未配置查询的费用池操作方式");
        reqVo.setOperationTypeList(operationTypeList);

        List<String> useTypeList = FeePoolConfigUtil.getConfigUseTypeList(configCode);
        Assert.notEmpty(useTypeList, "配置【" + configCode + "】未配置使用费用类型");
        reqVo.setUseTypeList(useTypeList);

        return this.findFeePoolDetailLogList(reqVo);
    }

    @Override
    public PageResult<FeePoolDetailLogPageRespVo> findFeePoolDetailLogListByPoolCode(FeePoolDetailLogPageReqVo reqVo) {
        Assert.hasText(reqVo.getPoolCode(), "缺失费用池编号");
        return this.findFeePoolDetailLogList(reqVo);
    }

    @Override
    public PageResult<FeePoolDetailLogPageRespVo> findFeePoolDetailLogListByPoolDetailCode(FeePoolDetailLogPageReqVo reqVo) {
        Assert.hasText(reqVo.getPoolDetailCode(), "缺失费用池明细编号");
        return this.findFeePoolDetailLogList(reqVo);
    }

    @Override
    public PageResult<FeePoolDetailLogPageRespVo> findFeePoolDetailLogListByOperationCode(FeePoolDetailLogPageReqVo reqVo) {
        Assert.hasText(reqVo.getOperationCode(), "缺失操作记录编号");
        return this.findFeePoolDetailLogList(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void savePoolDetailLog(FeePoolDetailLogReqVo reqVo) {
        this.savePoolDetailLog(Collections.singletonList(reqVo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void savePoolDetailLog(List<FeePoolDetailLogReqVo> list) {
        this.saveBatch(CrmBeanUtil.copyList(list, FeePoolDetailLogEntity.class));
    }

    /**
     * 查询并设置附件
     *
     * @param list
     */
    protected void queryAndSetFileList(List<FeePoolDetailLogPageRespVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Set<String> collect = list.stream().map(FeePoolDetailLogPageRespVo::getOperationCode).collect(Collectors.toSet());
            final Map<String, List<FeePoolFileRespVo>> fileGroupMap = feePoolFileService.getFeePoolFileListGroupByOperationCodeList(new ArrayList<>(collect));
            if (!fileGroupMap.isEmpty()) {
                list.forEach(item -> {
                    if (fileGroupMap.containsKey(item.getOperationCode())) {
                        item.setFileList(fileGroupMap.get(item.getOperationCode()));
                    }
                });
            }
        }
    }

}
