package com.biz.crm.pool.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.nebular.fee.pool.req.FeePoolFileReqVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolFileRespVo;
import com.biz.crm.pool.mapper.FeePoolFileMapper;
import com.biz.crm.pool.model.FeePoolFileEntity;
import com.biz.crm.pool.service.FeePoolFileService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 费用池附件接口实现
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "FeePoolFileServiceExpandImpl")
public class FeePoolFileServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<FeePoolFileMapper, FeePoolFileEntity> implements FeePoolFileService {

    @Resource
    private FeePoolFileMapper feePoolFileMapper;

    @Override
    public Map<String, List<FeePoolFileRespVo>> getFeePoolFileListGroupByOperationCodeList(List<String> operationCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(operationCodeList)) {
            List<FeePoolFileEntity> fileEntityList = new ArrayList<>();
            List<List<String>> partition = Lists.partition(operationCodeList, 500);
            partition.forEach(item -> {
                List<FeePoolFileEntity> list = this.lambdaQuery()
                        .in(FeePoolFileEntity::getOperationCode, item)
                        .select(FeePoolFileEntity::getPoolCode, FeePoolFileEntity::getOperationCode, FeePoolFileEntity::getFileName, FeePoolFileEntity::getUrl)
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    fileEntityList.addAll(list);
                }
            });
            if (CollectionUtil.listNotEmptyNotSizeZero(fileEntityList)) {
                return CrmBeanUtil.copyList(fileEntityList, FeePoolFileRespVo.class).stream().collect(Collectors.groupingBy(FeePoolFileRespVo::getOperationCode));
            }
        }
        return Collections.emptyMap();
    }

    @Override
    public List<FeePoolFileRespVo> getFeePoolFileListByOperationCode(String operationCode) {
        if (StringUtils.isNotEmpty(operationCode)) {
            List<FeePoolFileEntity> list = this.lambdaQuery()
                    .eq(FeePoolFileEntity::getOperationCode, operationCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, FeePoolFileRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void savePoolFileList(String poolCode, String operationCode, List<FeePoolFileReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Assert.hasText(poolCode, "附件缺失费用池编号");
            Assert.hasText(operationCode, "附件缺失操作记录编号");
            list.forEach(item -> {
                Assert.hasText(item.getFileName(), "附件缺失名称");
                Assert.hasText(item.getUrl(), "附件缺失地址");
            });
            List<FeePoolFileEntity> entityList = CrmBeanUtil.copyList(list, FeePoolFileEntity.class);
            entityList.forEach(item -> {
                item.setPoolCode(poolCode);
                item.setOperationCode(operationCode);
            });
            this.saveBatch(entityList);
        }
    }
}
