package com.biz.crm.pool.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.nebular.fee.pool.req.*;
import com.biz.crm.nebular.fee.pool.resp.FeePoolAmountQueryRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolPageConfigVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolPageRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolRespVo;
import com.biz.crm.pool.service.FeePoolService;
import com.biz.crm.util.Result;
import com.biz.crm.util.fee.FeePoolConfigUtil;
import com.biz.crm.util.fee.FeePoolRollbackUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 费用池主表
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Slf4j
@RestController
@RequestMapping("/feePoolController")
@Api(tags = "FEE-产品组-费用池主表")
@CrmGlobalLog
public class FeePoolController {

    @Resource
    private FeePoolService feePoolService;

    @ApiOperation(value = "（费用池管理页面专用）费用池主表分页列表")
    @PostMapping("/findFeePoolPageList")
    public Result<PageResult<FeePoolPageRespVo>> findFeePoolPageList(@RequestBody FeePoolPageReqVo reqVo) {
        PageResult<FeePoolPageRespVo> result = feePoolService.findFeePoolPageList(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "（费用池管理页面专用）费用池主表分页列表")
    @PostMapping("/findFeePoolPageListByConfigCode")
    public Result<PageResult<FeePoolPageRespVo>> findFeePoolPageListByConfigCode(@RequestBody FeePoolPageReqVo reqVo) {
        PageResult<FeePoolPageRespVo> result = feePoolService.findFeePoolPageListByConfigCode(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "（费用池管理页面专用）通过配置编号查询页面配置", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "configCode", value = "配置编码", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/getFeePoolPageConfig")
    public Result<FeePoolPageConfigVo> getFeePoolPageConfig(@RequestParam(value = "configCode", required = true) String configCode) {
        return Result.ok(FeePoolConfigUtil.getConfig(configCode));
    }

    @ApiOperation(value = "（费用池管理页面专用）通过费用池编号查询费用池", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "poolCode", value = "费用池编号", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/queryByPoolCode")
    public Result<FeePoolRespVo> queryByPoolCode(@RequestParam(value = "poolCode", required = true) String poolCode) {
        return Result.ok(feePoolService.queryByPoolCode(poolCode));
    }

    @ApiOperation(value = "（费用池管理页面专用）维度调整")
    @PostMapping("/adjust")
    public Result adjust(@RequestBody FeePoolAdjustReqVo reqVo) {
        feePoolService.adjust(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "（费用池管理页面专用）调整，按照费用池编号")
    @PostMapping("/adjustByPoolCode")
    public Result adjustByPoolCode(@RequestBody FeePoolAdjustReqVo reqVo) {
        feePoolService.adjustByPoolCode(reqVo);
        return Result.ok();
    }

    /*------------------------------以下是Feign接口------------------------------*/

    @ApiOperation(value = "查询费用池可用金额/数量")
    @PostMapping("/queryPoolAmount")
    public Result<FeePoolAmountQueryRespVo> queryPoolAmount(@RequestBody FeePoolAmountQueryReqVo reqVo) {
        return Result.ok(feePoolService.queryPoolAmount(reqVo));
    }

    @ApiOperation(value = "上账")
    @PostMapping("/account")
    public Result<String> account(@RequestBody FeePoolAccountReqVo reqVo) {
        feePoolService.account(reqVo);
        Result<String> result = new Result<>();
        result.setResult(FeePoolRollbackUtil.generateRollbackCode());
        return result;
    }

    @ApiOperation(value = "上账（批量）")
    @PostMapping("/accountBatch")
    public Result<String> accountBatch(@RequestBody List<FeePoolAccountReqVo> list) {
        feePoolService.accountBatch(list);
        Result<String> result = new Result<>();
        result.setResult(FeePoolRollbackUtil.generateRollbackCode());
        return result;
    }

    @ApiOperation(value = "回滚")
    @PostMapping("/rollback")
    public Result rollback(@RequestParam(value = "rollbackCode",required = true) String rollbackCode) {
        feePoolService.rollback(rollbackCode);
        FeePoolRollbackUtil.removeRollbackCode(rollbackCode);
        return Result.ok();
    }

    @ApiOperation(value = "使用费用池")
    @PostMapping("/usePool")
    public Result usePool(@RequestBody FeePoolUseReqVo reqVo) {
        feePoolService.usePool(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "使用费用池（批量）")
    @PostMapping("/usePoolBatch")
    public Result usePoolBatch(@RequestBody List<FeePoolUseReqVo> list) {
        feePoolService.usePoolBatch(list);
        return Result.ok();
    }

    @ApiOperation(value = "先退回使用金额，再重新使用费用池金额")
    @PostMapping("/usePoolAfterBackByFromCode")
    public Result usePoolAfterBackByFromCode(@RequestBody FeePoolUseReqVo reqVo) {
        feePoolService.usePoolAfterBackByFromCode(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "先退回使用金额，再重新使用费用池金额（批量）")
    @PostMapping("/usePoolAfterBackByFromCodeBatch")
    public Result usePoolAfterBackByFromCodeBatch(@RequestBody List<FeePoolUseReqVo> list) {
        feePoolService.usePoolAfterBackByFromCodeBatch(list);
        return Result.ok();
    }

    @ApiOperation(value = "回退使用金额，根据来源单号")
    @PostMapping("/backUseByFromCode")
    public Result backUseByFromCode(@RequestBody FeePoolUseBackReqVo reqVo) {
        feePoolService.backUseByFromCode(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "回退使用金额，根据来源单号（批量）")
    @PostMapping("/backUseByFromCodeBatch")
    public Result backUseByFromCodeBatch(@RequestBody List<FeePoolUseBackReqVo> list) {
        feePoolService.backUseByFromCodeBatch(list);
        return Result.ok();
    }

    @ApiOperation(value = "全量回退使用金额，根据来源单号")
    @PostMapping("/backAllUseByFromCode")
    public Result backAllUseByFromCode(@RequestBody FeePoolUseBackAllReqVo reqVo) {
        feePoolService.backAllUseByFromCode(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "全量回退使用金额，根据来源单号（批量）")
    @PostMapping("/backAllUseByFromCodeBatch")
    public Result backAllUseByFromCodeBatch(@RequestBody List<FeePoolUseBackAllReqVo> list) {
        feePoolService.backAllUseByFromCodeBatch(list);
        return Result.ok();
    }

    @ApiOperation(value = "占用费用池")
    @PostMapping("/occupyPool")
    @Deprecated
    public Result occupyPool(@RequestBody FeePoolUseReqVo reqVo) {
        feePoolService.occupyPool(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "占用费用池（批量）")
    @PostMapping("/occupyPoolBatch")
    public Result occupyPoolBatch(@RequestBody List<FeePoolUseReqVo> list) {
        feePoolService.occupyPoolBatch(list);
        return Result.ok();
    }

    @ApiOperation(value = "全量回退占用金额，根据来源单号")
    @PostMapping("/backAllOccupyByFromCode")
    public Result backAllOccupyByFromCode(@RequestBody FeePoolOccupyBackAllReqVo reqVo) {
        feePoolService.backAllOccupyByFromCode(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "全量回退占用金额，根据来源单号（批量）")
    @PostMapping("/backAllOccupyByFromCodeBatch")
    public Result backAllOccupyByFromCodeBatch(@RequestBody List<FeePoolOccupyBackAllReqVo> list) {
        feePoolService.backAllOccupyByFromCodeBatch(list);
        return Result.ok();
    }

    @ApiOperation(value = "占用变更为使用，根据来源单号")
    @PostMapping("/occupyToUseByFromCode")
    public Result occupyToUseByFromCode(@RequestBody FeePoolOccupyToUseReqVo reqVo){
        feePoolService.occupyToUseByFromCode(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "占用变更为使用，根据来源单号（批量）")
    @PostMapping("/occupyToUseByFromCodeBatch")
    public Result occupyToUseByFromCodeBatch(@RequestBody List<FeePoolOccupyToUseReqVo> list){
        feePoolService.occupyToUseByFromCodeBatch(list);
        return Result.ok();
    }
}
