package com.biz.crm.pool.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.fee.FeePoolOperationTypeGroupEnum;
import com.biz.crm.eunm.fee.FeePoolTypeEnum;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailPageReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolOperationPageReqVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolAmountRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolDetailPageRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolMonthAmountRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolOperationPageRespVo;
import com.biz.crm.pool.service.FeePoolDetailService;
import com.biz.crm.pool.service.FeePoolOperationService;
import com.biz.crm.pool.service.FeePoolService;
import com.biz.crm.util.Result;
import com.biz.crm.util.fee.FeePoolConfigUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * 费用池主表
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Slf4j
@RestController
@RequestMapping("/feePoolDiscountController")
@Api(tags = "FEE-产品组-折扣相关接口（目前由DMS、小程序调用）")
@CrmGlobalLog
public class FeePoolDiscountController {

    @Resource
    private FeePoolService feePoolService;
    @Resource
    private FeePoolDetailService feePoolDetailService;
    @Resource
    private FeePoolOperationService feePoolOperationService;

    @ApiOperation(value = "（DMS、小程序）查询客户折扣费用池上账明细分页")
    @PostMapping("/findFeePoolDetailAccountList")
    @CrmDictMethod
    public Result<PageResult<FeePoolDetailPageRespVo>> findFeePoolDetailAccountList(@RequestBody FeePoolDetailPageReqVo reqVo) {
        Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");
        reqVo.setPoolType(FeePoolTypeEnum.DISCOUNT.getValue());
        return Result.ok(feePoolDetailService.findFeePoolDetailPageList(reqVo));
    }

    @ApiOperation(value = "（DMS、小程序）查询客户折扣费用池使用明细分页")
    @PostMapping("/findFeePoolUseList")
    @CrmDictMethod
    public Result<PageResult<FeePoolOperationPageRespVo>> findFeePoolUseList(@RequestBody FeePoolOperationPageReqVo reqVo) {
        Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");
        reqVo.setPoolType(FeePoolTypeEnum.DISCOUNT.getValue());
        List<String> operationTypeList = new ArrayList<>();
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.BACK));
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.USE));
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.FREEZE));
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.UNFREEZE));
        reqVo.setOperationTypeList(operationTypeList);
        return Result.ok(feePoolOperationService.findFeePoolOperationPageList(reqVo));
    }

    @ApiOperation(value = "通过客户编码获取客户下的折扣金额汇总", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "poolGroup", value = "费用池分组", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "customerCode", value = "客户编码", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/queryDiscountAmountByCustomerCode")
    public Result<FeePoolAmountRespVo> queryDiscountAmountByCustomerCode(@RequestParam(value = "poolGroup", required = false) String poolGroup, @RequestParam(value = "customerCode", required = true) String customerCode) {
        return Result.ok(feePoolService.queryAmountByCustomerCode(poolGroup, customerCode, FeePoolTypeEnum.DISCOUNT.getValue()));
    }

    @ApiOperation(value = "通过客户编码和年月获取客户下的月度折扣金额汇总", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "poolGroup", value = "费用池分组", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "customerCode", value = "客户编码", required = true, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "yearMonth", value = "年月 yyyy-MM", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/queryDiscountMonthAmountByCustomerCode")
    public Result<FeePoolMonthAmountRespVo> queryDiscountMonthAmountByCustomerCode(@RequestParam(value = "poolGroup", required = false) String poolGroup, @RequestParam(value = "customerCode", required = true) String customerCode, @RequestParam(value = "yearMonth", required = true) String yearMonth) {
        return Result.ok(feePoolService.queryMonthAmountByCustomerCode(poolGroup, customerCode, FeePoolTypeEnum.DISCOUNT.getValue(), yearMonth));
    }

}
