package com.biz.crm.pool.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.fee.FeePoolGroupEnum;
import com.biz.crm.eunm.fee.FeePoolOperationTypeGroupEnum;
import com.biz.crm.eunm.tpm.ActTypeEnum;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailAdjustReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailLogReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailPageReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolOperationReqVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolDetailPageRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolDetailRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolFileRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolRespVo;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.pool.mapper.FeePoolDetailMapper;
import com.biz.crm.pool.model.FeePoolDetailEntity;
import com.biz.crm.pool.service.*;
import com.biz.crm.util.*;
import com.biz.crm.util.fee.FeePoolConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 费用池明细表接口实现
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "FeePoolDetailServiceExpandImpl")
public class FeePoolDetailServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<FeePoolDetailMapper, FeePoolDetailEntity> implements FeePoolDetailService {

    @Resource
    private FeePoolDetailMapper feePoolDetailMapper;
    @Resource
    private FeePoolDetailLogService feePoolDetailLogService;
    @Resource
    private FeePoolService feePoolService;
    @Resource
    private FeePoolFileService feePoolFileService;
    @Resource
    private FeePoolOperationService feePoolOperationService;

    @Override
    public PageResult<FeePoolDetailPageRespVo> findFeePoolDetailPageList(FeePoolDetailPageReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getBeginDate())) {
            reqVo.setBeginDate(reqVo.getBeginDate() + " 00:00:00");
        }
        if (StringUtils.isNotEmpty(reqVo.getEndDate())) {
            reqVo.setEndDate(reqVo.getEndDate() + " 23:59:59");
        }
        if (StringUtils.isNotEmpty(reqVo.getPoolGroup())) {
            reqVo.setPoolGroup(FeePoolGroupEnum.DEFAULT.getValue());
        }
        Page<FeePoolDetailPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<FeePoolDetailPageRespVo> list = feePoolDetailMapper.findFeePoolDetailPageList(page, reqVo);
        this.queryAndSetFileList(list);
        return PageResult.<FeePoolDetailPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<FeePoolDetailPageRespVo> findFeePoolDetailPageListByConfigCode(FeePoolDetailPageReqVo reqVo) {
        String configCode = reqVo.getConfigCode();
        Assert.hasText(configCode, "缺失配置编码");

        String poolGroup = FeePoolConfigUtil.getConfigQueryPoolGroup(configCode);
        Assert.hasText(poolGroup, "配置【" + configCode + "】未配置查询的费用池分组");
        reqVo.setPoolGroup(poolGroup);

        List<String> poolTypeList = FeePoolConfigUtil.getConfigQueryPoolTypeList(configCode);
        Assert.notEmpty(poolTypeList, "配置【" + configCode + "】未配置查询的费用池类型");
        reqVo.setPoolTypeList(poolTypeList);

        List<String> operationTypeList = FeePoolConfigUtil.getConfigQueryOperationTypeList(configCode);
        Assert.notEmpty(operationTypeList, "配置【" + configCode + "】未配置查询的费用池操作方式");
        reqVo.setOperationTypeList(operationTypeList);

        List<String> useTypeList = FeePoolConfigUtil.getConfigUseTypeList(configCode);
        Assert.notEmpty(useTypeList, "配置【" + configCode + "】未配置使用费用类型");
        reqVo.setUseTypeList(useTypeList);

        return this.findFeePoolDetailPageList(reqVo);
    }

    @Override
    public PageResult<FeePoolDetailPageRespVo> findFeePoolDetailPageListByPoolCode(FeePoolDetailPageReqVo reqVo) {
        Assert.hasText(reqVo.getPoolCode(), "缺失费用池编号");
        return this.findFeePoolDetailPageList(reqVo);
    }

    @Override
    public FeePoolDetailRespVo queryByPoolDetailCode(String poolDetailCode) {
        if (StringUtils.isNotEmpty(poolDetailCode)) {
            FeePoolDetailEntity one = this.lambdaQuery()
                    .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                    .one();
            if (one != null) {
                FeePoolDetailRespVo poolDetail = new FeePoolDetailRespVo();
                FeePoolRespVo pool = feePoolService.queryByPoolCode(one.getPoolCode());
                if (pool != null) {
                    CrmBeanUtil.copyProperties(pool, poolDetail);
                }
                CrmBeanUtil.copyProperties(one, poolDetail);
                poolDetail.setFileList(feePoolFileService.getFeePoolFileListByOperationCode(poolDetail.getOperationCode()));
                return poolDetail;
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addTotalAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.notNull(poolDetail, "无效的费用池明细编号");
        poolDetail.setTotalAmount(poolDetail.getTotalAmount().add(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().add(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addHasUseAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.isTrue(poolDetail.getUsableAmount().compareTo(amount) >= 0, "使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        poolDetail.setHasUseAmount(poolDetail.getHasUseAmount().add(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().subtract(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void subtractHasUseAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.isTrue(poolDetail.getHasUseAmount().compareTo(amount) >= 0, "回退使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        poolDetail.setHasUseAmount(poolDetail.getHasUseAmount().subtract(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().add(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addOccupyAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.isTrue(poolDetail.getUsableAmount().compareTo(amount) >= 0, "占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        poolDetail.setOccupyAmount(poolDetail.getOccupyAmount().add(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().subtract(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void subtractOccupyAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.isTrue(poolDetail.getOccupyAmount().compareTo(amount) >= 0, "释放占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        poolDetail.setOccupyAmount(poolDetail.getOccupyAmount().subtract(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().add(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addFreezeAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.notNull(poolDetail, "无效的费用池明细编号");
        Assert.isTrue(poolDetail.getUsableAmount().compareTo(amount) >= 0, "冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        poolDetail.setFreezeAmount(poolDetail.getFreezeAmount().add(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().subtract(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void subtractFreezeAmountByPoolDetailCode(String poolDetailCode, BigDecimal amount) {
        Assert.hasText(poolDetailCode, "缺失费用池明细编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolDetailEntity poolDetail = this.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolDetailCode, poolDetailCode)
                .one();
        Assert.notNull(poolDetail, "无效的费用池明细编号");
        Assert.isTrue(poolDetail.getFreezeAmount().compareTo(amount) >= 0, "解冻" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        poolDetail.setFreezeAmount(poolDetail.getFreezeAmount().subtract(amount));
        poolDetail.setUsableAmount(poolDetail.getUsableAmount().add(amount));
        this.updateById(poolDetail);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void useByPoolDetailCode(FeePoolDetailAdjustReqVo reqVo) {

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        FeePoolDetailRespVo poolDetail = this.queryByPoolDetailCode(reqVo.getPoolDetailCode());
        Assert.notNull(poolDetail, "无效的费用池明细编号");
        Assert.isTrue(poolDetail.getUsableAmount().compareTo(reqVo.getAmount()) >= 0, "使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "超过可用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);

        //修改明细表
        this.addHasUseAmountByPoolDetailCode(reqVo.getPoolDetailCode(), reqVo.getAmount());

        //操作记录表
        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(poolDetail.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType()));
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(reqVo.getAmount().multiply(FeePoolOperationTypeGroupEnum.USE.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        //主表
        feePoolService.addHasUseAmountByPoolCode(poolDetail.getPoolCode(), reqVo.getAmount());

        //明细操作记录表
        FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
        detailLog.setPoolCode(poolDetail.getPoolCode());
        detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
        detailLog.setOperationCode(operationCode);
        detailLog.setOperationType(reqVo.getOperationType());
        detailLog.setFromCode(reqVo.getFromCode());
        detailLog.setFromDesc(DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType()));
        detailLog.setRemarks(reqVo.getRemarks());
        detailLog.setOperationAmount(reqVo.getAmount().multiply(FeePoolOperationTypeGroupEnum.USE.getUsableAmountWeight()));
        detailLog.setOperationDateTime(operationDateTime);
        feePoolDetailLogService.savePoolDetailLog(detailLog);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void freezeByPoolDetailCode(FeePoolDetailAdjustReqVo reqVo) {
        Assert.hasText(reqVo.getPoolDetailCode(), "缺失费用池明细编号");
        BigDecimal freezeAmount = reqVo.getAmount();
        Assert.isTrue(freezeAmount != null, "缺失冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(freezeAmount.compareTo(BigDecimal.ZERO) > 0, "冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "必须大于0");
        FeePoolDetailRespVo poolDetail = this.queryByPoolDetailCode(reqVo.getPoolDetailCode());
        Assert.notNull(poolDetail, "无效的费用池明细编号");
        Assert.isTrue(poolDetail.getUsableAmount().compareTo(freezeAmount) >= 0, "冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "大于可用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        //保存操作记录
        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(poolDetail.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType()));
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(freezeAmount.multiply(FeePoolOperationTypeGroupEnum.FREEZE.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        //修改费用池明细表
        this.addFreezeAmountByPoolDetailCode(poolDetail.getPoolDetailCode(), freezeAmount);

        //修改费用池主表
        feePoolService.addFreezeAmountByPoolCode(poolDetail.getPoolCode(), freezeAmount);

        //保存明细表操作记录
        FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
        detailLog.setPoolCode(poolDetail.getPoolCode());
        detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
        detailLog.setOperationCode(operationCode);
        detailLog.setOperationType(reqVo.getOperationType());
        detailLog.setFromCode(reqVo.getFromCode());
        detailLog.setFromDesc(DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType()));
        detailLog.setRemarks(reqVo.getRemarks());
        detailLog.setOperationAmount(freezeAmount.multiply(FeePoolOperationTypeGroupEnum.FREEZE.getUsableAmountWeight()));
        detailLog.setOperationDateTime(operationDateTime);
        feePoolDetailLogService.savePoolDetailLog(detailLog);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unfreezeByPoolDetailCode(FeePoolDetailAdjustReqVo reqVo) {
        Assert.hasText(reqVo.getPoolDetailCode(), "缺失费用池明细编号");
        BigDecimal unfreezeAmount = reqVo.getAmount();
        Assert.isTrue(unfreezeAmount != null, "缺失解冻" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(unfreezeAmount.compareTo(BigDecimal.ZERO) > 0, "冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "必须大于0");
        FeePoolDetailRespVo poolDetail = this.queryByPoolDetailCode(reqVo.getPoolDetailCode());
        Assert.notNull(poolDetail, "无效的费用池明细编号");
        Assert.isTrue(poolDetail.getFreezeAmount().compareTo(unfreezeAmount) >= 0, "解冻" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "大于已冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        //保存操作记录
        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(poolDetail.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType()));
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(unfreezeAmount.multiply(FeePoolOperationTypeGroupEnum.UNFREEZE.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        //修改费用池明细表
        this.subtractFreezeAmountByPoolDetailCode(poolDetail.getPoolDetailCode(), unfreezeAmount);

        //修改费用池主表
        feePoolService.subtractFreezeAmountByPoolCode(poolDetail.getPoolCode(), unfreezeAmount);

        //保存明细表操作记录
        FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
        detailLog.setPoolCode(poolDetail.getPoolCode());
        detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
        detailLog.setOperationCode(operationCode);
        detailLog.setOperationType(reqVo.getOperationType());
        detailLog.setFromCode(reqVo.getFromCode());
        detailLog.setFromDesc(DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType()));
        detailLog.setRemarks(reqVo.getRemarks());
        detailLog.setOperationAmount(unfreezeAmount.multiply(FeePoolOperationTypeGroupEnum.UNFREEZE.getUsableAmountWeight()));
        detailLog.setOperationDateTime(operationDateTime);
        feePoolDetailLogService.savePoolDetailLog(detailLog);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjustByPoolDetailCode(FeePoolDetailAdjustReqVo reqVo) {
        Assert.hasText(reqVo.getPoolDetailCode(), "缺失费用池明细编号");
        Assert.isTrue(reqVo.getAmount() != null, "缺失调整" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getAmount().compareTo(BigDecimal.ZERO) > 0, "调整" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "必须大于0");
        Assert.hasText(reqVo.getOperationType(), "缺失操作类型");
        FeePoolOperationTypeGroupEnum operationTypeGroup = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        Assert.isTrue(FeePoolOperationTypeGroupEnum.ACCOUNT != operationTypeGroup, "行调整不支持" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.ACCOUNT.getValue()) + "操作");
        Assert.isTrue(FeePoolOperationTypeGroupEnum.BACK != operationTypeGroup, "行调整不支持" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.BACK.getValue()) + "操作");
        switch (operationTypeGroup) {
            case USE:
                this.useByPoolDetailCode(reqVo);
                ;
                break;
            case FREEZE:
                this.freezeByPoolDetailCode(reqVo);
                ;
                break;
            case UNFREEZE:
                this.unfreezeByPoolDetailCode(reqVo);
                ;
                break;
            default:
                ;
        }
    }

    /**
     * 查询并设置附件
     *
     * @param list
     */
    protected void queryAndSetFileList(List<FeePoolDetailPageRespVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Set<String> collect = list.stream().map(FeePoolDetailPageRespVo::getOperationCode).collect(Collectors.toSet());
            final Map<String, List<FeePoolFileRespVo>> fileGroupMap = feePoolFileService.getFeePoolFileListGroupByOperationCodeList(new ArrayList<>(collect));
            if (!fileGroupMap.isEmpty()) {
                list.forEach(item -> {
                    if (fileGroupMap.containsKey(item.getOperationCode())) {
                        item.setFileList(fileGroupMap.get(item.getOperationCode()));
                    }
                    if(StringUtils.isNotEmpty(item.getActType())){
                        item.setActTypeName(ActTypeEnum.getStatusName(item.getActType()));
                    }
                });
            }
        }
    }

}
