package com.biz.crm.pool.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.fee.FeePoolGroupEnum;
import com.biz.crm.nebular.fee.pool.req.FeePoolOperationPageReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolOperationReqVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolFileRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolOperationPageRespVo;
import com.biz.crm.pool.mapper.FeePoolOperationMapper;
import com.biz.crm.pool.model.FeePoolOperationEntity;
import com.biz.crm.pool.service.FeePoolFileService;
import com.biz.crm.pool.service.FeePoolOperationService;
import com.biz.crm.util.*;
import com.biz.crm.util.fee.FeePoolConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 费用池操作记录表接口实现
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "FeePoolOperationServiceExpandImpl")
public class FeePoolOperationServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<FeePoolOperationMapper, FeePoolOperationEntity> implements FeePoolOperationService {

    @Resource
    private FeePoolOperationMapper feePoolOperationMapper;
    @Resource
    private FeePoolFileService feePoolFileService;

    @Override
    public PageResult<FeePoolOperationPageRespVo> findFeePoolOperationPageList(FeePoolOperationPageReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getBeginDate())) {
            reqVo.setBeginDate(reqVo.getBeginDate() + " 00:00:00");
        }
        if (StringUtils.isNotEmpty(reqVo.getEndDate())) {
            reqVo.setEndDate(reqVo.getEndDate() + " 23:59:59");
        }
        if (StringUtils.isNotEmpty(reqVo.getPoolGroup())) {
            reqVo.setPoolGroup(FeePoolGroupEnum.DEFAULT.getValue());
        }
        Page<FeePoolOperationPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<FeePoolOperationPageRespVo> list = feePoolOperationMapper.findFeePoolOperationPageList(page, reqVo);
        this.queryAndSetFileList(list);
        return PageResult.<FeePoolOperationPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<FeePoolOperationPageRespVo> findFeePoolOperationPageListByConfigCode(FeePoolOperationPageReqVo reqVo) {
        String configCode = reqVo.getConfigCode();
        Assert.hasText(configCode, "缺失配置编码");

        String poolGroup = FeePoolConfigUtil.getConfigQueryPoolGroup(configCode);
        Assert.hasText(poolGroup, "配置【" + configCode + "】未配置查询的费用池分组");
        reqVo.setPoolGroup(poolGroup);

        List<String> poolTypeList = FeePoolConfigUtil.getConfigQueryPoolTypeList(configCode);
        Assert.notEmpty(poolTypeList, "配置【" + configCode + "】未配置查询的费用池类型");
        reqVo.setPoolTypeList(poolTypeList);

        List<String> operationTypeList = FeePoolConfigUtil.getConfigQueryOperationTypeList(configCode);
        Assert.notEmpty(operationTypeList, "配置【" + configCode + "】未配置查询的费用池操作方式");
        reqVo.setOperationTypeList(operationTypeList);

        List<String> useTypeList = FeePoolConfigUtil.getConfigUseTypeList(configCode);
        Assert.notEmpty(useTypeList, "配置【" + configCode + "】未配置使用费用类型");
        reqVo.setUseTypeList(useTypeList);

        return this.findFeePoolOperationPageList(reqVo);
    }

    @Override
    public PageResult<FeePoolOperationPageRespVo> findFeePoolOperationPageListByPoolCode(FeePoolOperationPageReqVo reqVo) {
        Assert.hasText(reqVo.getPoolCode(), "缺失费用池编号");
        return this.findFeePoolOperationPageList(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String savePoolOperation(FeePoolOperationReqVo reqVo) {
        Assert.hasText(reqVo.getPoolCode(), "缺失费用池编号");
        Assert.hasText(reqVo.getOperationType(), "缺失操作类型");
        Assert.isTrue(reqVo.getOperationAmount() != null, "缺失操作金额");
        FeePoolOperationEntity entity = CrmBeanUtil.copy(reqVo, FeePoolOperationEntity.class);
        entity.setOperationCode(CodeUtil.getCodeDefault());
        this.save(entity);
        feePoolFileService.savePoolFileList(reqVo.getPoolCode(), entity.getOperationCode(), reqVo.getFileList());
        return entity.getOperationCode();
    }

    /**
     * 查询并设置附件
     *
     * @param list
     */
    protected void queryAndSetFileList(List<FeePoolOperationPageRespVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Set<String> collect = list.stream().map(FeePoolOperationPageRespVo::getOperationCode).collect(Collectors.toSet());
            final Map<String, List<FeePoolFileRespVo>> fileGroupMap = feePoolFileService.getFeePoolFileListGroupByOperationCodeList(new ArrayList<>(collect));
            if (!fileGroupMap.isEmpty()) {
                list.forEach(item -> {
                    if (fileGroupMap.containsKey(item.getOperationCode())) {
                        item.setFileList(fileGroupMap.get(item.getOperationCode()));
                    }
                });
            }
        }
    }

}
