package com.biz.crm.pool.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.fee.FeePoolGroupEnum;
import com.biz.crm.eunm.fee.FeePoolOperationTypeEnum;
import com.biz.crm.eunm.fee.FeePoolOperationTypeGroupEnum;
import com.biz.crm.eunm.fee.FeePoolTypeEnum;
import com.biz.crm.eunm.fee.FeePoolUseTypeEnum;
import com.biz.crm.mdm.customer.MdmCustomerMsgFeign;
import com.biz.crm.mdm.product.MdmProductFeign;
import com.biz.crm.nebular.fee.pool.req.FeePoolAccountReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolAdjustReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolAmountQueryReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolDetailLogReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolDiscountUseReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolFileReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolGoodsUseBackItemReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolOccupyBackAllReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolOccupyToUseReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolOperationReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolPageReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolUseBackAllReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolUseBackReqVo;
import com.biz.crm.nebular.fee.pool.req.FeePoolUseReqVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolAmountGroupByUseTypeRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolAmountQueryRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolAmountRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolMonthAmountRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolPageRespVo;
import com.biz.crm.nebular.fee.pool.resp.FeePoolRespVo;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductRespVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelRespVo;
import com.biz.crm.pool.mapper.FeePoolMapper;
import com.biz.crm.pool.model.FeePoolDetailEntity;
import com.biz.crm.pool.model.FeePoolDetailLogEntity;
import com.biz.crm.pool.model.FeePoolEntity;
import com.biz.crm.pool.model.FeePoolOperationEntity;
import com.biz.crm.pool.service.FeePoolDetailLogService;
import com.biz.crm.pool.service.FeePoolDetailService;
import com.biz.crm.pool.service.FeePoolGoodsService;
import com.biz.crm.pool.service.FeePoolOperationService;
import com.biz.crm.pool.service.FeePoolService;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.ProductLevelUtil;
import com.biz.crm.util.Result;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.fee.FeePoolConfigUtil;
import com.biz.crm.util.fee.FeePoolRollbackUtil;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 费用池主表接口实现
 *
 * @author Tao.Chen
 * @date 2021-01-20 13:44:35
 */
@Service
@ConditionalOnMissingBean(name = "FeePoolServiceExpandImpl")
public class FeePoolServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<FeePoolMapper, FeePoolEntity> implements FeePoolService {

    @Resource
    private FeePoolMapper feePoolMapper;
    @Resource
    private FeePoolOperationService feePoolOperationService;
    @Resource
    private FeePoolDetailService feePoolDetailService;
    @Resource
    private FeePoolDetailLogService feePoolDetailLogService;
    @Resource
    private MdmCustomerMsgFeign mdmCustomerMsgFeign;
    @Resource
    private MdmProductFeign mdmProductFeign;
    @Resource
    private FeePoolGoodsService feePoolGoodsService;

    @Override
    public PageResult<FeePoolPageRespVo> findFeePoolPageList(FeePoolPageReqVo reqVo) {
        Page<FeePoolPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (StringUtils.isEmpty(reqVo.getPoolGroup())) {
            reqVo.setPoolGroup(FeePoolGroupEnum.DEFAULT.getValue());
        }
        List<FeePoolPageRespVo> list = feePoolMapper.findFeePoolPageList(page, reqVo);
        return PageResult.<FeePoolPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<FeePoolPageRespVo> findFeePoolPageListByConfigCode(FeePoolPageReqVo reqVo) {
        String configCode = reqVo.getConfigCode();
        Assert.hasText(configCode, "缺失配置编码");

        String poolGroup = FeePoolConfigUtil.getConfigQueryPoolGroup(configCode);
        Assert.hasText(poolGroup, "配置【" + configCode + "】未配置查询的费用池分组");
        reqVo.setPoolGroup(poolGroup);

        List<String> poolTypeList = FeePoolConfigUtil.getConfigQueryPoolTypeList(configCode);
        Assert.notEmpty(poolTypeList, "配置【" + configCode + "】未配置查询的费用池类型");
        reqVo.setPoolTypeList(poolTypeList);

        List<String> useTypeList = FeePoolConfigUtil.getConfigUseTypeList(configCode);
        Assert.notEmpty(useTypeList, "配置【" + configCode + "】未配置使用费用类型");
        reqVo.setUseTypeList(useTypeList);

        return this.findFeePoolPageList(reqVo);
    }

    @Override
    public void saveFeePool(FeePoolReqVo reqVo) {
        FeePoolEntity entity = CrmBeanUtil.copy(reqVo, FeePoolEntity.class);
        this.save(entity);
    }

    @Override
    public FeePoolRespVo queryByPoolCode(String poolCode) {
        if (StringUtils.isNotEmpty(poolCode)) {
            FeePoolEntity one = this.lambdaQuery()
                    .eq(FeePoolEntity::getPoolCode, poolCode)
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, FeePoolRespVo.class);
            }
        }
        return null;
    }

    @Override
    public FeePoolAmountQueryRespVo queryPoolAmount(FeePoolAmountQueryReqVo reqVo) {
        FeePoolAmountQueryRespVo respVo = new FeePoolAmountQueryRespVo();
        if (StringUtils.isNotEmpty(reqVo.getCustomerCode())) {
            respVo.setDiscountUseTypeUsableAmountList(this.lambdaQuery()
                    .eq(FeePoolEntity::getPoolGroup, StringUtils.isNotEmpty(reqVo.getPoolGroup()) ? reqVo.getPoolGroup() : FeePoolGroupEnum.DEFAULT.getValue())
                    .eq(FeePoolEntity::getPoolType, FeePoolTypeEnum.DISCOUNT.getValue())
                    .eq(FeePoolEntity::getCustomerCode, reqVo.getCustomerCode())
                    .select(FeePoolEntity::getUseType, FeePoolEntity::getUsableAmount)
                    .list()
                    .stream().collect(Collectors.groupingBy(FeePoolEntity::getUseType)).entrySet().stream().map(x -> {
                        FeePoolAmountGroupByUseTypeRespVo useTypeRespVo = new FeePoolAmountGroupByUseTypeRespVo();
                        useTypeRespVo.setUseType(x.getKey());
                        useTypeRespVo.setUseTypeName(DictUtil.dictValue(DictConstant.FEE_POOL_USE_TYPE, x.getKey()));
                        useTypeRespVo.setAmount(x.getValue()
                                .stream()
                                .map(FeePoolEntity::getUsableAmount)
                                .reduce(BigDecimal.ZERO, BigDecimal::add));
                        return useTypeRespVo;
                    }).collect(Collectors.toList()));
            respVo.setDiscountUsableAmount(respVo.getDiscountUseTypeUsableAmountList()
                    .stream()
                    .map(FeePoolAmountGroupByUseTypeRespVo::getAmount)
                    .reduce(BigDecimal.ZERO, BigDecimal::add));
            respVo.setPoolUsableAmountMap(this.lambdaQuery()
                    .eq(FeePoolEntity::getCustomerCode, reqVo.getCustomerCode())
                    .in(CollectionUtil.listNotEmptyNotSizeZero(reqVo.getPoolCodeList()), FeePoolEntity::getPoolCode, reqVo.getPoolCodeList())
                    .select(FeePoolEntity::getPoolCode, FeePoolEntity::getUsableAmount)
                    .list()
                    .stream().collect(Collectors.toMap(FeePoolEntity::getPoolCode, FeePoolEntity::getUsableAmount)));
        }
        return respVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void account(FeePoolAccountReqVo reqVo) {
        Assert.hasText(reqVo.getPoolType(), "缺失费用池类型");
        FeePoolTypeEnum poolTypeEnum = FeePoolTypeEnum.getEnum(reqVo.getPoolType());
        Assert.notNull(poolTypeEnum, "无效的费用池类型");
        Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");
        Assert.hasText(reqVo.getOperationType(), "缺失操作类型");
        Assert.isTrue(reqVo.getAmount() != null, "缺失上账" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getAmount().compareTo(BigDecimal.ZERO) >= 0, "上账" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        Assert.isTrue(FeePoolConfigUtil.checkOperationTypeGroup(reqVo.getOperationType(), FeePoolOperationTypeGroupEnum.ACCOUNT), "该操作类型不支持上账");
        String payType = FeePoolConfigUtil.getPayType(poolTypeEnum.getValue());
        if (StringUtils.isEmpty(reqVo.getPoolGroup())) {
            reqVo.setPoolGroup(FeePoolGroupEnum.DEFAULT.getValue());
        }
        if (StringUtils.isEmpty(reqVo.getUseType())) {
            reqVo.setUseType(FeePoolUseTypeEnum.DEFAULT.getValue());
        }
        FeePoolAdjustReqVo adjust = CrmBeanUtil.copy(reqVo, FeePoolAdjustReqVo.class);

        if (FeePoolTypeEnum.GOODS == poolTypeEnum) {
            Assert.isTrue(StringUtils.isNotEmpty(reqVo.getProductLevelCode()) || CollectionUtil.listNotEmptyNotSizeZero(reqVo.getProductCodeList()), "产品层级和商品不能都为空");
            adjust.setGoodsProductLevelCode(reqVo.getProductLevelCode());
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getProductCodeList())) {
                adjust.setGoodsProductCode(String.join(",", reqVo.getProductCodeList()));
            }
        }

        adjust.setPoolGroup(reqVo.getPoolGroup());
        adjust.setUseType(reqVo.getUseType());
        adjust.setPoolType(poolTypeEnum.getValue());
        adjust.setCustomerCode(reqVo.getCustomerCode());
        adjust.setOperationType(reqVo.getOperationType());
        adjust.setFromCode(reqVo.getFromCode());
        adjust.setFromDesc(reqVo.getFromDesc());
        adjust.setRemarks(reqVo.getRemarks());
        adjust.setAmount(reqVo.getAmount());
        adjust.setPayType(payType);
        adjust.setFileList(reqVo.getFileList());
//        CrmBeanUtil.copyExt10(reqVo, adjust);
        this.adjust(adjust);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void accountFeePool(FeePoolAdjustReqVo reqVo) {

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        if (StringUtils.isNotEmpty(reqVo.getPoolCode())) {
            FeePoolRespVo pool = this.queryByPoolCode(reqVo.getPoolCode());
            Assert.notNull(pool, "无效的费用池编号");
            this.addTotalAmountByPoolCode(reqVo.getPoolCode(), reqVo.getAmount());
        } else {

            if (StringUtils.isEmpty(reqVo.getCustomerName())) {
                //验证客户编码是否有效
                Result<MdmCustomerMsgRespVo> customerResult = mdmCustomerMsgFeign.query(null, reqVo.getCustomerCode());
                Assert.isTrue(customerResult.isSuccess(), "查询客户信息失败，请稍后重试");
                MdmCustomerMsgRespVo customer = customerResult.getResult();
                Assert.isTrue(customer != null && StringUtils.isNotEmpty(customer.getCustomerCode()), "客户不存在");
                reqVo.setCustomerName(customer.getCustomerName());
            }
            if (StringUtils.isNotEmpty(reqVo.getGoodsProductLevelCode()) && StringUtils.isEmpty(reqVo.getGoodsProductLevelName())) {
                MdmProductLevelRespVo productLevel = ProductLevelUtil.getProductLevelByCode(reqVo.getGoodsProductLevelCode());
                Assert.isTrue(productLevel != null && StringUtils.isNotEmpty(productLevel.getProductLevelCode()), "产品层级不存在");
                reqVo.setGoodsProductLevelName(productLevel.getProductLevelName());
            }
            if (StringUtils.isNotEmpty(reqVo.getGoodsProductCode()) && StringUtils.isEmpty(reqVo.getGoodsProductName())) {
                Map<String, MdmProductRespVo> productMap = new HashMap<>(16);
                List<String> sortedProductCodeList = Arrays.stream(reqVo.getGoodsProductCode().split(",")).sorted().collect(Collectors.toList());
                Result<List<MdmProductRespVo>> productResult = mdmProductFeign.queryBatchByProductCodeList(sortedProductCodeList);
                Assert.isTrue(productResult.isSuccess(), "查询商品失败");
                productMap.putAll(productResult.getResult().stream().collect(Collectors.toMap(MdmProductRespVo::getProductCode, v -> v)));
                sortedProductCodeList.forEach(productCode -> {
                    Assert.isTrue(productMap.containsKey(productCode), "商品编码[" + productCode + "]无效");
                });
                List<String> sortedProductNameList = sortedProductCodeList.stream().map(productCode ->
                        productMap.get(productCode).getProductName()
                ).collect(Collectors.toList());
                reqVo.setGoodsProductCode(String.join(",", sortedProductCodeList));
            }

            FeePoolReqVo poolReqVo = CrmBeanUtil.copy(reqVo, FeePoolReqVo.class);
            poolReqVo.setPoolGroup(StringUtils.isNotEmpty(reqVo.getPoolGroup()) ? reqVo.getPoolGroup() : FeePoolGroupEnum.DEFAULT.getValue());
            poolReqVo.setPoolCode(CodeUtil.generateCode(CodeRuleEnum.FEE_POOL_CODE.getCode()));
            poolReqVo.setPayType(StringUtils.isNotEmpty(reqVo.getPayType()) ? reqVo.getPayType() : FeePoolConfigUtil.getPayType(reqVo.getPoolType()));
            poolReqVo.setUseType(StringUtils.isNotEmpty(reqVo.getUseType()) ? reqVo.getUseType() : FeePoolUseTypeEnum.DEFAULT.getValue());
            poolReqVo.setTotalAmount(reqVo.getAmount());
            poolReqVo.setUsableAmount(reqVo.getAmount());
            poolReqVo.setHasUseAmount(BigDecimal.ZERO);
            poolReqVo.setFreezeAmount(BigDecimal.ZERO);
            poolReqVo.setOccupyAmount(BigDecimal.ZERO);
            this.saveFeePool(poolReqVo);
            reqVo.setPoolCode(poolReqVo.getPoolCode());
        }

        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(reqVo.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(reqVo.getFromDesc());
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(reqVo.getAmount().multiply(FeePoolOperationTypeGroupEnum.ACCOUNT.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);
        FeePoolRollbackUtil.addRollbackOperationCode(operationCode);

        FeePoolDetailEntity poolDetail = CrmBeanUtil.copy(reqVo, FeePoolDetailEntity.class);
        poolDetail.setId(null);
//        CrmBeanUtil.copyExt10(reqVo, poolDetail);
        poolDetail.setPoolCode(reqVo.getPoolCode());
        poolDetail.setPoolDetailCode(CodeUtil.generateCode(CodeRuleEnum.FEE_POOL_DETAIL_CODE.getCode()));
        poolDetail.setOperationCode(operationCode);
        poolDetail.setOperationType(reqVo.getOperationType());
        poolDetail.setFromCode(reqVo.getFromCode());
        poolDetail.setFromDesc(reqVo.getFromDesc());
        poolDetail.setAccountDateTime(operationDateTime);
        poolDetail.setTotalAmount(reqVo.getAmount());
        poolDetail.setUsableAmount(reqVo.getAmount());
        poolDetail.setHasUseAmount(BigDecimal.ZERO);
        poolDetail.setFreezeAmount(BigDecimal.ZERO);
        poolDetail.setOccupyAmount(BigDecimal.ZERO);
        poolDetail.setRemarks(reqVo.getRemarks());
        feePoolDetailService.save(poolDetail);

        FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
        detailLog.setPoolCode(poolDetail.getPoolCode());
        detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
        detailLog.setOperationCode(operationCode);
        detailLog.setOperationType(reqVo.getOperationType());
        detailLog.setFromCode(reqVo.getFromCode());
        detailLog.setFromDesc(reqVo.getFromDesc());
        detailLog.setRemarks(reqVo.getRemarks());
        detailLog.setOperationAmount(reqVo.getAmount().multiply(FeePoolOperationTypeGroupEnum.ACCOUNT.getUsableAmountWeight()));
        detailLog.setOperationDateTime(operationDateTime);
        feePoolDetailLogService.savePoolDetailLog(detailLog);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void accountBatch(List<FeePoolAccountReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::account);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void rollback(String rollbackCode) {
        List<String> list = FeePoolRollbackUtil.getOperationCodeListByRollbackCode(rollbackCode);
        if (list != null && list.size() > 0) {
            List<FeePoolOperationEntity> operationEntityList = feePoolOperationService.lambdaQuery()
                    .in(FeePoolOperationEntity::getOperationCode, list)
                    .list();
            final Map<String, BigDecimal> poolOperationMap = operationEntityList.stream().collect(Collectors.groupingBy(FeePoolOperationEntity::getPoolCode, Collectors.mapping(FeePoolOperationEntity::getOperationAmount, Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));

            List<FeePoolEntity> poolEntityList = this.lambdaQuery()
                    .in(FeePoolEntity::getPoolCode, poolOperationMap.keySet())
                    .list();

            if (poolEntityList.size() > 0) {
                final Set<String> deletePoolCodeSet = poolEntityList.stream().filter(x -> {
                    if (x.getHasUseAmount().compareTo(BigDecimal.ZERO) == 0) {
                        if (x.getFreezeAmount().compareTo(BigDecimal.ZERO) == 0) {
                            if ((x.getOccupyAmount() == null ? BigDecimal.ZERO : x.getOccupyAmount()).compareTo(BigDecimal.ZERO) == 0) {
                                if (x.getTotalAmount().compareTo(x.getHasUseAmount()) == 0) {
                                    BigDecimal bigDecimal = poolOperationMap.get(x.getPoolCode());
                                    if (x.getTotalAmount().compareTo(bigDecimal) == 0) {
                                        return true;
                                    }
                                }
                            }
                        }
                    }
                    return false;
                }).map(FeePoolEntity::getPoolCode).collect(Collectors.toSet());

                List<FeePoolEntity> updatePoolEntityList = poolEntityList.stream().filter(x -> !deletePoolCodeSet.contains(x.getPoolCode())).collect(Collectors.toList());
                if (updatePoolEntityList.size() > 0) {
                    updatePoolEntityList.forEach(x -> {
                        BigDecimal bigDecimal = poolOperationMap.get(x.getPoolCode());
                        x.setTotalAmount(x.getTotalAmount().subtract(bigDecimal));
                        x.setUsableAmount(x.getUsableAmount().subtract(bigDecimal));
                    });
                    this.updateBatchById(updatePoolEntityList);
                }

                if (!deletePoolCodeSet.isEmpty()) {
                    this.lambdaUpdate()
                            .in(FeePoolEntity::getPoolCode, deletePoolCodeSet)
                            .remove();
                }
            }

            feePoolDetailLogService.lambdaUpdate()
                    .in(FeePoolDetailLogEntity::getOperationCode, list)
                    .remove();
            feePoolDetailService.lambdaUpdate()
                    .in(FeePoolDetailEntity::getOperationCode, list)
                    .remove();
            feePoolOperationService.lambdaUpdate()
                    .in(FeePoolOperationEntity::getOperationCode, list)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjust(FeePoolAdjustReqVo reqVo) {

        reqVo.setId(null);
        Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");
        Assert.hasText(reqVo.getOperationType(), "缺失调整类型");
        Assert.hasText(reqVo.getPoolType(), "缺失费用类型");
        FeePoolTypeEnum poolTypeEnum = FeePoolTypeEnum.getEnum(reqVo.getPoolType());
        Assert.notNull(poolTypeEnum, "无效的费用类型");
        Assert.isTrue(reqVo.getAmount() != null, "缺失" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getAmount().compareTo(BigDecimal.ZERO) > 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "必须大于0");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType());
        Assert.hasText(operationTypeValue, "无效的操作类型");

        //来源描述
        reqVo.setFromDesc(StringUtils.isNotEmpty(reqVo.getFromDesc()) ? reqVo.getFromDesc() : operationTypeValue);
        //操作类型分组
        FeePoolOperationTypeGroupEnum operationTypeGroup = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        //支付方式
        reqVo.setPayType(FeePoolConfigUtil.getPayType(poolTypeEnum.getValue()));
        //费用池分组
        reqVo.setPoolGroup(StringUtils.isNotEmpty(reqVo.getPoolGroup()) ? reqVo.getPoolGroup() : FeePoolGroupEnum.DEFAULT.getValue());
        //使用类型
        reqVo.setUseType(StringUtils.isNotEmpty(reqVo.getUseType()) ? reqVo.getUseType() : FeePoolUseTypeEnum.DEFAULT.getValue());

        Assert.isTrue(FeePoolOperationTypeGroupEnum.BACK != operationTypeGroup, "头操作不支持" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.BACK.getValue()) + "操作");

        //验证客户编码是否有效
        Result<MdmCustomerMsgRespVo> customerResult = mdmCustomerMsgFeign.query(null, reqVo.getCustomerCode());
        Assert.isTrue(customerResult.isSuccess(), "查询客户信息失败，请稍后重试");
        MdmCustomerMsgRespVo customer = customerResult.getResult();
        Assert.isTrue(customer != null && StringUtils.isNotEmpty(customer.getCustomerCode()), "客户不存在");

        FeePoolEntity pool = null;
        LambdaQueryWrapper<FeePoolEntity> poolLambdaQueryWrapper = new LambdaQueryWrapper<>();
        poolLambdaQueryWrapper.eq(FeePoolEntity::getPoolGroup, reqVo.getPoolGroup());
        poolLambdaQueryWrapper.eq(FeePoolEntity::getPoolType, poolTypeEnum.getValue());
        poolLambdaQueryWrapper.eq(FeePoolEntity::getUseType, reqVo.getUseType());
        poolLambdaQueryWrapper.eq(FeePoolEntity::getCustomerCode, reqVo.getCustomerCode());
        String goodsProductLevelCode = null;
        String goodsProductLevelName = null;
        String goodsProductCode = null;
        String goodsProductName = null;
        if (poolTypeEnum == FeePoolTypeEnum.DISCOUNT) {
            //折扣

        } else if (poolTypeEnum == FeePoolTypeEnum.GOODS) {
            //货补
            Assert.isTrue(StringUtils.isNotEmpty(reqVo.getGoodsProductLevelCode()) || StringUtils.isNotEmpty(reqVo.getGoodsProductCode()), "至少选择一个商品或一个产品层级");

            Map<String, MdmProductRespVo> productMap = new HashMap<>(16);
            if (StringUtils.isNotEmpty(reqVo.getGoodsProductCode())) {
                List<String> sortedProductCodeList = Arrays.stream(reqVo.getGoodsProductCode().split(",")).sorted().collect(Collectors.toList());
                Result<List<MdmProductRespVo>> productResult = mdmProductFeign.queryBatchByProductCodeList(sortedProductCodeList);
                Assert.isTrue(productResult.isSuccess(), "查询商品失败");
                productMap.putAll(productResult.getResult().stream().collect(Collectors.toMap(MdmProductRespVo::getProductCode, v -> v)));
                sortedProductCodeList.forEach(productCode -> {
                    Assert.isTrue(productMap.containsKey(productCode), "商品编码[" + productCode + "]无效");
                });
                List<String> sortedProductNameList = sortedProductCodeList.stream().map(productCode ->
                        productMap.get(productCode).getProductName()
                ).collect(Collectors.toList());
                reqVo.setGoodsProductCode(String.join(",", sortedProductCodeList));
                goodsProductCode = String.join(",", sortedProductCodeList);
                goodsProductName = String.join(",", sortedProductNameList);
            }
            if (StringUtils.isNotEmpty(reqVo.getGoodsProductLevelCode())) {
                MdmProductLevelRespVo productLevel = ProductLevelUtil.getProductLevelByCode(reqVo.getGoodsProductLevelCode());
                Assert.notNull(productLevel, "产品层级编码无效");
                goodsProductLevelCode = productLevel.getProductLevelCode();
                goodsProductLevelName = productLevel.getProductLevelName();
                final Set<String> childrenCodeSet = new HashSet<>(ProductLevelUtil.getChildrenProductLevelCodeListIncludeSelf(productLevel.getProductLevelCode()));
                if (StringUtils.isNotEmpty(reqVo.getGoodsProductCode())) {
                    List<String> goodsProductCodeList = Arrays.stream(reqVo.getGoodsProductCode().split(",")).collect(Collectors.toList());
                    goodsProductCodeList.forEach(productCode -> {
                        Assert.isTrue(childrenCodeSet.contains(productMap.get(productCode).getProductLevelCode()), "商品[" + productCode + "]不属于产品层级[" + reqVo.getGoodsProductLevelCode() + "]");
                    });
                }
            }

            if (StringUtils.isNotEmpty(goodsProductLevelCode)) {
                poolLambdaQueryWrapper.eq(FeePoolEntity::getGoodsProductLevelCode, goodsProductLevelCode);
            } else {
                poolLambdaQueryWrapper.and(x -> x.eq(FeePoolEntity::getGoodsProductLevelCode, "").or().isNull(FeePoolEntity::getGoodsProductLevelCode));
            }
            if (StringUtils.isNotEmpty(goodsProductCode)) {
                poolLambdaQueryWrapper.eq(FeePoolEntity::getGoodsProductCode, goodsProductCode);
            } else {
                poolLambdaQueryWrapper.and(x -> x.eq(FeePoolEntity::getGoodsProductCode, "").or().isNull(FeePoolEntity::getGoodsProductCode));
            }

        } else {
            //到这儿就有问题了
            throw new BusinessException("不识别的费用池类型");
        }

        final String goodsProductLevelCodeFinal = goodsProductLevelCode;
        final String goodsProductCodeFinal = goodsProductCode;
        List<FeePoolEntity> poolEntityList = feePoolMapper.selectList(poolLambdaQueryWrapper)
                .stream()
                .filter(x -> {
                    if (poolTypeEnum == FeePoolTypeEnum.GOODS) {
                        if (!(Optional.ofNullable(goodsProductLevelCodeFinal).orElse("")).equals(Optional.ofNullable(x.getGoodsProductLevelCode()).orElse(""))) {
                            return false;
                        }
                        if (!(Optional.ofNullable(goodsProductCodeFinal).orElse("")).equals(Optional.ofNullable(x.getGoodsProductCode()).orElse(""))) {
                            return false;
                        }
                    }
                    return true;
                }).collect(Collectors.toList());

        if (CollectionUtil.listNotEmptyNotSizeZero(poolEntityList)) {
            pool = poolEntityList.get(0);
            reqVo.setPoolCode(pool.getPoolCode());
        }

        switch (operationTypeGroup) {
            case ACCOUNT:
                if (FeePoolOperationTypeEnum.INIT.getValue().equals(reqVo.getOperationType())) {
                    if (pool != null) {
                        StringBuilder builder = new StringBuilder("已存在");
                        builder.append("客户【").append(customer.getCustomerName()).append("】");
                        if (!FeePoolGroupEnum.DEFAULT.getValue().equals(reqVo.getPoolGroup())) {
                            builder.append("，费用池分组【").append(DictUtil.dictValue(DictConstant.FEE_POOL_GROUP, reqVo.getPoolGroup())).append("】");
                        }
                        builder.append("，费用类型【").append(poolTypeEnum.getDesc()).append("】");
                        if (!FeePoolUseTypeEnum.DEFAULT.getValue().equals(reqVo.getUseType())) {
                            builder.append("，使用类型【").append(DictUtil.dictValue(DictConstant.FEE_POOL_USE_TYPE, reqVo.getUseType())).append("】");
                        }
                        if (FeePoolTypeEnum.GOODS == poolTypeEnum) {
                            if (StringUtils.isNotEmpty(goodsProductLevelName)) {
                                builder.append("，产品层级【").append(goodsProductLevelName).append("】");
                            }
                            if (StringUtils.isNotEmpty(goodsProductName)) {
                                builder.append("，商品【").append(goodsProductName).append("】");
                            }
                        }
                        builder.append("的费用维度，不能再期初上账");
                        throw new BusinessException(builder.toString());
                    }
                }

                reqVo.setCustomerName(customer.getCustomerName());
                reqVo.setGoodsProductLevelName(goodsProductLevelName);
                reqVo.setGoodsProductName(goodsProductName);
                this.accountFeePool(reqVo);

                break;
            case USE:
                Assert.notNull(pool, "不存在该维度，不能使用" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.USE.getValue()) + "操作");
                this.useByPoolCode(reqVo);
                break;
            case FREEZE:
                Assert.notNull(pool, "不存在该维度，不能使用" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.FREEZE.getValue()) + "操作");
                this.freezeByPoolCode(reqVo);
                break;
            case UNFREEZE:
                Assert.notNull(pool, "不存在该维度，不能使用" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.UNFREEZE.getValue()) + "操作");
                this.unfreezeByPoolCode(reqVo);
                break;
            default:
                ;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjustByPoolCode(FeePoolAdjustReqVo reqVo) {
        Assert.hasText(reqVo.getPoolCode(), "缺失费用池编号");
        Assert.hasText(reqVo.getOperationType(), "缺失调整类型");
        Assert.isTrue(reqVo.getAmount() != null, "缺失调整" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getAmount().compareTo(BigDecimal.ZERO) > 0, "调整" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "必须大于0");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType());
        Assert.hasText(operationTypeValue, "无效的操作类型");
        //操作类型分组
        FeePoolOperationTypeGroupEnum operationTypeGroup = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        Assert.isTrue(FeePoolOperationTypeGroupEnum.BACK != operationTypeGroup, "行操作不支持" + DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE_GROUP, FeePoolOperationTypeGroupEnum.BACK.getValue()) + "操作");

        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, reqVo.getPoolCode())
                .one();
        Assert.notNull(pool, "无效的费用池编号");
        String remarks = reqVo.getRemarks();
        CrmBeanUtil.copyProperties(pool, reqVo);
        //来源描述
        reqVo.setFromDesc(StringUtils.isNotEmpty(reqVo.getFromDesc()) ? reqVo.getFromDesc() : operationTypeValue);
        reqVo.setRemarks(remarks);

        switch (operationTypeGroup) {
            case ACCOUNT:
                this.accountFeePool(reqVo);
                break;
            case USE:
                this.useByPoolCode(reqVo);
                ;
                break;
            case FREEZE:
                this.freezeByPoolCode(reqVo);
                ;
                break;
            case UNFREEZE:
                this.unfreezeByPoolCode(reqVo);
                ;
                break;
            default:
                ;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void usePool(FeePoolUseReqVo reqVo) {

        Assert.hasText(reqVo.getOperationType(), "缺失操作类型");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType());
        reqVo.setFromDesc(StringUtils.isNotEmpty(reqVo.getFromDesc()) ? reqVo.getFromDesc() : operationTypeValue);
        List<FeePoolFileReqVo> fileList = reqVo.getFileList();
        Assert.isTrue(FeePoolConfigUtil.checkOperationTypeGroup(reqVo.getOperationType(), FeePoolOperationTypeGroupEnum.USE), "该操作类型不能使用费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getDiscount() != null || reqVo.getGoods() != null, "缺失使用明细");

        if (reqVo.getDiscount() != null && reqVo.getDiscount().getAmount().compareTo(BigDecimal.ZERO) != 0) {
            Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");

            //折扣使用
            FeePoolDiscountUseReqVo discount = reqVo.getDiscount();
            Assert.isTrue(discount.getAmount() != null, "缺失使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
            Assert.isTrue(discount.getAmount().compareTo(BigDecimal.ZERO) >= 0, "使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
            List<FeePoolEntity> feeDiscountList = this.lambdaQuery()
                    .eq(FeePoolEntity::getPoolGroup, StringUtils.isNotEmpty(reqVo.getPoolGroup()) ? reqVo.getPoolGroup() : FeePoolGroupEnum.DEFAULT.getValue())
                    .eq(FeePoolEntity::getCustomerCode, reqVo.getCustomerCode())
                    .eq(FeePoolEntity::getPoolType, FeePoolTypeEnum.DISCOUNT.getValue())
                    .eq(FeePoolEntity::getUseType, StringUtils.isNotEmpty(discount.getUseType()) ? discount.getUseType() : FeePoolUseTypeEnum.DEFAULT.getValue())
                    .list();
            Assert.notEmpty(feeDiscountList, "未找到客户[" + reqVo.getCustomerCode() + "]" + (StringUtils.isNotEmpty(discount.getUseType()) ? ("使用类型[" + DictUtil.dictValue(DictConstant.FEE_POOL_USE_TYPE, discount.getUseType()) + "]") : "") + "的折扣费用池");
            FeePoolEntity pool = feeDiscountList.get(0);

            FeePoolAdjustReqVo adjustReqVo = CrmBeanUtil.copy(pool, FeePoolAdjustReqVo.class);
            adjustReqVo.setAmount(discount.getAmount());
            adjustReqVo.setUseType(discount.getUseType());
            adjustReqVo.setFromCode(reqVo.getFromCode());
            adjustReqVo.setFromDesc(reqVo.getFromDesc());
            adjustReqVo.setOperationType(reqVo.getOperationType());
            adjustReqVo.setRemarks(reqVo.getRemarks());
            adjustReqVo.setFileList(reqVo.getFileList());
            this.useByPoolCode(adjustReqVo);
        }

        if (reqVo.getGoods() != null) {
            //货补使用

            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getGoods().getGoodsUseByPoolProductList())) {
                //按照费用池+商品使用
                feePoolGoodsService.useGoodsAmountByPoolProductList(reqVo.getPoolGroup(), reqVo.getGoods().getUseType(), reqVo.getOperationType(), reqVo.getFromCode(), reqVo.getFromDesc(), reqVo.getRemarks(), reqVo.getGoods().getGoodsUseByPoolProductList(), fileList);
            }
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getGoods().getGoodsUseByProductLevelList())) {
                //按照产品层级使用
                feePoolGoodsService.useGoodsAmountByProductLevelList(reqVo.getPoolGroup(), reqVo.getGoods().getUseType(), reqVo.getOperationType(), reqVo.getFromCode(), reqVo.getFromDesc(), reqVo.getRemarks(), reqVo.getGoods().getGoodsUseByProductLevelList(), fileList, reqVo.getCustomerCode());
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void usePoolBatch(List<FeePoolUseReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::usePool);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void usePoolAfterBackByFromCode(FeePoolUseReqVo reqVo) {
        String operationType = reqVo.getOperationType();
        Assert.hasText(reqVo.getFromCode(), "缺失来源单号");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        String fromDesc = StringUtils.isNotEmpty(reqVo.getFromDesc()) ? reqVo.getFromDesc() : operationTypeValue;

        Assert.isTrue(FeePoolConfigUtil.checkOperationTypeGroup(operationType, FeePoolOperationTypeGroupEnum.USE), "该操作类型不能使用费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getDiscount() != null || reqVo.getGoods() != null, "缺失使用明细");

        FeePoolUseBackAllReqVo backAllReqVo = new FeePoolUseBackAllReqVo();
        backAllReqVo.setFromCode(reqVo.getFromCode());
        backAllReqVo.setFromDesc(fromDesc + "先回退");
        backAllReqVo.setRemarks(reqVo.getRemarks());
        backAllReqVo.setOperationType(FeePoolOperationTypeEnum.ORDER_BACK.getValue());
        this.backAllUseByFromCode(backAllReqVo);
        this.usePool(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void usePoolAfterBackByFromCodeBatch(List<FeePoolUseReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::usePoolAfterBackByFromCode);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void backUseByFromCode(FeePoolUseBackReqVo reqVo) {
        String operationType = reqVo.getOperationType();
        Assert.hasText(operationType, "缺失操作类型");
        String fromCode = reqVo.getFromCode();
        Assert.hasText(fromCode, "缺失来源单号");
        String fromDesc = StringUtils.isNotEmpty(reqVo.getFromDesc()) ? reqVo.getFromDesc() : DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);

        if (reqVo.getDiscountBackAmount() != null && reqVo.getDiscountBackAmount().compareTo(BigDecimal.ZERO) != 0) {
            //折扣回退
            backDiscountUseAmount(operationType, fromCode, fromDesc, reqVo.getDiscountBackAmount(), reqVo.getRemarks(), null);
        }

        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getGoodsBackItemList())) {
            //货补回退
            backGoodsUseAmount(operationType, fromCode, fromDesc, reqVo.getGoodsBackItemList(), reqVo.getRemarks(), null);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void backUseByFromCodeBatch(List<FeePoolUseBackReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::backUseByFromCode);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void useByPoolCode(FeePoolAdjustReqVo reqVo) {
        FeePoolOperationTypeGroupEnum operationTypeGroupEnum = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        Assert.isTrue(operationTypeGroupEnum == FeePoolOperationTypeGroupEnum.USE, "该操作类型不支持使用扣减");

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        this.addHasUseAmountByPoolCode(reqVo.getPoolCode(), reqVo.getAmount());

        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(reqVo.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(reqVo.getFromDesc());
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(reqVo.getAmount().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        List<FeePoolDetailEntity> poolDetailList = feePoolDetailService.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolCode, reqVo.getPoolCode())
                .orderByAsc(FeePoolDetailEntity::getAccountDateTime)
                .list();
        BigDecimal restAmount = reqVo.getAmount();
        List<FeePoolDetailLogReqVo> detailLogList = new ArrayList<>();
        for (FeePoolDetailEntity poolDetail :
                poolDetailList) {
            if (restAmount.compareTo(BigDecimal.ZERO) <= 0) {
                break;
            }
            if (poolDetail.getUsableAmount().compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal itemUse = BigDecimal.ZERO;
                if (poolDetail.getUsableAmount().compareTo(restAmount) >= 0) {
                    itemUse = restAmount;
                } else {
                    itemUse = poolDetail.getUsableAmount();
                }
                restAmount = restAmount.subtract(itemUse);
                feePoolDetailService.addHasUseAmountByPoolDetailCode(poolDetail.getPoolDetailCode(), itemUse);

                FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                detailLog.setPoolCode(reqVo.getPoolCode());
                detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
                detailLog.setOperationCode(operationCode);
                detailLog.setOperationType(reqVo.getOperationType());
                detailLog.setFromCode(reqVo.getFromCode());
                detailLog.setFromDesc(reqVo.getFromDesc());
                detailLog.setRemarks(reqVo.getRemarks());
                detailLog.setOperationAmount(itemUse.multiply(operationTypeGroupEnum.getUsableAmountWeight()));
                detailLog.setOperationDateTime(operationDateTime);
                detailLogList.add(detailLog);
            }
        }
        Assert.isTrue(restAmount.compareTo(BigDecimal.ZERO) == 0, "费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "异常");
        if (CollectionUtil.listNotEmptyNotSizeZero(detailLogList)) {
            feePoolDetailLogService.savePoolDetailLog(detailLogList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void occupyByPoolCode(FeePoolAdjustReqVo reqVo) {
        FeePoolOperationTypeGroupEnum operationTypeGroupEnum = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        Assert.isTrue(operationTypeGroupEnum == FeePoolOperationTypeGroupEnum.OCCUPY, "该操作类型不支持使用占用");

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        this.addOccupyAmountByPoolCode(reqVo.getPoolCode(), reqVo.getAmount());

        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(reqVo.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(reqVo.getFromDesc());
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(reqVo.getAmount().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        List<FeePoolDetailEntity> poolDetailList = feePoolDetailService.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolCode, reqVo.getPoolCode())
                .orderByAsc(FeePoolDetailEntity::getAccountDateTime)
                .list();
        BigDecimal restAmount = reqVo.getAmount();
        List<FeePoolDetailLogReqVo> detailLogList = new ArrayList<>();
        for (FeePoolDetailEntity poolDetail :
                poolDetailList) {
            if (restAmount.compareTo(BigDecimal.ZERO) <= 0) {
                break;
            }
            if (poolDetail.getUsableAmount().compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal itemUse = BigDecimal.ZERO;
                if (poolDetail.getUsableAmount().compareTo(restAmount) >= 0) {
                    itemUse = restAmount;
                } else {
                    itemUse = poolDetail.getUsableAmount();
                }
                restAmount = restAmount.subtract(itemUse);
                feePoolDetailService.addOccupyAmountByPoolDetailCode(poolDetail.getPoolDetailCode(), itemUse);

                FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();

                detailLog.setPoolCode(reqVo.getPoolCode());
                detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
                detailLog.setOperationCode(operationCode);
                detailLog.setOperationType(reqVo.getOperationType());
                detailLog.setFromCode(reqVo.getFromCode());
                detailLog.setFromDesc(reqVo.getFromDesc());
                detailLog.setRemarks(reqVo.getRemarks());
                detailLog.setOperationAmount(itemUse.multiply(operationTypeGroupEnum.getUsableAmountWeight()));
                detailLog.setOperationDateTime(operationDateTime);
                detailLogList.add(detailLog);
            }
        }
        Assert.isTrue(restAmount.compareTo(BigDecimal.ZERO) == 0, "费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "异常");
        if (CollectionUtil.listNotEmptyNotSizeZero(detailLogList)) {
            feePoolDetailLogService.savePoolDetailLog(detailLogList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void freezeByPoolCode(FeePoolAdjustReqVo reqVo) {

        FeePoolOperationTypeGroupEnum operationTypeGroupEnum = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        Assert.isTrue(operationTypeGroupEnum == FeePoolOperationTypeGroupEnum.FREEZE, "该操作类型不支持冻结");

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        this.addFreezeAmountByPoolCode(reqVo.getPoolCode(), reqVo.getAmount());

        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(reqVo.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(reqVo.getFromDesc());
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(reqVo.getAmount().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        List<FeePoolDetailEntity> poolDetailList = feePoolDetailService.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolCode, reqVo.getPoolCode())
                .orderByAsc(FeePoolDetailEntity::getAccountDateTime)
                .list();
        BigDecimal restAmount = reqVo.getAmount();
        List<FeePoolDetailLogReqVo> detailLogList = new ArrayList<>();
        for (FeePoolDetailEntity poolDetail :
                poolDetailList) {
            if (restAmount.compareTo(BigDecimal.ZERO) <= 0) {
                break;
            }
            if (poolDetail.getUsableAmount().compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal itemUse = BigDecimal.ZERO;
                if (poolDetail.getUsableAmount().compareTo(restAmount) >= 0) {
                    itemUse = restAmount;
                } else {
                    itemUse = poolDetail.getUsableAmount();
                }
                restAmount = restAmount.subtract(itemUse);
                feePoolDetailService.addFreezeAmountByPoolDetailCode(poolDetail.getPoolDetailCode(), itemUse);

                FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                detailLog.setPoolCode(reqVo.getPoolCode());
                detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
                detailLog.setOperationCode(operationCode);
                detailLog.setOperationType(reqVo.getOperationType());
                detailLog.setFromCode(reqVo.getFromCode());
                detailLog.setFromDesc(reqVo.getFromDesc());
                detailLog.setRemarks(reqVo.getRemarks());
                detailLog.setOperationAmount(itemUse.multiply(operationTypeGroupEnum.getUsableAmountWeight()));
                detailLog.setOperationDateTime(operationDateTime);
                detailLogList.add(detailLog);
            }
        }
        Assert.isTrue(restAmount.compareTo(BigDecimal.ZERO) == 0, "费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "异常");
        if (CollectionUtil.listNotEmptyNotSizeZero(detailLogList)) {
            feePoolDetailLogService.savePoolDetailLog(detailLogList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unfreezeByPoolCode(FeePoolAdjustReqVo reqVo) {

        FeePoolOperationTypeGroupEnum operationTypeGroupEnum = FeePoolConfigUtil.getOperationTypeGroup(reqVo.getOperationType());
        Assert.isTrue(operationTypeGroupEnum == FeePoolOperationTypeGroupEnum.UNFREEZE, "该操作类型不支持解冻");

        //操作时间
        String operationDateTime = DateUtil.dateNowHms();

        //修改主表
        this.subtractFreezeAmountByPoolCode(reqVo.getPoolCode(), reqVo.getAmount());

        //主表操作记录表
        FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
        operationReqVo.setPoolCode(reqVo.getPoolCode());
        operationReqVo.setOperationType(reqVo.getOperationType());
        operationReqVo.setFromCode(reqVo.getFromCode());
        operationReqVo.setFromDesc(reqVo.getFromDesc());
        operationReqVo.setOperationDateTime(operationDateTime);
        operationReqVo.setOperationAmount(reqVo.getAmount().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
        operationReqVo.setFileList(reqVo.getFileList());
        operationReqVo.setRemarks(reqVo.getRemarks());
        String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

        List<FeePoolDetailEntity> poolDetailList = feePoolDetailService.lambdaQuery()
                .eq(FeePoolDetailEntity::getPoolCode, reqVo.getPoolCode())
                .orderByAsc(FeePoolDetailEntity::getAccountDateTime)
                .list();
        BigDecimal restAmount = reqVo.getAmount();
        List<FeePoolDetailLogReqVo> detailLogList = new ArrayList<>();
        for (FeePoolDetailEntity poolDetail :
                poolDetailList) {
            if (restAmount.compareTo(BigDecimal.ZERO) <= 0) {
                break;
            }
            if (poolDetail.getFreezeAmount().compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal itemUse = BigDecimal.ZERO;
                if (poolDetail.getFreezeAmount().compareTo(restAmount) >= 0) {
                    itemUse = restAmount;
                } else {
                    itemUse = poolDetail.getFreezeAmount();
                }
                restAmount = restAmount.subtract(itemUse);
                feePoolDetailService.subtractFreezeAmountByPoolDetailCode(poolDetail.getPoolDetailCode(), itemUse);

                FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                detailLog.setPoolCode(reqVo.getPoolCode());
                detailLog.setPoolDetailCode(poolDetail.getPoolDetailCode());
                detailLog.setOperationCode(operationCode);
                detailLog.setOperationType(reqVo.getOperationType());
                detailLog.setFromCode(reqVo.getFromCode());
                detailLog.setFromDesc(reqVo.getFromDesc());
                detailLog.setRemarks(reqVo.getRemarks());
                detailLog.setOperationAmount(itemUse.multiply(operationTypeGroupEnum.getUsableAmountWeight()));
                detailLog.setOperationDateTime(operationDateTime);
                detailLogList.add(detailLog);
            }
        }
        Assert.isTrue(restAmount.compareTo(BigDecimal.ZERO) == 0, "费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "异常");
        if (CollectionUtil.listNotEmptyNotSizeZero(detailLogList)) {
            feePoolDetailLogService.savePoolDetailLog(detailLogList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void backAllUseByFromCode(FeePoolUseBackAllReqVo reqVo) {
        String fromCode = reqVo.getFromCode();
        String fromDesc = reqVo.getFromDesc();
        String remarks = reqVo.getRemarks();
        Assert.hasText(fromCode, "缺失来源单号");
        String operationType = reqVo.getOperationType();
        if (StringUtils.isEmpty(operationType)) {
            operationType = FeePoolOperationTypeEnum.ORDER_BACK.getValue();
        }
        FeePoolOperationTypeGroupEnum operationTypeGroupEnum = FeePoolConfigUtil.getOperationTypeGroup(operationType);
        Assert.isTrue(operationTypeGroupEnum == FeePoolOperationTypeGroupEnum.BACK, "操作类型不支持回退");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);

        Set<String> operationTypeSet = new HashSet<>(16);
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.USE));
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.BACK));
        List<FeePoolOperationEntity> list = feePoolOperationService.lambdaQuery()
                .eq(FeePoolOperationEntity::getFromCode, fromCode)
                .in(FeePoolOperationTypeEnum.ORDER_BACK.getValue().equals(reqVo.getOperationType()), FeePoolOperationEntity::getOperationType, Arrays.asList(FeePoolOperationTypeEnum.ORDER_USE.getValue(), FeePoolOperationTypeEnum.ORDER_BACK.getValue()))
                .in(FeePoolOperationEntity::getOperationType, operationTypeSet)
                .list();
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        final Map<String, List<FeePoolOperationEntity>> poolOperationGroup = list.stream().collect(Collectors.groupingBy(FeePoolOperationEntity::getPoolCode));

        String operationDateTime = DateUtil.dateNowHms();
        for (Map.Entry<String, List<FeePoolOperationEntity>> entry :
                poolOperationGroup.entrySet()) {
            String poolCode = entry.getKey();
            List<FeePoolOperationEntity> operationEntityList = entry.getValue();
            BigDecimal operationAmountTotal = operationEntityList.stream().map(FeePoolOperationEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            if (operationAmountTotal.compareTo(BigDecimal.ZERO) == 0) {
                continue;
            }
            Assert.isTrue(operationAmountTotal.compareTo(BigDecimal.ZERO) <= 0, "费用池【" + poolCode + "】数据异常");
            List<FeePoolDetailLogEntity> detailLogList = feePoolDetailLogService.lambdaQuery()
                    .eq(FeePoolDetailLogEntity::getPoolCode, poolCode)
                    .eq(FeePoolDetailLogEntity::getFromCode, fromCode)
                    .in(FeePoolOperationTypeEnum.ORDER_BACK.getValue().equals(reqVo.getOperationType()), FeePoolDetailLogEntity::getOperationType, Arrays.asList(FeePoolOperationTypeEnum.ORDER_USE.getValue(), FeePoolOperationTypeEnum.ORDER_BACK.getValue()))
                    .in(FeePoolDetailLogEntity::getOperationType, operationTypeSet)
                    .list();
            Assert.notEmpty(detailLogList, "费用池【" + poolCode + "】数据异常");

            final Set<String> poolDetailCodeSet = detailLogList.stream().map(FeePoolDetailLogEntity::getPoolDetailCode).collect(Collectors.toSet());
            final Map<String, String> productCodeNameMap = detailLogList.stream().filter(x -> StringUtils.isNotEmpty(x.getProductCode())).collect(Collectors.toMap(FeePoolDetailLogEntity::getProductCode, FeePoolDetailLogEntity::getProductName, (x1, x2) -> x1));

            Map<String, FeePoolDetailEntity> poolDetailEntityMap = feePoolDetailService.lambdaQuery()
                    .in(FeePoolDetailEntity::getPoolDetailCode, poolDetailCodeSet)
                    .list()
                    .stream().collect(Collectors.toMap(FeePoolDetailEntity::getPoolDetailCode, v -> v));

            List<FeePoolDetailEntity> hasEditDetailEntityList = new ArrayList<>();
            FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
            operationReqVo.setPoolCode(poolCode);
            operationReqVo.setOperationType(operationType);
            operationReqVo.setFromCode(fromCode);
            operationReqVo.setFromDesc(StringUtils.isNotEmpty(fromDesc) ? fromDesc : operationTypeValue);
            operationReqVo.setOperationDateTime(operationDateTime);
            operationReqVo.setOperationAmount(operationAmountTotal.abs().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
            operationReqVo.setFileList(null);
            operationReqVo.setRemarks(remarks);
            String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);
            List<FeePoolDetailLogReqVo> logList = new ArrayList<>();

            Map<String, List<FeePoolDetailLogEntity>> groupByDetailCodeMap = detailLogList.stream().collect(Collectors.groupingBy(FeePoolDetailLogEntity::getPoolDetailCode));
            for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByDetailCodeEntry :
                    groupByDetailCodeMap.entrySet()) {
                String poolDetailCode = groupByDetailCodeEntry.getKey();
                List<FeePoolDetailLogEntity> groupByDetailLogList = groupByDetailCodeEntry.getValue();
                BigDecimal thisDetailAmount = groupByDetailLogList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                Assert.isTrue(thisDetailAmount.compareTo(BigDecimal.ZERO) <= 0, "费用池明细【" + poolDetailCode + "】退还" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "有误");
                Assert.isTrue(poolDetailEntityMap.containsKey(poolDetailCode), "费用池【" + poolCode + "】数据有误");
                Map<String, List<FeePoolDetailLogEntity>> groupByProductCodeMap = groupByDetailLogList.stream().collect(Collectors.groupingBy(x -> x.getProductCode() == null ? "" : x.getProductCode()));
                for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByProductCodeEntry :
                        groupByProductCodeMap.entrySet()) {
                    String productCode = groupByProductCodeEntry.getKey();
                    List<FeePoolDetailLogEntity> groupByProductLogList = groupByProductCodeEntry.getValue();

                    BigDecimal thisDetailProductAmount = groupByProductLogList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                    Assert.isTrue(thisDetailProductAmount.compareTo(BigDecimal.ZERO) <= 0, "费用池明细【" + poolDetailCode + "】" + (StringUtils.isNotEmpty(productCode) ? ("商品【" + productCode + "】") : "") + "退还" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "有误");
                    FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                    detailLog.setPoolCode(poolCode);
                    detailLog.setPoolDetailCode(poolDetailCode);
                    detailLog.setOperationCode(operationCode);
                    detailLog.setOperationType(operationType);
                    detailLog.setFromCode(fromCode);
                    detailLog.setFromDesc(StringUtils.isNotEmpty(fromDesc) ? fromDesc : operationTypeValue);
                    detailLog.setRemarks(remarks);
                    detailLog.setOperationAmount(thisDetailProductAmount.abs().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
                    detailLog.setOperationDateTime(operationDateTime);
                    if (StringUtils.isNotEmpty(productCode)) {
                        detailLog.setProductCode(productCode);
                        detailLog.setProductName(productCodeNameMap.get(productCode));
                    }
                    logList.add(detailLog);
                }
                FeePoolDetailEntity feePoolDetailEntity = poolDetailEntityMap.get(poolDetailCode);
                feePoolDetailEntity.setHasUseAmount(feePoolDetailEntity.getHasUseAmount().subtract(thisDetailAmount.abs()));
                feePoolDetailEntity.setUsableAmount(feePoolDetailEntity.getUsableAmount().add(thisDetailAmount.abs()));
                hasEditDetailEntityList.add(feePoolDetailEntity);
            }
            this.subtractHasUseAmountByPoolCode(poolCode, operationAmountTotal.abs());
            feePoolDetailService.updateBatchById(hasEditDetailEntityList);
            feePoolDetailLogService.savePoolDetailLog(logList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void backAllUseByFromCodeBatch(List<FeePoolUseBackAllReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::backAllUseByFromCode);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void occupyPool(FeePoolUseReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getOperationType())) {
            reqVo.setOperationType(FeePoolOperationTypeEnum.OCCUPY.getValue());
        }
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, reqVo.getOperationType());
        reqVo.setFromDesc(StringUtils.isNotEmpty(reqVo.getFromDesc()) ? reqVo.getFromDesc() : operationTypeValue);
        List<FeePoolFileReqVo> fileList = reqVo.getFileList();
        Assert.isTrue(FeePoolConfigUtil.checkOperationTypeGroup(reqVo.getOperationType(), FeePoolOperationTypeGroupEnum.OCCUPY), "该操作类型不能占用费用池" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(reqVo.getDiscount() != null || reqVo.getGoods() != null, "缺失使用明细");

        if (reqVo.getDiscount() != null && reqVo.getDiscount().getAmount().compareTo(BigDecimal.ZERO) != 0) {
            Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");

            //折扣占用
            FeePoolDiscountUseReqVo discount = reqVo.getDiscount();
            Assert.isTrue(discount.getAmount() != null, "缺失占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
            Assert.isTrue(discount.getAmount().compareTo(BigDecimal.ZERO) >= 0, "占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
            List<FeePoolEntity> feeDiscountList = this.lambdaQuery()
                    .eq(FeePoolEntity::getPoolGroup, StringUtils.isNotEmpty(reqVo.getPoolGroup()) ? reqVo.getPoolGroup() : FeePoolGroupEnum.DEFAULT.getValue())
                    .eq(FeePoolEntity::getCustomerCode, reqVo.getCustomerCode())
                    .eq(FeePoolEntity::getPoolType, FeePoolTypeEnum.DISCOUNT.getValue())
                    .eq(FeePoolEntity::getUseType, StringUtils.isNotEmpty(discount.getUseType()) ? discount.getUseType() : FeePoolUseTypeEnum.DEFAULT.getValue())
                    .list();
            Assert.notEmpty(feeDiscountList, "未找到客户[" + reqVo.getCustomerCode() + "]" + (StringUtils.isNotEmpty(discount.getUseType()) ? ("使用类型[" + DictUtil.dictValue(DictConstant.FEE_POOL_USE_TYPE, discount.getUseType()) + "]") : "") + "的折扣费用池");
            FeePoolEntity pool = feeDiscountList.get(0);

            FeePoolAdjustReqVo adjustReqVo = CrmBeanUtil.copy(pool, FeePoolAdjustReqVo.class);
            adjustReqVo.setAmount(discount.getAmount());
            adjustReqVo.setUseType(discount.getUseType());
            adjustReqVo.setFromCode(reqVo.getFromCode());
            adjustReqVo.setFromDesc(reqVo.getFromDesc());
            adjustReqVo.setOperationType(reqVo.getOperationType());
            adjustReqVo.setRemarks(reqVo.getRemarks());
            adjustReqVo.setFileList(reqVo.getFileList());
            this.occupyByPoolCode(adjustReqVo);
        }

        if (reqVo.getGoods() != null) {
            //货补使用

            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getGoods().getGoodsUseByPoolProductList())) {
                //按照费用池+商品占用
                feePoolGoodsService.occupyGoodsAmountByPoolProductList(reqVo.getPoolGroup(), reqVo.getGoods().getUseType(), reqVo.getOperationType(), reqVo.getFromCode(), reqVo.getFromDesc(), reqVo.getRemarks(), reqVo.getGoods().getGoodsUseByPoolProductList(), fileList);
            }
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getGoods().getGoodsUseByProductLevelList())) {
                //按照产品层级占用
                feePoolGoodsService.occupyGoodsAmountByProductLevelList(reqVo.getPoolGroup(), reqVo.getGoods().getUseType(), reqVo.getOperationType(), reqVo.getFromCode(), reqVo.getFromDesc(), reqVo.getRemarks(), reqVo.getGoods().getGoodsUseByProductLevelList(), fileList, reqVo.getCustomerCode());
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void occupyPoolBatch(List<FeePoolUseReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::occupyPool);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void backAllOccupyByFromCode(FeePoolOccupyBackAllReqVo reqVo) {
        String fromCode = reqVo.getFromCode();
        String fromDesc = reqVo.getFromDesc();
        String remarks = reqVo.getRemarks();
        String operationType = reqVo.getOperationType();
        Assert.hasText(fromCode, "缺失来源单号");
        if (StringUtils.isEmpty(operationType)) {
            operationType = FeePoolOperationTypeEnum.UNOCCUPY.getValue();
        }
        FeePoolOperationTypeGroupEnum operationTypeGroupEnum = FeePoolConfigUtil.getOperationTypeGroup(operationType);
        Assert.isTrue(operationTypeGroupEnum == FeePoolOperationTypeGroupEnum.UNOCCUPY, "该操作类型不支持释放");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);

        Set<String> operationTypeSet = new HashSet<>(16);
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.OCCUPY));
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.UNOCCUPY));
        List<FeePoolOperationEntity> list = feePoolOperationService.lambdaQuery()
                .eq(FeePoolOperationEntity::getFromCode, fromCode)
                .in(FeePoolOperationEntity::getOperationType, operationTypeSet)
                .list();
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        final Map<String, List<FeePoolOperationEntity>> poolOperationGroup = list.stream().collect(Collectors.groupingBy(FeePoolOperationEntity::getPoolCode));

        String operationDateTime = DateUtil.dateNowHms();
        for (Map.Entry<String, List<FeePoolOperationEntity>> entry :
                poolOperationGroup.entrySet()) {
            String poolCode = entry.getKey();
            List<FeePoolOperationEntity> operationEntityList = entry.getValue();
            BigDecimal operationAmountTotal = operationEntityList.stream().map(FeePoolOperationEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            if (operationAmountTotal.compareTo(BigDecimal.ZERO) == 0) {
                continue;
            }
            Assert.isTrue(operationAmountTotal.compareTo(BigDecimal.ZERO) <= 0, "费用池【" + poolCode + "】数据异常");
            List<FeePoolDetailLogEntity> detailLogList = feePoolDetailLogService.lambdaQuery()
                    .eq(FeePoolDetailLogEntity::getPoolCode, poolCode)
                    .eq(FeePoolDetailLogEntity::getFromCode, fromCode)
                    .in(FeePoolDetailLogEntity::getOperationType, operationTypeSet)
                    .list();
            Assert.notEmpty(detailLogList, "费用池【" + poolCode + "】数据异常");

            final Set<String> poolDetailCodeSet = detailLogList.stream().map(FeePoolDetailLogEntity::getPoolDetailCode).collect(Collectors.toSet());
            final Map<String, String> productCodeNameMap = detailLogList.stream().filter(x -> StringUtils.isNotEmpty(x.getProductCode())).collect(Collectors.toMap(FeePoolDetailLogEntity::getProductCode, FeePoolDetailLogEntity::getProductName, (x1, x2) -> x1));

            Map<String, FeePoolDetailEntity> poolDetailEntityMap = feePoolDetailService.lambdaQuery()
                    .in(FeePoolDetailEntity::getPoolDetailCode, poolDetailCodeSet)
                    .list()
                    .stream().collect(Collectors.toMap(FeePoolDetailEntity::getPoolDetailCode, v -> v));

            List<FeePoolDetailEntity> hasEditDetailEntityList = new ArrayList<>();
            FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
            operationReqVo.setPoolCode(poolCode);
            operationReqVo.setOperationType(operationType);
            operationReqVo.setFromCode(fromCode);
            operationReqVo.setFromDesc(StringUtils.isNotEmpty(fromDesc) ? fromDesc : operationTypeValue);
            operationReqVo.setOperationDateTime(operationDateTime);
            operationReqVo.setOperationAmount(operationAmountTotal.abs().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
            operationReqVo.setFileList(null);
            operationReqVo.setRemarks(remarks);
            String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);
            List<FeePoolDetailLogReqVo> logList = new ArrayList<>();

            Map<String, List<FeePoolDetailLogEntity>> groupByDetailCodeMap = detailLogList.stream().collect(Collectors.groupingBy(FeePoolDetailLogEntity::getPoolDetailCode));
            for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByDetailCodeEntry :
                    groupByDetailCodeMap.entrySet()) {
                String poolDetailCode = groupByDetailCodeEntry.getKey();
                List<FeePoolDetailLogEntity> groupByDetailLogList = groupByDetailCodeEntry.getValue();
                BigDecimal thisDetailAmount = groupByDetailLogList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                Assert.isTrue(thisDetailAmount.compareTo(BigDecimal.ZERO) <= 0, "费用池明细【" + poolDetailCode + "】释放" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "有误");
                Assert.isTrue(poolDetailEntityMap.containsKey(poolDetailCode), "费用池【" + poolCode + "】数据有误");
                Map<String, List<FeePoolDetailLogEntity>> groupByProductCodeMap = groupByDetailLogList.stream().collect(Collectors.groupingBy(x -> x.getProductCode() == null ? "" : x.getProductCode()));
                for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByProductCodeEntry :
                        groupByProductCodeMap.entrySet()) {
                    String productCode = groupByProductCodeEntry.getKey();
                    List<FeePoolDetailLogEntity> groupByProductLogList = groupByProductCodeEntry.getValue();

                    BigDecimal thisDetailProductAmount = groupByProductLogList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                    Assert.isTrue(thisDetailProductAmount.compareTo(BigDecimal.ZERO) <= 0, "费用池明细【" + poolDetailCode + "】" + (StringUtils.isNotEmpty(productCode) ? ("商品【" + productCode + "】") : "") + "释放" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "有误");
                    FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                    detailLog.setPoolCode(poolCode);
                    detailLog.setPoolDetailCode(poolDetailCode);
                    detailLog.setOperationCode(operationCode);
                    detailLog.setOperationType(operationType);
                    detailLog.setFromCode(fromCode);
                    detailLog.setFromDesc(StringUtils.isNotEmpty(fromDesc) ? fromDesc : operationTypeValue);
                    detailLog.setRemarks(remarks);
                    detailLog.setOperationAmount(thisDetailProductAmount.abs().multiply(operationTypeGroupEnum.getUsableAmountWeight()));
                    detailLog.setOperationDateTime(operationDateTime);
                    if (StringUtils.isNotEmpty(productCode)) {
                        detailLog.setProductCode(productCode);
                        detailLog.setProductName(productCodeNameMap.get(productCode));
                    }
                    logList.add(detailLog);
                }
                FeePoolDetailEntity feePoolDetailEntity = poolDetailEntityMap.get(poolDetailCode);
                feePoolDetailEntity.setOccupyAmount(feePoolDetailEntity.getOccupyAmount().subtract(thisDetailAmount.abs()));
                feePoolDetailEntity.setUsableAmount(feePoolDetailEntity.getUsableAmount().add(thisDetailAmount.abs()));
                hasEditDetailEntityList.add(feePoolDetailEntity);
            }
            this.subtractOccupyAmountByPoolCode(poolCode, operationAmountTotal.abs());
            feePoolDetailService.updateBatchById(hasEditDetailEntityList);
            feePoolDetailLogService.savePoolDetailLog(logList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void backAllOccupyByFromCodeBatch(List<FeePoolOccupyBackAllReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::backAllOccupyByFromCode);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void occupyToUseByFromCode(FeePoolOccupyToUseReqVo reqVo) {

        String fromCode = reqVo.getFromCode();
        Assert.hasText(fromCode, "缺失来源单号");
        String fromDesc = reqVo.getFromDesc();
        String remarks = reqVo.getRemarks();
        String operationType = reqVo.getOperationType();
        if (StringUtils.isEmpty(operationType)) {
            operationType = FeePoolOperationTypeEnum.ORDER_USE.getValue();
        }
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        String unoccupyOperationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, FeePoolOperationTypeEnum.UNOCCUPY.getValue());
        if (StringUtils.isEmpty(fromDesc)) {
            fromDesc = operationTypeValue;
        }

        Set<String> operationTypeSet = new HashSet<>(16);
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.OCCUPY));
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.UNOCCUPY));
        List<FeePoolOperationEntity> list = feePoolOperationService.lambdaQuery()
                .eq(FeePoolOperationEntity::getFromCode, fromCode)
                .in(FeePoolOperationEntity::getOperationType, operationTypeSet)
                .list();
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        final Map<String, List<FeePoolOperationEntity>> poolOperationGroup = list.stream().collect(Collectors.groupingBy(FeePoolOperationEntity::getPoolCode));

        String operationDateTime = DateUtil.dateNowHms();
        for (Map.Entry<String, List<FeePoolOperationEntity>> entry :
                poolOperationGroup.entrySet()) {
            String poolCode = entry.getKey();
            List<FeePoolOperationEntity> operationEntityList = entry.getValue();
            BigDecimal operationAmountTotal = operationEntityList.stream().map(FeePoolOperationEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            if (operationAmountTotal.compareTo(BigDecimal.ZERO) == 0) {
                continue;
            }
            Assert.isTrue(operationAmountTotal.compareTo(BigDecimal.ZERO) <= 0, "费用池【" + poolCode + "】数据异常");
            List<FeePoolDetailLogEntity> detailLogList = feePoolDetailLogService.lambdaQuery()
                    .eq(FeePoolDetailLogEntity::getPoolCode, poolCode)
                    .eq(FeePoolDetailLogEntity::getFromCode, fromCode)
                    .in(FeePoolDetailLogEntity::getOperationType, operationTypeSet)
                    .list();
            Assert.notEmpty(detailLogList, "费用池【" + poolCode + "】数据异常");

            final Set<String> poolDetailCodeSet = detailLogList.stream().map(FeePoolDetailLogEntity::getPoolDetailCode).collect(Collectors.toSet());
            final Map<String, String> productCodeNameMap = detailLogList.stream().filter(x -> StringUtils.isNotEmpty(x.getProductCode())).collect(Collectors.toMap(FeePoolDetailLogEntity::getProductCode, FeePoolDetailLogEntity::getProductName, (x1, x2) -> x1));

            Map<String, FeePoolDetailEntity> poolDetailEntityMap = feePoolDetailService.lambdaQuery()
                    .in(FeePoolDetailEntity::getPoolDetailCode, poolDetailCodeSet)
                    .list()
                    .stream().collect(Collectors.toMap(FeePoolDetailEntity::getPoolDetailCode, v -> v));

            List<FeePoolDetailEntity> hasEditDetailEntityList = new ArrayList<>();
            FeePoolOperationReqVo occupyOperationReqVo = new FeePoolOperationReqVo();
            occupyOperationReqVo.setPoolCode(poolCode);
            occupyOperationReqVo.setOperationType(FeePoolOperationTypeEnum.UNOCCUPY.getValue());
            occupyOperationReqVo.setFromCode(fromCode);
            occupyOperationReqVo.setFromDesc(unoccupyOperationTypeValue);
            occupyOperationReqVo.setOperationDateTime(operationDateTime);
            occupyOperationReqVo.setOperationAmount(operationAmountTotal.abs().multiply(FeePoolOperationTypeGroupEnum.UNOCCUPY.getUsableAmountWeight()));
            occupyOperationReqVo.setFileList(null);
            occupyOperationReqVo.setRemarks(remarks);
            String unoccupyOperationCode = feePoolOperationService.savePoolOperation(occupyOperationReqVo);

            FeePoolOperationReqVo useOperationReqVo = new FeePoolOperationReqVo();
            useOperationReqVo.setPoolCode(poolCode);
            useOperationReqVo.setOperationType(operationType);
            useOperationReqVo.setFromCode(fromCode);
            useOperationReqVo.setFromDesc(fromDesc);
            useOperationReqVo.setOperationDateTime(operationDateTime);
            useOperationReqVo.setOperationAmount(operationAmountTotal.abs().multiply(FeePoolOperationTypeGroupEnum.USE.getUsableAmountWeight()));
            useOperationReqVo.setFileList(null);
            useOperationReqVo.setRemarks(remarks);
            String useOperationCode = feePoolOperationService.savePoolOperation(useOperationReqVo);
            List<FeePoolDetailLogReqVo> logList = new ArrayList<>();

            Map<String, List<FeePoolDetailLogEntity>> groupByDetailCodeMap = detailLogList.stream().collect(Collectors.groupingBy(FeePoolDetailLogEntity::getPoolDetailCode));
            for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByDetailCodeEntry :
                    groupByDetailCodeMap.entrySet()) {
                String poolDetailCode = groupByDetailCodeEntry.getKey();
                List<FeePoolDetailLogEntity> groupByDetailLogList = groupByDetailCodeEntry.getValue();
                BigDecimal thisDetailAmount = groupByDetailLogList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                Assert.isTrue(thisDetailAmount.compareTo(BigDecimal.ZERO) <= 0, "费用池明细【" + poolDetailCode + "】占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "有误");
                Assert.isTrue(poolDetailEntityMap.containsKey(poolDetailCode), "费用池【" + poolCode + "】数据有误");
                Map<String, List<FeePoolDetailLogEntity>> groupByProductCodeMap = groupByDetailLogList.stream().collect(Collectors.groupingBy(x -> x.getProductCode() == null ? "" : x.getProductCode()));
                for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByProductCodeEntry :
                        groupByProductCodeMap.entrySet()) {
                    String productCode = groupByProductCodeEntry.getKey();
                    List<FeePoolDetailLogEntity> groupByProductLogList = groupByProductCodeEntry.getValue();

                    BigDecimal thisDetailProductAmount = groupByProductLogList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                    Assert.isTrue(thisDetailProductAmount.compareTo(BigDecimal.ZERO) <= 0, "费用池明细【" + poolDetailCode + "】" + (StringUtils.isNotEmpty(productCode) ? ("商品【" + productCode + "】") : "") + "占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "有误");
                    FeePoolDetailLogReqVo unoccupyDetailLog = new FeePoolDetailLogReqVo();
                    unoccupyDetailLog.setPoolCode(poolCode);
                    unoccupyDetailLog.setPoolDetailCode(poolDetailCode);
                    unoccupyDetailLog.setOperationCode(unoccupyOperationCode);
                    unoccupyDetailLog.setOperationType(FeePoolOperationTypeEnum.UNOCCUPY.getValue());
                    unoccupyDetailLog.setFromCode(fromCode);
                    unoccupyDetailLog.setFromDesc(unoccupyOperationTypeValue);
                    unoccupyDetailLog.setRemarks(remarks);
                    unoccupyDetailLog.setOperationAmount(thisDetailProductAmount.abs().multiply(FeePoolOperationTypeGroupEnum.UNOCCUPY.getUsableAmountWeight()));
                    unoccupyDetailLog.setOperationDateTime(operationDateTime);
                    if (StringUtils.isNotEmpty(productCode)) {
                        unoccupyDetailLog.setProductCode(productCode);
                        unoccupyDetailLog.setProductName(productCodeNameMap.get(productCode));
                    }
                    logList.add(unoccupyDetailLog);

                    FeePoolDetailLogReqVo useDetailLog = new FeePoolDetailLogReqVo();
                    useDetailLog.setPoolCode(poolCode);
                    useDetailLog.setPoolDetailCode(poolDetailCode);
                    useDetailLog.setOperationCode(useOperationCode);
                    useDetailLog.setOperationType(operationType);
                    useDetailLog.setFromCode(fromCode);
                    useDetailLog.setFromDesc(fromDesc);
                    useDetailLog.setRemarks(remarks);
                    useDetailLog.setOperationAmount(thisDetailProductAmount.abs().multiply(FeePoolOperationTypeGroupEnum.USE.getUsableAmountWeight()));
                    useDetailLog.setOperationDateTime(operationDateTime);
                    if (StringUtils.isNotEmpty(productCode)) {
                        useDetailLog.setProductCode(productCode);
                        useDetailLog.setProductName(productCodeNameMap.get(productCode));
                    }
                    logList.add(useDetailLog);
                }

                FeePoolDetailEntity feePoolDetailEntity = poolDetailEntityMap.get(poolDetailCode);
                feePoolDetailEntity.setOccupyAmount(feePoolDetailEntity.getOccupyAmount().subtract(thisDetailAmount.abs()));
                feePoolDetailEntity.setHasUseAmount(feePoolDetailEntity.getHasUseAmount().add(thisDetailAmount.abs()));
                hasEditDetailEntityList.add(feePoolDetailEntity);
            }
            this.subtractOccupyAmountByPoolCode(poolCode, operationAmountTotal.abs());
            this.addHasUseAmountByPoolCode(poolCode, operationAmountTotal.abs());
            feePoolDetailService.updateBatchById(hasEditDetailEntityList);
            feePoolDetailLogService.savePoolDetailLog(logList);
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void occupyToUseByFromCodeBatch(List<FeePoolOccupyToUseReqVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(this::occupyToUseByFromCode);
        }
    }

    @Override
    public FeePoolAmountRespVo queryAmountByCustomerCode(String poolGroup, String customerCode, String poolType) {
        Assert.hasText(customerCode, "缺失客户编码");
        if (StringUtils.isEmpty(poolGroup)) {
            poolGroup = FeePoolGroupEnum.DEFAULT.getValue();
        }
        List<FeePoolEntity> list = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolGroup, poolGroup)
                .eq(StringUtils.isNotEmpty(poolType), FeePoolEntity::getPoolType, poolType)
                .eq(FeePoolEntity::getCustomerCode, customerCode)
                .select(FeePoolEntity::getTotalAmount, FeePoolEntity::getFreezeAmount, FeePoolEntity::getHasUseAmount, FeePoolEntity::getUsableAmount)
                .list();
        FeePoolAmountRespVo respVo = new FeePoolAmountRespVo();
        respVo.setCustomerCode(customerCode);
        BigDecimal totalAmount = BigDecimal.ZERO;
        BigDecimal freezeAmount = BigDecimal.ZERO;
        BigDecimal hasUseAmount = BigDecimal.ZERO;
        BigDecimal occupyAmount = BigDecimal.ZERO;
        BigDecimal usableAmount = BigDecimal.ZERO;
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            totalAmount = list.stream().map(FeePoolEntity::getTotalAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            freezeAmount = list.stream().map(FeePoolEntity::getFreezeAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            hasUseAmount = list.stream().map(FeePoolEntity::getHasUseAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            occupyAmount = list.stream().map(FeePoolEntity::getOccupyAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            usableAmount = list.stream().map(FeePoolEntity::getUsableAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        respVo.setTotalAmount(totalAmount);
        respVo.setFreezeAmount(freezeAmount);
        respVo.setHasUseAmount(hasUseAmount);
        respVo.setOccupyAmount(occupyAmount);
        respVo.setUsableAmount(usableAmount);
        return respVo;
    }

    @Override
    public FeePoolMonthAmountRespVo queryMonthAmountByCustomerCode(String poolGroup, String customerCode, String poolType, String yearMonth) {

        Assert.hasText(customerCode, "缺失客户编码");
        Assert.hasText(yearMonth, "缺失年月");
        if (StringUtils.isEmpty(poolGroup)) {
            poolGroup = FeePoolGroupEnum.DEFAULT.getValue();
        }
        boolean flag = false;
        try {
            Date parse = DateUtil.yyyy_MM.parse(yearMonth);
            String format = DateUtil.yyyy_MM.format(parse);
            if (format.equals(yearMonth)) {
                flag = true;
            }
        } catch (Exception e) {
        }
        Assert.isTrue(flag, "年月格式不合法");

        FeePoolMonthAmountRespVo respVo = new FeePoolMonthAmountRespVo();

        //剩余可用余额
        BigDecimal usableAmount = feePoolMapper.sumUsableAmount(poolGroup, poolType, customerCode);

        //本月剩余可用余额
        BigDecimal thisMonthUsableAmount = feePoolMapper.sumThisMonthUsableAmount(poolGroup, poolType, customerCode, yearMonth);

        //本月上账费用
        BigDecimal thisMonthAccountAmount = feePoolMapper.sumThisMonthAccountAmount(poolGroup, poolType, customerCode, yearMonth);

        //上月余额
        BigDecimal lastMonthUsableAmount = feePoolMapper.sumLastMonthUsableAmount(poolGroup, poolType, customerCode, yearMonth);

        //本月使用费用
        List<String> operationTypeList = new ArrayList<>();
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.USE));
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.BACK));
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.FREEZE));
        operationTypeList.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.UNFREEZE));
        BigDecimal thisMonthHasUseAmount = feePoolMapper.sumThisMonthHasUseAmount(poolGroup, poolType, customerCode, yearMonth, operationTypeList);

        respVo.setCustomerCode(customerCode);
        respVo.setYearMonth(yearMonth);
        respVo.setUsableAmount(usableAmount == null ? BigDecimal.ZERO : usableAmount);
        respVo.setThisMonthUsableAmount(thisMonthUsableAmount == null ? BigDecimal.ZERO : thisMonthUsableAmount);
        respVo.setThisMonthAccountAmount(thisMonthAccountAmount == null ? BigDecimal.ZERO : thisMonthAccountAmount);
        respVo.setLastMonthUsableAmount(lastMonthUsableAmount == null ? BigDecimal.ZERO : lastMonthUsableAmount);
        respVo.setThisMonthHasUseAmount(thisMonthHasUseAmount == null ? BigDecimal.ZERO : thisMonthHasUseAmount.negate());

        return respVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addTotalAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.notNull(pool, "无效的费用池编号");
        pool.setTotalAmount(pool.getTotalAmount().add(amount));
        pool.setUsableAmount(pool.getUsableAmount().add(amount));
        this.updateById(pool);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addHasUseAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.isTrue(pool.getUsableAmount().compareTo(amount) >= 0, "使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        pool.setHasUseAmount(pool.getHasUseAmount().add(amount));
        pool.setUsableAmount(pool.getUsableAmount().subtract(amount));
        this.updateById(pool);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void subtractHasUseAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.isTrue(pool.getHasUseAmount().compareTo(amount) >= 0, "回退使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        pool.setHasUseAmount(pool.getHasUseAmount().subtract(amount));
        pool.setUsableAmount(pool.getUsableAmount().add(amount));
        this.updateById(pool);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addOccupyAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.notNull(pool, "无效的费用池编号");
        Assert.isTrue(pool.getUsableAmount().compareTo(amount) >= 0, "占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        pool.setOccupyAmount(Optional.ofNullable(pool.getOccupyAmount()).orElse(BigDecimal.ZERO).add(amount));
        pool.setUsableAmount(pool.getUsableAmount().subtract(amount));
        this.updateById(pool);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void subtractOccupyAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.notNull(pool, "无效的费用池编号");
        Assert.isTrue(Optional.ofNullable(pool.getOccupyAmount()).orElse(BigDecimal.ZERO).compareTo(amount) >= 0, "释放" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过已占用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        pool.setOccupyAmount(Optional.ofNullable(pool.getOccupyAmount()).orElse(BigDecimal.ZERO).subtract(amount));
        pool.setUsableAmount(pool.getUsableAmount().add(amount));
        this.updateById(pool);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addFreezeAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.notNull(pool, "无效的费用池编号");
        Assert.isTrue(pool.getUsableAmount().compareTo(amount) >= 0, "冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过可使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        pool.setFreezeAmount(pool.getFreezeAmount().add(amount));
        pool.setUsableAmount(pool.getUsableAmount().subtract(amount));
        this.updateById(pool);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void subtractFreezeAmountByPoolCode(String poolCode, BigDecimal amount) {
        Assert.hasText(poolCode, "缺失费用池编号");
        Assert.isTrue(amount != null, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为空");
        Assert.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能小于0");
        FeePoolEntity pool = this.lambdaQuery()
                .eq(FeePoolEntity::getPoolCode, poolCode)
                .one();
        Assert.notNull(pool, "无效的费用池编号");
        Assert.isTrue(pool.getFreezeAmount().compareTo(amount) >= 0, "解冻" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能超过已冻结" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        pool.setFreezeAmount(pool.getFreezeAmount().subtract(amount));
        pool.setUsableAmount(pool.getUsableAmount().add(amount));
        this.updateById(pool);
    }

    @Override
    public void backDiscountUseAmount(String operationType, String fromCode, String fromDesc, BigDecimal backAmount, String remarks, List<FeePoolFileReqVo> fileList) {
        Assert.isTrue(backAmount.compareTo(BigDecimal.ZERO) >= 0, "折扣回退" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不能为负数");

        //使用和回退的类型
        Set<String> operationTypeSet = new HashSet<>(16);
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.USE));
        operationTypeSet.addAll(FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.BACK));

        //已经使用的金额
        final List<FeePoolDetailLogEntity> useDetailLogEntityList = feePoolDetailLogService.lambdaQuery()
                .eq(FeePoolDetailLogEntity::getFromCode, fromCode)
                .in(FeePoolDetailLogEntity::getOperationType, operationTypeSet)
                .list();
        Assert.notEmpty(useDetailLogEntityList, "未找到来源单号【" + fromCode + "】对应的费用池使用记录");
        BigDecimal notBackAmount = useDetailLogEntityList.stream().map(FeePoolDetailLogEntity::getOperationAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        Assert.isTrue(notBackAmount.compareTo(BigDecimal.ZERO) == 0, "来源单号【" + fromCode + "】没有可回退的" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        Assert.isTrue(notBackAmount.compareTo(BigDecimal.ZERO) < 0, "来源单号【" + fromCode + "】数据异常");
        Assert.isTrue(notBackAmount.abs().compareTo(backAmount) >= 0, "来源单号【" + fromCode + "】回退折扣" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "超过剩余可回退的" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);

        List<FeePoolEntity> poolEntityList = this.lambdaQuery()
                .in(FeePoolEntity::getPoolType, FeePoolTypeEnum.DISCOUNT.getValue())
                .in(FeePoolEntity::getPoolCode, useDetailLogEntityList.stream().map(FeePoolDetailLogEntity::getPoolCode).collect(Collectors.toSet()))
                .list();
        final Set<String> poolCodeSet = poolEntityList.stream().map(FeePoolEntity::getPoolCode).collect(Collectors.toSet());

        BigDecimal restAmount = backAmount;
        String operationDateTime = DateUtil.dateNowHms();

        //按照费用池编号分组
        Map<String, List<FeePoolDetailLogEntity>> groupByPoolCodeMap = useDetailLogEntityList.stream().filter(x -> poolCodeSet.contains(x.getPoolCode())).collect(Collectors.groupingBy(FeePoolDetailLogEntity::getPoolCode));
        for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByPoolCodeEntry :
                groupByPoolCodeMap.entrySet()) {
            List<FeePoolDetailLogReqVo> detailLogList = new ArrayList<>();
            BigDecimal poolCodeBackAmount = BigDecimal.ZERO;
            Map<String, BigDecimal> poolDetailCodeBackAmountMap = new HashMap<>(16);
            String poolCode = groupByPoolCodeEntry.getKey();
            List<FeePoolDetailLogEntity> groupByPoolCodeList = groupByPoolCodeEntry.getValue();
            Map<String, BigDecimal> poolDetailCodeUseAmountMap = groupByPoolCodeList.stream().collect(Collectors.groupingBy(FeePoolDetailLogEntity::getPoolDetailCode, Collectors.mapping(FeePoolDetailLogEntity::getOperationAmount, Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));
            for (Map.Entry<String, BigDecimal> poolDetailCodeUseAmountEntry :
                    poolDetailCodeUseAmountMap.entrySet()) {
                String poolDetailCode = poolDetailCodeUseAmountEntry.getKey();
                BigDecimal useNotBackAmountAbs = poolDetailCodeUseAmountEntry.getValue().abs();
                if (useNotBackAmountAbs.compareTo(BigDecimal.ZERO) == 0) {
                    continue;
                }
                if (restAmount.compareTo(BigDecimal.ZERO) == 0) {
                    continue;
                }
                BigDecimal itemBack = BigDecimal.ZERO;
                if (useNotBackAmountAbs.compareTo(restAmount) >= 0) {
                    itemBack = restAmount;
                } else {
                    itemBack = useNotBackAmountAbs;
                }
                restAmount = restAmount.subtract(itemBack);
                FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                detailLog.setPoolCode(poolCode);
                detailLog.setPoolDetailCode(poolDetailCode);
                detailLog.setOperationType(operationType);
                detailLog.setFromCode(fromCode);
                detailLog.setFromDesc(fromDesc);
                detailLog.setRemarks(remarks);
                detailLog.setOperationAmount(itemBack.multiply(FeePoolOperationTypeGroupEnum.BACK.getUsableAmountWeight()));
                detailLogList.add(detailLog);
                poolCodeBackAmount = poolCodeBackAmount.add(itemBack);
                poolDetailCodeBackAmountMap.put(poolDetailCode, itemBack.add(poolDetailCodeBackAmountMap.getOrDefault(poolDetailCode, BigDecimal.ZERO)));
            }

            if (poolCodeBackAmount.compareTo(BigDecimal.ZERO) > 0) {
                FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
                operationReqVo.setPoolCode(poolCode);
                operationReqVo.setOperationType(operationType);
                operationReqVo.setFromCode(fromCode);
                operationReqVo.setFromDesc(fromDesc);
                operationReqVo.setOperationDateTime(operationDateTime);
                operationReqVo.setOperationAmount(poolCodeBackAmount.multiply(FeePoolOperationTypeGroupEnum.BACK.getUsableAmountWeight()));
                operationReqVo.setFileList(fileList);
                operationReqVo.setRemarks(remarks);
                String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);

                subtractHasUseAmountByPoolCode(poolCode, poolCodeBackAmount);
                List<FeePoolDetailEntity> list = feePoolDetailService.lambdaQuery()
                        .in(FeePoolDetailEntity::getPoolDetailCode, poolDetailCodeBackAmountMap.keySet())
                        .list();
                list.forEach(x -> {
                    if (poolDetailCodeBackAmountMap.containsKey(x.getPoolDetailCode())) {
                        BigDecimal poolDetailBackAmount = poolDetailCodeBackAmountMap.get(x.getPoolDetailCode());
                        x.setHasUseAmount(x.getHasUseAmount().subtract(poolDetailBackAmount));
                        x.setUsableAmount(x.getUsableAmount().add(poolDetailBackAmount));
                    }
                });
                feePoolDetailService.updateBatchById(list);
                detailLogList.forEach(x -> x.setOperationCode(operationCode));
                feePoolDetailLogService.savePoolDetailLog(detailLogList);
            }
        }
    }

    @Override
    public void backGoodsUseAmount(String operationType, String fromCode, String fromDesc, List<FeePoolGoodsUseBackItemReqVo> itemList, String remarks, List<FeePoolFileReqVo> fileList) {
        Assert.hasText(operationType, "缺失操作类型");
        Assert.hasText(fromCode, "缺失来源单号");
        Assert.notEmpty(itemList, "缺失退还明细");
        String operationTypeValue = DictUtil.dictValue(DictConstant.FEE_POOL_OPERATION_TYPE, operationType);
        Assert.hasText(operationTypeValue, "无效的操作类型");
        Assert.isTrue(FeePoolConfigUtil.checkOperationTypeGroup(operationType, FeePoolOperationTypeGroupEnum.BACK), "该操作类型不支持回退" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        itemList.forEach(item -> {
            Assert.hasText(item.getProductCode(), "明细缺失商品编码");
            Assert.isTrue(item.getAmount() != null, "明细缺失回退" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
            Assert.isTrue(item.getAmount().compareTo(BigDecimal.ZERO) > 0, "明细回退" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "必须大于0");
        });

        //已经使用的金额
        final List<FeePoolDetailLogEntity> useDetailLogEntityList = feePoolDetailLogService.lambdaQuery()
                .eq(FeePoolDetailLogEntity::getFromCode, fromCode)
                .in(FeePoolDetailLogEntity::getOperationType, FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.USE))
                .orderByAsc(FeePoolDetailLogEntity::getOperationDateTime)
                .list();
        Assert.notEmpty(useDetailLogEntityList, "未找到来源单号【" + fromCode + "】对应的费用池使用记录");

        //已经回退的金额
        Map<String, BigDecimal> hasBackAmountMap = new HashMap<>(16);
        final List<FeePoolDetailLogEntity> backDetailLogEntityList = feePoolDetailLogService.lambdaQuery()
                .eq(FeePoolDetailLogEntity::getFromCode, fromCode)
                .in(FeePoolDetailLogEntity::getOperationType, FeePoolConfigUtil.getOperationTypeList(FeePoolOperationTypeGroupEnum.BACK))
                .orderByAsc(FeePoolDetailLogEntity::getOperationDateTime)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(backDetailLogEntityList)) {
            for (FeePoolDetailLogEntity item :
                    backDetailLogEntityList) {
                String key = item.getPoolCode() + "_" + item.getPoolDetailCode() + "_" + item.getProductCode();
                BigDecimal bigDecimal = hasBackAmountMap.getOrDefault(key, BigDecimal.ZERO);
                bigDecimal = bigDecimal.add(item.getOperationAmount());
                hasBackAmountMap.put(key, bigDecimal);
            }
        }


        //本次回退的金额
        Map<String, BigDecimal> backProductAmountMap = itemList.stream().collect(Collectors.groupingBy(FeePoolGoodsUseBackItemReqVo::getProductCode, Collectors.mapping(FeePoolGoodsUseBackItemReqVo::getAmount, Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));

        //使用明细按照费用池编号分组
        Map<String, List<FeePoolDetailLogEntity>> groupByPoolCodeMap = useDetailLogEntityList.stream().collect(Collectors.groupingBy(FeePoolDetailLogEntity::getPoolCode));

        Map<String, BigDecimal> poolDetailCodeNeedBackAmountMap = new LinkedHashMap<>(16);
        for (Map.Entry<String, List<FeePoolDetailLogEntity>> groupByPoolCodeEntry :
                groupByPoolCodeMap.entrySet()) {
            String poolCode = groupByPoolCodeEntry.getKey();
            List<FeePoolDetailLogEntity> groupByPoolCodeList = groupByPoolCodeEntry.getValue();
            List<FeePoolDetailLogReqVo> detailLogList = new ArrayList<>();
            BigDecimal poolCodeNeedBackAmount = BigDecimal.ZERO;
            for (FeePoolDetailLogEntity item :
                    groupByPoolCodeList) {
                String key = item.getPoolCode() + "_" + item.getPoolDetailCode() + "_" + item.getProductCode();
                //计算本条明细使用使用但是没有回退的金额
                BigDecimal notBackAmount = item.getOperationAmount().abs();
                if (hasBackAmountMap.containsKey(key)) {
                    BigDecimal hasBack = hasBackAmountMap.get(key);
                    if (hasBack.compareTo(BigDecimal.ZERO) > 0) {
                        if (notBackAmount.compareTo(hasBack) >= 0) {
                            notBackAmount = notBackAmount.subtract(hasBack);
                            hasBack = BigDecimal.ZERO;
                        } else {
                            hasBack = hasBack.subtract(notBackAmount);
                            notBackAmount = BigDecimal.ZERO;
                        }
                    }
                    hasBackAmountMap.put(key, hasBack);
                }
                if (notBackAmount.compareTo(BigDecimal.ZERO) <= 0) {
                    continue;
                }
                if (!backProductAmountMap.containsKey(item.getProductCode())) {
                    continue;
                }
                BigDecimal productBackAmount = backProductAmountMap.get(item.getProductCode());
                if (productBackAmount.compareTo(BigDecimal.ZERO) <= 0) {
                    continue;
                }
                BigDecimal thisBackAmount = BigDecimal.ZERO;
                if (productBackAmount.compareTo(notBackAmount) <= 0) {
                    thisBackAmount = productBackAmount;
                } else {
                    thisBackAmount = notBackAmount;
                }
                notBackAmount = notBackAmount.subtract(thisBackAmount);
                productBackAmount = productBackAmount.subtract(thisBackAmount);
                backProductAmountMap.put(item.getProductCode(), productBackAmount);
                poolCodeNeedBackAmount = poolCodeNeedBackAmount.add(thisBackAmount);

                BigDecimal poolDetailCodeNeedBackAmount = thisBackAmount;
                if (poolDetailCodeNeedBackAmountMap.containsKey(item.getPoolDetailCode())) {
                    poolDetailCodeNeedBackAmount = poolDetailCodeNeedBackAmount.add(poolDetailCodeNeedBackAmountMap.get(item.getPoolDetailCode()));
                }
                poolDetailCodeNeedBackAmountMap.put(item.getPoolDetailCode(), poolDetailCodeNeedBackAmount);

                FeePoolDetailLogReqVo detailLog = new FeePoolDetailLogReqVo();
                detailLog.setPoolCode(poolCode);
                detailLog.setPoolDetailCode(item.getPoolDetailCode());
                detailLog.setOperationType(operationType);
                detailLog.setFromCode(fromCode);
                detailLog.setFromDesc(StringUtils.isNotEmpty(fromDesc) ? fromDesc : operationTypeValue);
                detailLog.setRemarks(remarks);
                detailLog.setOperationAmount(thisBackAmount.multiply(FeePoolOperationTypeGroupEnum.BACK.getUsableAmountWeight()));
                detailLog.setProductCode(item.getProductCode());
                detailLog.setProductName(item.getProductName());
                detailLogList.add(detailLog);
            }

            if (poolCodeNeedBackAmount.compareTo(BigDecimal.ZERO) > 0) {
                String operationDateTime = DateUtil.dateNowHms();
                FeePoolOperationReqVo operationReqVo = new FeePoolOperationReqVo();
                operationReqVo.setPoolCode(poolCode);
                operationReqVo.setOperationType(operationType);
                operationReqVo.setFromCode(fromCode);
                operationReqVo.setFromDesc(StringUtils.isNotEmpty(fromDesc) ? fromDesc : operationTypeValue);
                operationReqVo.setOperationDateTime(operationDateTime);
                operationReqVo.setOperationAmount(poolCodeNeedBackAmount.multiply(FeePoolOperationTypeGroupEnum.BACK.getUsableAmountWeight()));
                operationReqVo.setFileList(fileList);
                operationReqVo.setRemarks(remarks);
                String operationCode = feePoolOperationService.savePoolOperation(operationReqVo);


                detailLogList.forEach(item -> {
                    item.setOperationCode(operationCode);
                    item.setOperationDateTime(operationDateTime);
                });
                List<FeePoolDetailEntity> list = feePoolDetailService.lambdaQuery().in(FeePoolDetailEntity::getPoolDetailCode, poolDetailCodeNeedBackAmountMap.keySet()).list();
                list.forEach(x -> {
                    if (poolDetailCodeNeedBackAmountMap.containsKey(x.getPoolDetailCode())) {
                        BigDecimal detailNeedBack = poolDetailCodeNeedBackAmountMap.get(x.getPoolDetailCode());
                        if (x.getHasUseAmount().compareTo(detailNeedBack) < 0) {
                            throw new BusinessException("费用池明细退还" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "不足");
                        }
                        x.setHasUseAmount(x.getHasUseAmount().subtract(detailNeedBack));
                        x.setUsableAmount(x.getUsableAmount().add(detailNeedBack));
                    }
                });
                feePoolDetailService.updateBatchById(list);
                this.subtractHasUseAmountByPoolCode(poolCode, poolCodeNeedBackAmount);
                feePoolDetailLogService.savePoolDetailLog(detailLogList);
            }
        }

        for (Map.Entry<String, BigDecimal> entry :
                backProductAmountMap.entrySet()) {
            Assert.isTrue(entry.getValue().compareTo(BigDecimal.ZERO) == 0, "商品" + entry.getKey() + "退还" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC + "超过已使用" + FeePoolConfigUtil.FEE_POOL_AMOUNT_DESC);
        }
    }
}
